/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.linuxtools.internal.statesystem.core.backend.historytree;

import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.eclipse.linuxtools.internal.statesystem.core.backend.historytree.CoreNode;
import org.eclipse.linuxtools.internal.statesystem.core.backend.historytree.HTConfig;
import org.eclipse.linuxtools.internal.statesystem.core.backend.historytree.HTInterval;
import org.eclipse.linuxtools.internal.statesystem.core.backend.historytree.LeafNode;
import org.eclipse.linuxtools.statesystem.core.exceptions.TimeRangeException;
import org.eclipse.linuxtools.statesystem.core.interval.ITmfStateInterval;
import org.eclipse.linuxtools.statesystem.core.statevalue.TmfStateValue;

public abstract class HTNode {
    private final HTConfig config;
    private final long nodeStart;
    private long nodeEnd;
    private final int sequenceNumber;
    private int parentSequenceNumber;
    private int stringSectionOffset;
    private int sizeOfIntervalSection;
    private volatile boolean isOnDisk;
    private final List<HTInterval> intervals;
    private final ReentrantReadWriteLock rwl = new ReentrantReadWriteLock(false);
    private static final int COMMON_HEADER_SIZE = 34;

    protected HTNode(HTConfig config, int seqNumber, int parentSeqNumber, long start) {
        this.config = config;
        this.nodeStart = start;
        this.sequenceNumber = seqNumber;
        this.parentSequenceNumber = parentSeqNumber;
        this.stringSectionOffset = config.getBlockSize();
        this.sizeOfIntervalSection = 0;
        this.isOnDisk = false;
        this.intervals = new ArrayList<HTInterval>();
    }

    public static final HTNode readNode(HTConfig config, FileChannel fc) throws IOException {
        HTNode newNode = null;
        ByteBuffer buffer = ByteBuffer.allocate(config.getBlockSize());
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        int res = fc.read(buffer);
        assert (res == config.getBlockSize());
        buffer.flip();
        byte typeByte = buffer.get();
        NodeType type = NodeType.fromByte(typeByte);
        long start = buffer.getLong();
        long end = buffer.getLong();
        int seqNb = buffer.getInt();
        int parentSeqNb = buffer.getInt();
        int intervalCount = buffer.getInt();
        int stringSectionOffset = buffer.getInt();
        buffer.get();
        switch (type) {
            case CORE: {
                newNode = new CoreNode(config, seqNb, parentSeqNb, start);
                newNode.readSpecificHeader(buffer);
                break;
            }
            case LEAF: {
                newNode = new LeafNode(config, seqNb, parentSeqNb, start);
                newNode.readSpecificHeader(buffer);
                break;
            }
            default: {
                throw new IOException();
            }
        }
        int i = 0;
        while (i < intervalCount) {
            newNode.intervals.add(HTInterval.readFrom(buffer));
            ++i;
        }
        newNode.nodeEnd = end;
        newNode.stringSectionOffset = stringSectionOffset;
        newNode.isOnDisk = true;
        return newNode;
    }

    public final void writeSelf(FileChannel fc) throws IOException {
        this.rwl.readLock().lock();
        try {
            int blockSize;
            int curStringsEntryEndPos = blockSize = this.config.getBlockSize();
            ByteBuffer buffer = ByteBuffer.allocate(blockSize);
            buffer.order(ByteOrder.LITTLE_ENDIAN);
            buffer.clear();
            buffer.put(this.getNodeType().toByte());
            buffer.putLong(this.nodeStart);
            buffer.putLong(this.nodeEnd);
            buffer.putInt(this.sequenceNumber);
            buffer.putInt(this.parentSequenceNumber);
            buffer.putInt(this.intervals.size());
            buffer.putInt(this.stringSectionOffset);
            buffer.put((byte)1);
            this.writeSpecificHeader(buffer);
            for (HTInterval interval : this.intervals) {
                int size = interval.writeInterval(buffer, curStringsEntryEndPos);
                curStringsEntryEndPos -= size;
            }
            while (buffer.position() < this.stringSectionOffset) {
                buffer.put((byte)0);
            }
            assert (curStringsEntryEndPos == this.stringSectionOffset);
            buffer.position(blockSize);
            buffer.flip();
            int res = fc.write(buffer);
            assert (res == blockSize);
        }
        finally {
            this.rwl.readLock().unlock();
        }
        this.isOnDisk = true;
    }

    protected HTConfig getConfig() {
        return this.config;
    }

    public long getNodeStart() {
        return this.nodeStart;
    }

    public long getNodeEnd() {
        if (this.isOnDisk) {
            return this.nodeEnd;
        }
        return 0L;
    }

    public int getSequenceNumber() {
        return this.sequenceNumber;
    }

    public int getParentSequenceNumber() {
        return this.parentSequenceNumber;
    }

    public void setParentSequenceNumber(int newParent) {
        this.parentSequenceNumber = newParent;
    }

    public boolean isOnDisk() {
        return this.isOnDisk;
    }

    public void addInterval(HTInterval newInterval) {
        this.rwl.writeLock().lock();
        try {
            assert (newInterval.getIntervalSize() <= this.getNodeFreeSpace());
            this.intervals.add(newInterval);
            this.sizeOfIntervalSection += newInterval.getIntervalSize();
            this.stringSectionOffset -= newInterval.getStringsEntrySize();
        }
        finally {
            this.rwl.writeLock().unlock();
        }
    }

    public void closeThisNode(long endtime) {
        this.rwl.writeLock().lock();
        try {
            assert (endtime >= this.nodeStart);
            if (!this.intervals.isEmpty()) {
                Collections.sort(this.intervals);
                assert (endtime >= this.intervals.get(this.intervals.size() - 1).getEndTime());
            }
            this.nodeEnd = endtime;
        }
        finally {
            this.rwl.writeLock().unlock();
        }
    }

    public void writeInfoFromNode(List<ITmfStateInterval> stateInfo, long t) throws TimeRangeException {
        this.rwl.readLock().lock();
        try {
            int i = this.getStartIndexFor(t);
            while (i < this.intervals.size()) {
                ITmfStateInterval interval = this.intervals.get(i);
                if (interval.getStartTime() <= t && interval.getAttribute() < stateInfo.size()) {
                    stateInfo.set(interval.getAttribute(), interval);
                }
                ++i;
            }
        }
        finally {
            this.rwl.readLock().unlock();
        }
    }

    public HTInterval getRelevantInterval(int key, long t) throws TimeRangeException {
        this.rwl.readLock().lock();
        try {
            int i = this.getStartIndexFor(t);
            while (i < this.intervals.size()) {
                HTInterval curInterval = this.intervals.get(i);
                if (curInterval.getAttribute() == key && curInterval.getStartTime() <= t && curInterval.getEndTime() >= t) {
                    HTInterval hTInterval = curInterval;
                    return hTInterval;
                }
                ++i;
            }
            return null;
        }
        finally {
            this.rwl.readLock().unlock();
        }
    }

    private int getStartIndexFor(long t) throws TimeRangeException {
        if (this.intervals.isEmpty()) {
            return 0;
        }
        HTInterval dummy = new HTInterval(0L, t, 0, TmfStateValue.nullValue());
        int index = Collections.binarySearch(this.intervals, dummy);
        if (index < 0) {
            index = -index - 1;
        }
        if (index < 0) {
            index = 0;
        }
        if (index >= this.intervals.size()) {
            index = this.intervals.size() - 1;
        }
        while (index > 0 && this.intervals.get(index - 1).compareTo(this.intervals.get(index)) == 0) {
            --index;
        }
        return index;
    }

    public final int getTotalHeaderSize() {
        return 34 + this.getSpecificHeaderSize();
    }

    private int getDataSectionEndOffset() {
        return this.getTotalHeaderSize() + this.sizeOfIntervalSection;
    }

    public int getNodeFreeSpace() {
        this.rwl.readLock().lock();
        int ret = this.stringSectionOffset - this.getDataSectionEndOffset();
        this.rwl.readLock().unlock();
        return ret;
    }

    public long getNodeUsagePercent() {
        this.rwl.readLock().lock();
        try {
            int blockSize = this.config.getBlockSize();
            float freePercent = (float)this.getNodeFreeSpace() / (float)(blockSize - this.getTotalHeaderSize()) * 100.0f;
            long l = (long)(100.0f - freePercent);
            return l;
        }
        finally {
            this.rwl.readLock().unlock();
        }
    }

    public String toString() {
        StringBuffer buf = new StringBuffer("Node #" + this.sequenceNumber + ", ");
        buf.append(this.toStringSpecific());
        buf.append(String.valueOf(this.intervals.size()) + " intervals (" + this.getNodeUsagePercent() + "% used), ");
        buf.append("[" + this.nodeStart + " - ");
        buf = this.isOnDisk ? buf.append(this.nodeEnd + "]") : buf.append("...]");
        return buf.toString();
    }

    public void debugPrintIntervals(PrintWriter writer) {
        writer.println("Node #" + this.sequenceNumber + ":");
        if (this.getNodeType() == NodeType.CORE) {
            CoreNode thisNode = (CoreNode)this;
            writer.print("  " + thisNode.getNbChildren() + " children");
            if (thisNode.getNbChildren() >= 1) {
                writer.print(": [ " + thisNode.getChild(0));
                int i = 1;
                while (i < thisNode.getNbChildren()) {
                    writer.print(", " + thisNode.getChild(i));
                    ++i;
                }
                writer.print(']');
            }
            writer.print('\n');
        }
        writer.println("  Intervals contained:");
        int i = 0;
        while (i < this.intervals.size()) {
            writer.println(this.intervals.get(i).toString());
            ++i;
        }
        writer.println('\n');
    }

    public abstract NodeType getNodeType();

    protected abstract int getSpecificHeaderSize();

    protected abstract void readSpecificHeader(ByteBuffer var1);

    protected abstract void writeSpecificHeader(ByteBuffer var1);

    protected abstract String toStringSpecific();

    public static enum NodeType {
        CORE,
        LEAF;


        public static NodeType fromByte(byte rep) throws IOException {
            switch (rep) {
                case 1: {
                    return CORE;
                }
                case 2: {
                    return LEAF;
                }
            }
            throw new IOException();
        }

        public byte toByte() {
            switch (this) {
                case CORE: {
                    return 1;
                }
                case LEAF: {
                    return 2;
                }
            }
            throw new IllegalStateException();
        }
    }
}

