/*******************************************************************************
 * Copyright (c) 2013, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Markus Schorn - Bug 449306: Unnecessary access to secure storage.
 *******************************************************************************/
package org.eclipse.remote.internal.jsch.core;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.equinox.security.storage.ISecurePreferences;
import org.eclipse.equinox.security.storage.SecurePreferencesFactory;
import org.eclipse.equinox.security.storage.StorageException;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;

public class JSchConnectionAttributes {
	public static final String CONNECTIONS_KEY = "connections"; //$NON-NLS-1$

	public static final String ADDRESS_ATTR = "JSCH_ADDRESS_ATTR"; //$NON-NLS-1$
	public static final String USERNAME_ATTR = "JSCH_USERNAME_ATTR"; //$NON-NLS-1$
	public static final String PASSWORD_ATTR = "JSCH_PASSWORD_ATTR"; //$NON-NLS-1$
	public static final String PORT_ATTR = "JSCH_PORT_ATTR"; //$NON-NLS-1$
	public static final String PROXYCONNECTION_ATTR = "JSCH_PROXYCONNECTION_ATTR"; //$NON-NLS-1$
	public static final String PROXYCOMMAND_ATTR = "JSCH_PROXYCOMMAND_ATTR"; //$NON-NLS-1$
	public static final String IS_PASSWORD_ATTR = "JSCH_IS_PASSWORD_ATTR"; //$NON-NLS-1$
	public static final String PASSPHRASE_ATTR = "JSCH_PASSPHRASE_ATTR"; //$NON-NLS-1$
	public static final String TIMEOUT_ATTR = "JSCH_TIMEOUT_ATTR"; //$NON-NLS-1$
	public static final String USE_LOGIN_SHELL_ATTR = "JSCH_USE_LOGIN_SHELL_ATTR"; //$NON-NLS-1$

	private String fName;
	private String fNewName;

	private final Map<String, String> fAttributes = Collections.synchronizedMap(new HashMap<String, String>());
	private final Map<String, String> fSecureAttributes = Collections.synchronizedMap(new HashMap<String, String>());

	private boolean fSecureAttributesLoaded;

	public JSchConnectionAttributes(String name) {
		fName = name;
		load();
	}

	private void clearPreferences() {
		try {
			getPreferences().clear();
		} catch (BackingStoreException e) {
			Activator.log(e.getMessage());
		}
		getSecurePreferences().clear();
	}

	public JSchConnectionAttributes copy() {
		JSchConnectionAttributes copy = new JSchConnectionAttributes(fName);
		copy.getAttributes().putAll(fAttributes);
		copy.setSecureAttributes(getSecureAttributes());
		return copy;
	}

	private void flushPreferences() {
		try {
			getPreferences().flush();
		} catch (BackingStoreException e) {
			Activator.log(e.getMessage());
		}
		try {
			getSecurePreferences().flush();
		} catch (IOException e) {
			Activator.log(e.getMessage());
		}
	}

	public String getAttribute(String key, String def) {
		if (fAttributes.containsKey(key)) {
			return fAttributes.get(key);
		}
		return def;
	}

	public Map<String, String> getAttributes() {
		return fAttributes;
	}

	public boolean getBoolean(String key, boolean def) {
		if (fAttributes.containsKey(key)) {
			return Boolean.parseBoolean(fAttributes.get(key));
		}
		return def;
	}

	public int getInt(String key, int def) {
		try {
			return Integer.parseInt(fAttributes.get(key));
		} catch (NumberFormatException e) {
			return def;
		}
	}

	public String getName() {
		if (fNewName == null) {
			return fName;
		}
		return fNewName;
	}

	private Preferences getPreferences() {
		IEclipsePreferences root = InstanceScope.INSTANCE.getNode(Activator.getUniqueIdentifier());
		Preferences connections = root.node(CONNECTIONS_KEY);
		return connections.node(fName);
	}

	public String getSecureAttribute(String key, String def) {
		loadSecureAttributes();
		if (fSecureAttributes.containsKey(key)) {
			return fSecureAttributes.get(key);
		}
		return def;
	}

	public Map<String, String> getSecureAttributes() {
		loadSecureAttributes();
		return fSecureAttributes;
	}

	void setSecureAttributes(Map<String, String> secureAttributes) {
		fSecureAttributes.clear();
		fSecureAttributes.putAll(secureAttributes);
		fSecureAttributesLoaded = true;
	}

	private ISecurePreferences getSecurePreferences() {
		ISecurePreferences secRoot = SecurePreferencesFactory.getDefault();
		ISecurePreferences secConnections = secRoot.node(CONNECTIONS_KEY);
		return secConnections.node(fName);
	}

	private void load() {
		IEclipsePreferences root = InstanceScope.INSTANCE.getNode(Activator.getUniqueIdentifier());
		Preferences connections = root.node(CONNECTIONS_KEY);
		Preferences nodes = connections.node(fName);
		try {
			loadAttributes(nodes);
		} catch (BackingStoreException e) {
			Activator.log(e.getMessage());
		}
		fSecureAttributesLoaded = false;
	}

	private void loadAttributes(Preferences node) throws BackingStoreException {
		fAttributes.clear();
		for (String key : node.keys()) {
			fAttributes.put(key, node.get(key, null));
		}
	}

	private void loadSecureAttributes() {
		if (fSecureAttributesLoaded) {
			return;
		}

		ISecurePreferences secNode = getSecurePreferences();
		try {
			fSecureAttributes.clear();
			for (String key : secNode.keys()) {
				fSecureAttributes.put(key, secNode.get(key, null));
			}
		} catch (StorageException e) {
			Activator.log(e.getMessage());
		}
		fSecureAttributesLoaded = true;
	}

	public void remove() {
		clearPreferences();
		flushPreferences();
	}

	public void save() {
		clearPreferences();
		if (fNewName != null) {
			fName = fNewName;
			fNewName = null;
		}
		savePreferences();
		flushPreferences();
	}

	private void savePreferences() {
		Preferences node = getPreferences();
		synchronized (fAttributes) {
			for (Entry<String, String> entry : fAttributes.entrySet()) {
				if (entry.getValue() != null) {
					node.put(entry.getKey(), entry.getValue());
				} else {
					node.remove(entry.getKey());
				}
			}
		}
		if (fSecureAttributesLoaded) {
			try {
				ISecurePreferences secNode = getSecurePreferences();
				synchronized (fSecureAttributes) {
					for (Entry<String, String> entry : fSecureAttributes.entrySet()) {
						secNode.put(entry.getKey(), entry.getValue(), true);
					}
				}
			} catch (StorageException e) {
				Activator.log(e.getMessage());
			}
		}
	}

	public void setAttribute(String key, String value) {
		fAttributes.put(key, value);
	}

	public void setName(String name) {
		fNewName = name;
	}

	public void setSecureAttribute(String key, String value) {
		loadSecureAttributes();
		fSecureAttributes.put(key, value);
	}
}
