/**********************************************************************
 * Copyright (c) 2014 Ericsson
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Jonathan Rajotte - Support of machine interface
 **********************************************************************/

package org.eclipse.tracecompass.lttng2.control.ui.tests.service;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URL;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.IChannelInfo;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.IDomainInfo;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.IEventInfo;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.ISessionInfo;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.TraceChannelOutputType;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.TraceEnablement;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.TraceEventType;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.TraceLogLevel;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.TraceSessionState;
import org.eclipse.tracecompass.internal.lttng2.control.core.model.impl.SessionInfo;
import org.eclipse.tracecompass.internal.lttng2.control.ui.views.service.ILttngControlService;
import org.eclipse.tracecompass.internal.lttng2.control.ui.views.service.LTTngControlServiceConstants;
import org.eclipse.tracecompass.internal.lttng2.control.ui.views.service.LTTngControlServiceMI;
import org.junit.Test;

/**
 * @author ejorajo
 *
 */
public class LTTngControlServiceMiTest extends LTTngControlServiceTest {

    private static final String MI_TEST_STREAM = "LTTngServiceMiTest.cfg";

    private static final String SCEN_SESSION_WITH_SYSCALLS = "GetSessionWithSyscalls";

    @Override
    protected ILttngControlService getControlService() {
        try {
            URL resource = LTTngControlServiceMI.class.getResource(LTTngControlServiceConstants.MI_XSD_FILENAME);
            return new LTTngControlServiceMI(getShell(), resource);
        } catch (ExecutionException e) {
            return null;
        }
    }

    @Override
    public void testGetSessionNameGarbage() {
        try {
            fShell.setScenario(SCEN_GET_SESSION_GARBAGE_OUT);
            fService.getSessionNames(new NullProgressMonitor());
        } catch (ExecutionException e) {
            // Success. Parsing of garbage result in an ExecutionException
            // generated by the XML document parser: Unable to parse the xml
            // document.
        }
    }

    @Override
    public void testCreateLiveSession() throws ExecutionException {
        fShell.setScenario(SCEN_CREATE_LIVE_SESSION);

        ISessionInfo params = new SessionInfo("mysession");
        params.setLive(true);
        params.setStreamedTrace(true);
        params.setNetworkUrl("net://127.0.0.1");
        ISessionInfo sessionInfo = fService.createSession(params, new NullProgressMonitor());
        assertNotNull(sessionInfo);
        assertEquals("mysession", sessionInfo.getName());
        assertEquals(TraceSessionState.INACTIVE, sessionInfo.getSessionState());
        assertTrue(sessionInfo.isStreamedTrace());
        assertTrue(sessionInfo.isLive());
        assertEquals("tcp4://127.0.0.1:5342/ [data: 5343]", sessionInfo.getSessionPath());
        List<String> names = fService.getSessionNames(new NullProgressMonitor());
        assertEquals(names.get(0), "mysession");
        fService.destroySession("mysession", new NullProgressMonitor());
    }

    @Override
    protected String getTestStream() {
        return MI_TEST_STREAM;
    }

    @Override
    protected TraceLogLevel getAllEventTraceLogLevel() {
        return TraceLogLevel.TRACE_DEBUG;
    }

    @Override
    public void testGetKernelProviderNoUstVerbose() {
        // Verbose mode in machine interface is deactivated. This test is
        // ignored.
    }

    @Override
    public void testCreateSession2_1() {
        // 2.1 is not supported by mi. This test is ignored.
    }

    @Override
    public void testGetKernelProviderNoUst3() {
        // Verbose mode in machine interface is deactivated. This test is
        // ignored.
    }

    @Override
    public void testGetKernelProviderNoKernelVerbose() {
        // Verbose mode in machine interface is deactivated. This test is
        // ignored.
    }

    @Override
    public void testCreateSessionVerbose2_1() {
        // Verbose mode in machine interface is deactivated. This test is
        // ignored.
    }

    @Override
    public void testDestroySessionVerbose() {
        // Verbose mode in machine interface is deactivated. This test is
        // ignored.
    }

    @Override
    public void testCreateSessionWithPrompt() {
        // TODO Investigate if this case can happen in production. If yes than
        // we need to rethinks the MI fetching and parsing.
    }

    @Override
    public void testAddContext() {
        // TODO This does not use mi feature.And currently the context enabling
        // is wrong for 2.6.
    }

    @Override
    public void testAddContextFailure() {
        // TODO This does not use mi feature.And currently the context enabling
        // is wrong for 2.6.
    }

    @Override
    public void testCreateSnapshotSession2_5() {
        // not applicable for MI
    }

    /**
     * Tests the listing of syscalls
     */
    @Test
    public void testListSycallEvents() {
        try {
            fShell.setScenario(SCEN_SESSION_WITH_SYSCALLS);
            ISessionInfo session = fService.getSession("mysession", new NullProgressMonitor());

            // Verify Session
            assertNotNull(session);
            assertEquals("mysession", session.getName());
            assertEquals("/home/user/lttng-traces/mysession-20120129-084256", session.getSessionPath());
            assertEquals(TraceSessionState.INACTIVE, session.getSessionState());

            IDomainInfo[] domains = session.getDomains();
            assertNotNull(domains);
            assertEquals(1, domains.length);

            // Verify Kernel domain
            assertEquals("Kernel", domains[0].getName());
            IChannelInfo[] channels =  domains[0].getChannels();
            assertNotNull(channels);
            assertEquals(1, channels.length);

            // Verify Kernel's channel0
            assertEquals("channel0", channels[0].getName());
            assertEquals(4, channels[0].getNumberOfSubBuffers());
            assertEquals("splice()", channels[0].getOutputType().getInName());
            assertEquals(TraceChannelOutputType.SPLICE, channels[0].getOutputType());
            assertEquals(false, channels[0].isOverwriteMode());
            assertEquals(200, channels[0].getReadTimer());
            assertEquals(TraceEnablement.ENABLED, channels[0].getState());
            assertEquals(262144, channels[0].getSubBufferSize());
            assertEquals(0, channels[0].getSwitchTimer());

            // Verify event info
            IEventInfo[] channel0Events = channels[0].getEvents();
            assertNotNull(channel0Events);
            assertEquals(2, channel0Events.length);
            assertEquals("read", channel0Events[0].getName());
            assertEquals(TraceEventType.SYSCALL, channel0Events[0].getEventType());
            assertEquals(TraceEnablement.ENABLED, channel0Events[0].getState());

            assertEquals("write", channel0Events[1].getName());
            assertEquals(TraceEventType.SYSCALL, channel0Events[1].getEventType());
            assertEquals(TraceEnablement.ENABLED, channel0Events[1].getState());
        } catch (ExecutionException e) {
            fail(e.toString());
        }
    }
}
