/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.util.Comparator;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.http.DateGenerator;
import org.eclipse.jetty.http.HttpContent;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.resource.ResourceFactory;

public class ResourceCache {
    private static final Logger LOG = Log.getLogger(ResourceCache.class);
    private final ConcurrentMap<String, Content> _cache;
    private final AtomicInteger _cachedSize;
    private final AtomicInteger _cachedFiles;
    private final ResourceFactory _factory;
    private final ResourceCache _parent;
    private final MimeTypes _mimeTypes;
    private final boolean _etagSupported;
    private final boolean _useFileMappedBuffer;
    private int _maxCachedFileSize = 0x8000000;
    private int _maxCachedFiles = 2048;
    private int _maxCacheSize = 0x10000000;

    public ResourceCache(ResourceCache parent, ResourceFactory factory, MimeTypes mimeTypes, boolean useFileMappedBuffer, boolean etags) {
        this._factory = factory;
        this._cache = new ConcurrentHashMap<String, Content>();
        this._cachedSize = new AtomicInteger();
        this._cachedFiles = new AtomicInteger();
        this._mimeTypes = mimeTypes;
        this._parent = parent;
        this._useFileMappedBuffer = useFileMappedBuffer;
        this._etagSupported = etags;
    }

    public int getCachedSize() {
        return this._cachedSize.get();
    }

    public int getCachedFiles() {
        return this._cachedFiles.get();
    }

    public int getMaxCachedFileSize() {
        return this._maxCachedFileSize;
    }

    public void setMaxCachedFileSize(int maxCachedFileSize) {
        this._maxCachedFileSize = maxCachedFileSize;
        this.shrinkCache();
    }

    public int getMaxCacheSize() {
        return this._maxCacheSize;
    }

    public void setMaxCacheSize(int maxCacheSize) {
        this._maxCacheSize = maxCacheSize;
        this.shrinkCache();
    }

    public int getMaxCachedFiles() {
        return this._maxCachedFiles;
    }

    public void setMaxCachedFiles(int maxCachedFiles) {
        this._maxCachedFiles = maxCachedFiles;
        this.shrinkCache();
    }

    public boolean isUseFileMappedBuffer() {
        return this._useFileMappedBuffer;
    }

    public void flushCache() {
        if (this._cache != null) {
            while (this._cache.size() > 0) {
                for (String path : this._cache.keySet()) {
                    Content content = (Content)this._cache.remove(path);
                    if (content == null) continue;
                    content.invalidate();
                }
            }
        }
    }

    public HttpContent lookup(String pathInContext) throws IOException {
        HttpContent httpContent;
        Content content = (Content)this._cache.get(pathInContext);
        if (content != null && content.isValid()) {
            return content;
        }
        Resource resource = this._factory.getResource(pathInContext);
        HttpContent loaded = this.load(pathInContext, resource);
        if (loaded != null) {
            return loaded;
        }
        if (this._parent != null && (httpContent = this._parent.lookup(pathInContext)) != null) {
            return httpContent;
        }
        return null;
    }

    protected boolean isCacheable(Resource resource) {
        long len = resource.length();
        return len > 0L && len < (long)this._maxCachedFileSize && len < (long)this._maxCacheSize;
    }

    private HttpContent load(String pathInContext, Resource resource) throws IOException {
        Content content = null;
        if (resource == null || !resource.exists()) {
            return null;
        }
        if (!resource.isDirectory() && this.isCacheable(resource)) {
            content = new Content(pathInContext, resource);
            this.shrinkCache();
            Content added = this._cache.putIfAbsent(pathInContext, content);
            if (added != null) {
                content.invalidate();
                content = added;
            }
            return content;
        }
        return new HttpContent.ResourceAsHttpContent(resource, this._mimeTypes.getMimeByExtension(resource.toString()), this.getMaxCachedFileSize(), this._etagSupported);
    }

    private void shrinkCache() {
        block0: while (this._cache.size() > 0 && (this._cachedFiles.get() > this._maxCachedFiles || this._cachedSize.get() > this._maxCacheSize)) {
            TreeSet<Content> sorted = new TreeSet<Content>(new Comparator<Content>(){

                @Override
                public int compare(Content c1, Content c2) {
                    if (c1._lastAccessed < c2._lastAccessed) {
                        return -1;
                    }
                    if (c1._lastAccessed > c2._lastAccessed) {
                        return 1;
                    }
                    if (c1._length < c2._length) {
                        return -1;
                    }
                    return c1._key.compareTo(c2._key);
                }
            });
            for (Content content : this._cache.values()) {
                sorted.add(content);
            }
            for (Content content : sorted) {
                if (this._cachedFiles.get() <= this._maxCachedFiles && this._cachedSize.get() <= this._maxCacheSize) continue block0;
                if (content != this._cache.remove(content.getKey())) continue;
                content.invalidate();
            }
        }
    }

    protected ByteBuffer getIndirectBuffer(Resource resource) {
        try {
            return BufferUtil.toBuffer((Resource)resource, (boolean)true);
        }
        catch (IOException | IllegalArgumentException e) {
            LOG.warn((Throwable)e);
            return null;
        }
    }

    protected ByteBuffer getDirectBuffer(Resource resource) {
        try {
            if (this._useFileMappedBuffer && resource.getFile() != null && resource.length() < Integer.MAX_VALUE) {
                return BufferUtil.toMappedBuffer((File)resource.getFile());
            }
            return BufferUtil.toBuffer((Resource)resource, (boolean)true);
        }
        catch (IOException | IllegalArgumentException e) {
            LOG.warn((Throwable)e);
            return null;
        }
    }

    public String toString() {
        return "ResourceCache[" + this._parent + "," + this._factory + "]@" + this.hashCode();
    }

    public class Content
    implements HttpContent {
        final Resource _resource;
        final int _length;
        final String _key;
        final long _lastModified;
        final ByteBuffer _lastModifiedBytes;
        final ByteBuffer _contentType;
        final String _etag;
        volatile long _lastAccessed;
        AtomicReference<ByteBuffer> _indirectBuffer = new AtomicReference();
        AtomicReference<ByteBuffer> _directBuffer = new AtomicReference();

        Content(String pathInContext, Resource resource) {
            this._key = pathInContext;
            this._resource = resource;
            String mimeType = ResourceCache.this._mimeTypes.getMimeByExtension(this._resource.toString());
            this._contentType = mimeType == null ? null : BufferUtil.toBuffer((String)mimeType);
            boolean exists = resource.exists();
            this._lastModified = exists ? resource.lastModified() : -1L;
            this._lastModifiedBytes = this._lastModified < 0L ? null : BufferUtil.toBuffer((String)DateGenerator.formatDate((long)this._lastModified));
            this._length = exists ? (int)resource.length() : 0;
            ResourceCache.this._cachedSize.addAndGet(this._length);
            ResourceCache.this._cachedFiles.incrementAndGet();
            this._lastAccessed = System.currentTimeMillis();
            this._etag = ResourceCache.this._etagSupported ? resource.getWeakETag() : null;
        }

        public String getKey() {
            return this._key;
        }

        public boolean isCached() {
            return this._key != null;
        }

        public boolean isMiss() {
            return false;
        }

        public Resource getResource() {
            return this._resource;
        }

        public String getETag() {
            return this._etag;
        }

        boolean isValid() {
            if (this._lastModified == this._resource.lastModified() && (long)this._length == this._resource.length()) {
                this._lastAccessed = System.currentTimeMillis();
                return true;
            }
            if (this == ResourceCache.this._cache.remove(this._key)) {
                this.invalidate();
            }
            return false;
        }

        protected void invalidate() {
            ResourceCache.this._cachedSize.addAndGet(-this._length);
            ResourceCache.this._cachedFiles.decrementAndGet();
            this._resource.close();
        }

        public String getLastModified() {
            return BufferUtil.toString((ByteBuffer)this._lastModifiedBytes);
        }

        public String getContentType() {
            return BufferUtil.toString((ByteBuffer)this._contentType);
        }

        public void release() {
        }

        public ByteBuffer getIndirectBuffer() {
            ByteBuffer buffer = this._indirectBuffer.get();
            if (buffer == null) {
                ByteBuffer buffer2 = ResourceCache.this.getIndirectBuffer(this._resource);
                if (buffer2 == null) {
                    LOG.warn("Could not load " + this, new Object[0]);
                } else {
                    buffer = this._indirectBuffer.compareAndSet(null, buffer2) ? buffer2 : this._indirectBuffer.get();
                }
            }
            if (buffer == null) {
                return null;
            }
            return buffer.slice();
        }

        public ByteBuffer getDirectBuffer() {
            ByteBuffer buffer = this._directBuffer.get();
            if (buffer == null) {
                ByteBuffer buffer2 = ResourceCache.this.getDirectBuffer(this._resource);
                if (buffer2 == null) {
                    LOG.warn("Could not load " + this, new Object[0]);
                } else {
                    buffer = this._directBuffer.compareAndSet(null, buffer2) ? buffer2 : this._directBuffer.get();
                }
            }
            if (buffer == null) {
                return null;
            }
            return buffer.asReadOnlyBuffer();
        }

        public long getContentLength() {
            return this._length;
        }

        public InputStream getInputStream() throws IOException {
            ByteBuffer indirect = this.getIndirectBuffer();
            if (indirect != null && indirect.hasArray()) {
                return new ByteArrayInputStream(indirect.array(), indirect.arrayOffset() + indirect.position(), indirect.remaining());
            }
            return this._resource.getInputStream();
        }

        public ReadableByteChannel getReadableByteChannel() throws IOException {
            return this._resource.getReadableByteChannel();
        }

        public String toString() {
            return String.format("CachedContent@%x{r=%s,e=%b,lm=%s,ct=%s}", this.hashCode(), this._resource, this._resource.exists(), BufferUtil.toString((ByteBuffer)this._lastModifiedBytes), this._contentType);
        }
    }
}

