/*******************************************************************************
 * Copyright (c) 2015 École Polytechnique de Montréal
 *
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/

package org.eclipse.tracecompass.analysis.graph.core.building;

import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.tracecompass.analysis.graph.core.base.TmfGraph;
import org.eclipse.tracecompass.tmf.core.event.ITmfEvent;
import org.eclipse.tracecompass.tmf.core.trace.ITmfTrace;

/**
 * This is the interface used to define the execution graph building logic.
 *
 * Usually the graph builder is the piece of the pipeline which converts trace
 * events to an execution graph.
 *
 * @author Francis Giraldeau
 * @author Geneviève Bastien
 */
public interface ITmfGraphProvider {

    /**
     * Get the trace with which this graph builder plugin is associated.
     *
     * @return The associated trace
     */
    ITmfTrace getTrace();

    /**
     * Return the start time of this "graph builder", which is normally the
     * start time of the originating trace (or it can be the time of the first
     * node in the graph).
     *
     * @return The start time
     */
    long getStartTime();

    /**
     * Assign the target graph where this builder will add new nodes and
     * vertices
     *
     * This needs to be called before .run()!
     *
     * @param graph
     *            Target graph for the state changes generated by this input
     *            plugin
     */
    void assignTargetGraph(TmfGraph graph);

    /**
     * Return the currently assigned target graph.
     *
     * @return Reference to the currently assigned graph, or {@code null} if no
     *         graph is assigned yet
     */
    @Nullable
    TmfGraph getAssignedGraph();

    /**
     * Send an event to this input plugin for processing. The implementation
     * should check the contents, and call the state-modifying methods of its
     * IStateSystemBuilder object accordingly.
     *
     * @param event
     *            The event (which should be safe to cast to the
     *            expectedEventType) that has to be processed.
     */
    void processEvent(ITmfEvent event);

    /**
     * Indicate to the graph building process that we are done (for now), and
     * that it should close the current graph.
     */
    void dispose();

    /**
     * Performs the necessary actions when the build is cancelled
     */
    void handleCancel();

    /**
     * Callback when graph is complete
     */
    void done();

}
