/* @(#) IClock.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class IClock.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 * 01, April, 1998              Initial Release
 * 08, July,  1998  Sunanda Bera    Prevented throwing of IllegalThreadStateException in
 *                                  start. Changes in method start() and stop().
 *
 */


package com.ibm.clock;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.*;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

/**
 * An <code>IClock</code> component is a Java bean that can be used
 * to display a clock in an application. The clock has two modes
 * of display - analog and digital. In both these modes, the developer
 * can completely customize the look and feel of the clock. The bean
 * supports different time-zones. Future releases, will let the developer 
 * set an alarm and use the bean as a stopwatch.
 *
 * <p>
 * <b>Usage:</b>
 * <p>
 * The <code>IClock</code> bean can be customized from within an
 * application builder tool or by manually writing the requisite code.
 * At any point, the developer can get the time shown by the clock by
 * calling the <code>getTime( )</code> method. The developer can also
 * set the time for the clock by calling the <code>setTime( )</code>
 * method. Time is represented by a separate class.
 * The clock can be started by calling the method <code>start()</code>
 * on the clock. The clock starts with the system time adjusted to the
 * currently set time zone. The time can be synchronized by setting the time
 * through the appropriate api at runtime.
 * The clock can be stopped by calling the method <code>stop()</code>.
 * <p>
 * The clock has two modes of display -- Analog and Digital. The display can
 * be set to either analog or display. Both the display modes support a variety
 * of customizations. For the customizations possible on each display see the
 * class documentations of each class. The IClock acts as a facade for each 
 * of the properties of Analog and Digital display.
 *
 * <p>
 * Apart from the view related customizations on each of the displays the
 * following properties can be set on the clock itself.
 * <ul>
 * <li> <i>timeZone</i> - the time-zone for which the clock shows 
 * time.
 * <li> <i>displayMode</i> - selects the display mode for the clock. This
 * can be one of ANALOG and DIGITAL. The display mode value can be set programmatically
 * as symbolic constants ( IClock.DIGITAL or IClock.ANALOG ) or as integer values
 * 4 ( Analog mode ) and 5 ( Digital mode ). However in VisualAge for java while
 * specifying the display mode as a parameter to a connection the integer values
 * 4 ( Analog mode ) and 5 ( Digital mode ) will have to be used.
 * <li> <i>time</i> - sets the time on the clock. <b>NOTE:</b> This is not
 * a design time property. It allows the clock to be synchronized at runtime.
 * </ul>
 * 
 * <p>
 * All the above properties let the developer have complete control over
 * the appearance and behaviour of the <code>IClock</code> bean. For more
 * information on how to set a particular property, please see the 
 * corresponding setter method.
 * 
 * <p>
 * The following code snippet demonstrates how to use the 
 * <code>IClock</code> bean from within an application to have a 
 * customised clock display:
 *
 * <pre>
 * ...
 * import com.ibm.clock.IClock;
 * import com.ibm.clock.AnalogDisplay;
 * import com.ibm.clock.DigitalDisplay;
 *     
 * import java.awt.Color;
 *     
 * ...
 *     
 *     // Create a new clock for the default time-zone and locale.
 * IClock myClock = new IClock( );
 *     
 *     
 *     // set the borderwidth of analog display to 5
 * myClock.setAnalogBorderWidth( 5 );
 *     
 *     // We wish to display ticks only at the 12, 1, 2 etc. marks.
 * myClock.setAnalogTicksStyle( AnalogDisplay.TICK_STYLE_MAJOR);
 *     
 *     // The numerals should be roman numerals.
 * myClock.setAnalogNumeralStyle( AnalogDisplay.NUMERAL_ROMAN );
 *     
 * ...
 * </pre>
 *
 * <p>
 * <b>Important Notes and Caveats:</b>
 * <p>
 * Due to a bug with the current release of JDK (1.1.4), for certain
 * time-zones the system reports incorrect time-zone values on Windows
 * NT and Windows 95. This causes the clock to show the time for a time-
 * zone other than the actual system time-zone. The only workaround
 * currently available seems to be to set the time-zone explicitly by
 * calling the <code>setTimeZone( )</code> method.
 *
 * <p>
 * <b>Note:</b> In the following descriptions, fully qualified class names
 * have been omitted for brevity. Therefore,
 * <ul>
 * <li><code>AnalogDisplay</code> stands for <code>com.ibm.clock.AnalogDisplay</code>
 * <li><code>DigitalDisplay</code> stands for <code>com.ibm.clock.DigitalDisplay</code>
 * <li><code>Color</code> stands for <code>java.awt.Color</code>
 * <li><code>Font</code> stands for <code>java.awt.Font</code>
 * <li><code>TimeZone</code> stands for <code>java.util.TimeZone</code>
 * </ul>
 *
 * in this document.
 *
 * @see Time
 * @see AnalogDisplay
 * @see DigitalDisplay
 *
 */

public class IClock 
extends Panel implements Serializable , PropertyChangeListener
{


/**
 * Analog mode of display.
 * @see #getDisplayMode
 * @see #setDisplayMode
 */
    public final static int ANALOG = 4;
    
/**
 * Digital mode of display.
 * @see #getDisplayMode
 * @see #setDisplayMode
 */
    public final static int DIGITAL= 5;

    // useful constants for calculation purpose.
    private static final long MILLISECONDS_IN_DAY      = 86400000;

    private transient PropertyChangeSupport changesNotifier         = null;
    private transient PropertyChangeListener propertyChangeListener = null;    
    // object of DigitalDiaplay
    private DigitalDisplay digitalDisplay;

    // object of Timer
    private transient Timer timer;
    private transient ClockRefresher refresher = null;

    private int displayMode = IClock.DIGITAL;

    
/**
 * Offset from the System time in GMT. This is set when the time is set via
 * the public api.
 */
    private long timeOffset = 0;

/**
 * The current time zone.
 */
    private TimeZone currentTimeZone = null;

/**
 * The calendar to be used.
 */    
    private transient Calendar calendar = null;
    
/**
 * The analog display to be used.
 */
    private AnalogDisplay analogDisplay = null;
    
    private transient ClockDisplay clockDisplay = null;
    
    // CF -- Sunanda Bera 8-th July 1998
    private transient boolean clockStarted = false;
    private transient boolean inConstructor = true;
    // CF

/**
 * Default Constructor. The default setting is analog clock.
 */
    public IClock()
    {
        super();
        // CF --
        inConstructor = true;
        // CF
        super.setLayout( new BorderLayout());
        setDisplayMode( ANALOG );
        start();
        // CF --
        inConstructor = false;
        // CF
    }//end of c'tor
    
/**
 * Overrides from super class. Setting a layout does nothing effectively in IClock.
 */    
    public void setLayout( LayoutManager layout )
    {
    }
    

/**
 * Start the clock. Calling this method will start the clock.
 */    
    public void start()
    {
        // CF -- Sunanda Bera 8-th July 1998
        if( clockStarted )
            return;
        // CF
        getTimer().addTimerListener( getRefresher() );
        getTimer().startTicking();
        // CF -- Sunanda Bera 8-th July 1998
        clockStarted = true;
        // CF
    }//end of start

/**
 * Stop the clock from runnning. Calling this method will stop the clock from
 * running.
 * @exception SecurityException If the current thread does not have access rights
 * for stopping the clock.
 * Note : The thread which started the clock has access rights to stop the clock.
 */    
    public void stop()
    throws SecurityException
    {
        // CF -- Sunanda Bera 8-th July 1998
        if( !clockStarted )
            return;
        // CF
        getTimer().stopTicking();
        getTimer().removeTimerListener( getRefresher() );
        timer = null;//invalidate the current timer.
        // CF -- Sunanda Bera 8-th July 1988
        clockStarted = false;
        // CF
    }//end of stop
    
    private ClockRefresher getRefresher()
    {
        if( refresher == null )
            refresher = new ClockRefresher( this );
        return refresher;
    }

/**
 * Get the digital display.
 * @return The digital display being used.
 */    
    private DigitalDisplay getDigitalDisplay()
    {
        if( digitalDisplay == null )
        {
            digitalDisplay = new DigitalDisplay();
            digitalDisplay.addPropertyChangeListener(this);
        }
        
        return digitalDisplay;
    }//end of getDigitalDisplay


/**
 * Get the current analog display.
 * @return The analog display being used.
 */    
    private AnalogDisplay getAnalogDisplay()
    {
        if( analogDisplay == null )
        {
            analogDisplay = new AnalogDisplay();
            analogDisplay.addPropertyChangeListener(this);
        } // endif    
        return analogDisplay;
    }//end of getAnalogDisplay

    
/**
 * Get the timer.
 */
    private Timer getTimer()
    {
        if( timer == null )
            timer = new Timer();
        return timer;
    }//end of getTimer

/**
 * Get the current time zone.
 * @return The current time zone.
 */    
    public TimeZone getTimeZone()
    {
        if( currentTimeZone == null )
            currentTimeZone = TimeZone.getDefault();
        
        return currentTimeZone;
    }// end of getTimeZone    

/**
 * Set the timezone.
 * @param tz The new time zone. Cannot be null.
 * @exception IllegalArgumentException If a null  argument is passed.
 */    
    public void setTimeZone( TimeZone tz )
    throws IllegalArgumentException
    {
            if( tz == null )
                throw new IllegalArgumentException();
            
            if( tz.equals( getTimeZone() ) )
                return; // no change
                
            currentTimeZone = tz;
            
            getDigitalDisplay().setTimeZone( currentTimeZone );
            invalidateCalendar(); // invalidate the current calendar
    }//end of setTimeZone

/**
 * Invalidate the calendar being used currently. This must be called when
 * the timezone is being changed.
 */    
    private void invalidateCalendar()
    {
        calendar = null;
    }//end of invalidateCalendar
    
/**
 * Get the calendar for the current time zone. The calendar is cached and
 * hence has to be invalidated when setting the time zone.
 */    
    private Calendar getCalendar()
    {
        if( calendar == null )
            calendar = Calendar.getInstance( getTimeZone() );

        return calendar;
    }//end of getCalendar

/**
 * Set the time. This api should be called to synchronize the clock at runtime.
 * The application builder can provide a dialog to edit this at run time.
 * @param time The time to be set.
 * @exception IllegalArgumentException if any of the parameters is incorrect.
 */    
     public void setTime( Time time )
     throws IllegalArgumentException
     {
            if( time == null )
                 throw new IllegalArgumentException();
            
            long currentTimeInMillis = System.currentTimeMillis();

            Calendar calendar = getCalendar();
            calendar.setTime( new Date( currentTimeInMillis ) );
            int hour = calendar.get( Calendar.HOUR_OF_DAY );
            int min  = calendar.get( Calendar.MINUTE );
            int sec  = calendar.get( Calendar.SECOND );
            Time currentTime = new Time( hour, min, sec );
            
            setTimeOffset( time.getInMillis() - currentTime.getInMillis() );

     }//end of setTime

/**
 * Get the time offset.
 */    
    private long getTimeOffset()
    {
        return timeOffset;
    }//end of getTimeOffset.

/**
 * Set the time offset.
 */    
    private void setTimeOffset( long offset )
    {
        timeOffset = offset;
    }// end of setTimeOffset


/**
 * Refresh the clock time.
 */
    void refresh() 
    {   
        synchronized( this )
        {
            Time time = getTime();
             // check whether analog or digital and call the appropriate setTime
            getClockDisplay().setTime( time.getHour(), time.getMinute(), time.getSecond());
         }//end of synchronized code.
    
    }// end of timeChanged

/**
 * The current display being used.
 */    
    private ClockDisplay getClockDisplay()
    {
        if( clockDisplay == null )
        {
            switch( getDisplayMode() )
            {
                case DIGITAL:
                    clockDisplay = getDigitalDisplay();
                    break;
                case ANALOG:
                    clockDisplay = getAnalogDisplay();
                    break;
                default:
            }//end switch
        }//end if
        return clockDisplay;
    }//end of getClockDisplay
    

public void propertyChange(PropertyChangeEvent e)
    {
        getPropertyChangeNotifier().firePropertyChange( e.getPropertyName(),
                                                        e.getOldValue(),
                                                        e.getNewValue());
    }    
    
/**
 * Get the current time as shown by the clock.
 * @return The time of the day.
 */
    public Time getTime()
    {
        Calendar calendar = getCalendar();
        
        long currentTime = System.currentTimeMillis();
        long actualTime  = currentTime + getTimeOffset();
        
        calendar.setTime( new Date( actualTime ) );
        
        Time time = new Time( calendar.get( Calendar.HOUR_OF_DAY ),
                              calendar.get( Calendar.MINUTE ),
                              calendar.get( Calendar.SECOND ) );
        return time;
    }//getTime
    
    private PropertyChangeSupport getPropertyChangeNotifier()
    {
        if( changesNotifier == null )
            changesNotifier = new PropertyChangeSupport( this );
        return changesNotifier;
    }//getPropertyChangeNotifier
    
/**
 * Add a property change listener.
 */
    public void addPropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().addPropertyChangeListener( pcl);
    }   


/**
 * Remove a property change listener.
 */    
    public void removePropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().removePropertyChangeListener( pcl);
    }

/**
 * Get the current display mode.
 * @return The current display mode.
 */
    public int getDisplayMode()
    {
        return displayMode;
    }//end of getDisplayMode    

/**
 * Set the display mode. This can either be <code>IClock.ANALOG</code>
 * or <code>IClock.DIGITAL</code>.
 * @param newDisplayMode The new display mode.
 * @exception IllegalArgumentException If the specified argument is illegal.
 */    
    public void setDisplayMode( int newDisplayMode) 
    throws IllegalArgumentException
    {
        if( getDisplayMode() == newDisplayMode )
                return;
        int oldDisplayMode = displayMode;
        this.displayMode = newDisplayMode;            
        switch( newDisplayMode )
        {
            case DIGITAL:
                setDisplay( getDigitalDisplay() );
                break;
            case ANALOG:
                setDisplay( getAnalogDisplay() );
                break;
            default:
                throw new IllegalArgumentException( " bad display mode = " + newDisplayMode );
                
        }   // end if
        
        getPropertyChangeNotifier().firePropertyChange( "displayMode", 
                    new Integer( oldDisplayMode), 
                    new Integer( newDisplayMode));
          
    }   // end of method setDisplayMode
    
    private void setDisplay( Component display )
    {
        removeAll();
        add( "Center", display );
        this.clockDisplay = null;
        // CF --
        //if( !inConstructor )
        //    setSize( display.getPreferredSize() );
        // CF
        validate();
                        
    }//end of setDisplay
    
    private void writeObject( ObjectOutputStream out )
    throws IOException
    {
        out.defaultWriteObject();
    }//writeObject
    
    private void readObject( ObjectInputStream in )
    throws IOException, ClassNotFoundException
    {
        in.defaultReadObject();
        start();
    }//readObject
    
    private static final boolean DEBUG = false;
    private final void debug( String s )
    {
        if( DEBUG )
            System.out.println( getClass().getName() + "::" + s );
    }//end of debug
    
    // get/set for Analog display.
    
/**
 * Get the analog clock border color.
 * @return the border color.
 * @see AnalogDisplay#getBorderColor
 */
    public Color getAnalogBorderColor()
    {
        return getAnalogDisplay().getBorderColor();
    }
/**
 * Set the border color of analog display.
 * @param c The new color. Cannot be null.
 * @see AnalogDisplay#setBorderColor
 */    
    public void setAnalogBorderColor( Color c )
    {
        getAnalogDisplay().setBorderColor( c );
    }

/**
 * Get the border width of analog display.
 * @return The border width.
 * @see AnalogDisplay#getBorderWidth
 */    
    public int getAnalogBorderWidth()
    {
        return getAnalogDisplay().getBorderWidth();
    }

/**
 * Set the border width of analog display.
 * @param w The new width.
 * @see AnalogDisplay#setBorderWidth
 */
    public void setAnalogBorderWidth( int w )
    {
        getAnalogDisplay().setBorderWidth( w );
    }

/**
 * Get the dial color of analog display.
 * @return The dial color.
 * @see AnalogDisplay#getDialColor
 */
    public Color getAnalogDialColor()
    {
        return getAnalogDisplay().getDialColor();
    }

/**
 * Set the dial color of the analog display.
 * @param c The new dial color. Cannot be null.
 * @see AnalogDisplay#setDialColor
 */
    public void setAnalogDialColor( Color c )
    {
        getAnalogDisplay().setDialColor( c );
    }

/**
 * Get the tick color of the analog display.
 * @return Color The tick color.
 * @see AnalogDisplay#getTickColor
 */    
    public Color getAnalogTickColor()
    {
        return getAnalogDisplay().getTickColor();
    }
    
/**
 * Set the tick color of the analog display.
 * @param c The new tick color. Cannot be null.
 * @see AnalogDisplay#setTickColor
 */
    public void setAnalogTickColor( Color c )
    {
        getAnalogDisplay().setTickColor( c );
    }

/**
 * Get the tick style of the analog display.
 * @return The tick style.
 * @see AnalogDisplay#getTickStyle
 */    
    public int getAnalogTickStyle()
    {
        return getAnalogDisplay().getTickStyle();
    }

/**
 * Set the tick style of the analog display.
 * @param w The new tick style.
 * @see AnalogDisplay#setTickStyle
 */
    public void setAnalogTickStyle( int w )
    {
        getAnalogDisplay().setTickStyle( w );
    }
    
/**
 * Get the numeral style of the analog display.
 * @return The numeral style.
 * @see AnalogDisplay#getNumeralStyle
 */
    public int getAnalogNumeralStyle()
    {
        return getAnalogDisplay().getNumeralStyle();
    }

/**
 * Set the numeral style of the analog display.
 * @param w The new numeral style.
 * @see AnalogDisplay#setNumeralStyle
 */
    public void setAnalogNumeralStyle( int w )
    {
        getAnalogDisplay().setNumeralStyle( w );
    }

/**
 * Get the numeral Color of the Analog display.
 * @return The numeral color.
 * @see AnalogDisplay#getNumeralColor
 */    
    public Color getAnalogNumeralColor()
    {
        return getAnalogDisplay().getNumeralColor();
    }

/**
 * Set the numeral color of the analog display.
 * @param c The new color.
 * @see AnalogDisplay#setNumeralColor
 */
    public void setAnalogNumeralColor( Color c )
    {
        getAnalogDisplay().setNumeralColor( c );
    }
    

/**
 * Get the font for the numerals used in Analog clock.
 * @return the current font.
 * @see AnalogDisplay#getNumeralFont
 */
    public Font getAnalogNumeralFont()
    {
        return getAnalogDisplay().getNumeralFont();
    }

/**
 * Set the numeral font to be used in the Analog display.
 * @param font The font to be used.
 * @see AnalogDisplay#setNumeralFont
 */
    public void setAnalogNumeralFont( Font f )
    {
        getAnalogDisplay().setNumeralFont( f );
    }

/**
 * Get the hour hand color in analog display.
 * @return The hour hand color.
 * @see AnalogDisplay#getHourHandColor
 */    
    public Color getAnalogHourHandColor()
    {
        return getAnalogDisplay().getHourHandColor();
    }

/**
 * Set the analog hour hand color.
 * @param c The new hour hand color.
 * @see AnalogDisplay#setHourHandColor
 */
    public void setAnalogHourHandColor( Color c )
    {
        getAnalogDisplay().setHourHandColor( c );
    }

/**
 * Is the analog hour hand visible ?
 * @return boolean The hour hand visibility.
 * @see AnalogDisplay#isHourHandVisible
 */    
    public boolean getAnalogHourHandVisible()
    {
        return getAnalogDisplay().isHourHandVisible();
    }

/**
 * Set the visibility of hour hand of analog display.
 * @param v true => visible, false otherwise.
 * @see AnalogDisplay#setHourHandVisible
 */
    public void setAnalogHourHandVisible( boolean v )
    {
        getAnalogDisplay().setHourHandVisible( v );
    }

/**
 * Get the minute hand color of analog display.
 * @return the minute hand color.
 * @see AnalogDisplay#getMinuteHandColor
 */    
    public Color getAnalogMinuteHandColor()
    {
        return getAnalogDisplay().getMinuteHandColor();
    }

/**
 * Set the minute hand color of analog display.
 * @param c The new color.
 * @see AnalogDisplay#setMinuteHandColor
 */    
    public void setAnalogMinuteHandColor( Color c )
    {
        getAnalogDisplay().setMinuteHandColor( c );
    }

/**
 * Is the analog minute hand visible ?
 * @return boolean The minute hand visibility.
 * @see AnalogDisplay#isMinuteHandVisible
 */    
    public boolean getAnalogMinuteHandVisible()
    {
        return getAnalogDisplay().isMinuteHandVisible();
    }

/**
 * Set the visibility of minute hand of analog display.
 * @param v true => visible, false otherwise.
 * @see AnalogDisplay#setMinuteHandVisible
 */
    public void setAnalogMinuteHandVisible( boolean v )
    {
        getAnalogDisplay().setMinuteHandVisible( v );
    }
    
/**
 * Get the second hand color of analog display.
 * @return the second hand color.
 * @see AnalogDisplay#getSecondHandColor
 */
    public Color getAnalogSecondHandColor()
    {
        return getAnalogDisplay().getSecondHandColor();
    }

/**
 * Set the second hand color of analog display.
 * @param c The new color.
 * @see AnalogDisplay#setSecondHandColor
 */
    public void setAnalogSecondHandColor( Color c )
    {
        getAnalogDisplay().setSecondHandColor( c );
    }
    
/**
 * Is the analog second hand visible ?
 * @return boolean The second hand visibility.
 * @see AnalogDisplay#isSecondHandVisible
 */
    public boolean getAnalogSecondHandVisible()
    {
        return getAnalogDisplay().isSecondHandVisible();
    }
    
/**
 * Set the visibility of second hand of analog display.
 * @param v true => visible, false otherwise.
 * @see AnalogDisplay#setSecondHandVisible
 */
    public void setAnalogSecondHandVisible( boolean v )
    {
        getAnalogDisplay().setSecondHandVisible( v );
    }

/**
 * Get the numeral font of digital display.
 * @return The numeral font.
 * @see DigitalDisplay#getNumeralFont
 */    
    public Font getDigitalNumeralFont()
    {
        return getDigitalDisplay().getNumeralFont();
    }

/**
 * Set the numeral font of digital display.
 * @param f The new font.
 * @see DigitalDisplay#setNumeralFont
 */
    public void setDigitalNumeralFont( Font f )
    {
        getDigitalDisplay().setNumeralFont( f );
    }

/**
 * Get foreground color of digital display.
 * @return The foreground color.
 * @see DigitalDisplay#getForeGroundColor
 */    
    public Color getDigitalForeGroundColor()
    {
        return getDigitalDisplay().getForeGroundColor();
    }
/**
 * Set the foreground color of digital display.
 * @param c The new color.
 * @see DigitalDisplay#setForeGroundColor
 */
    public void setDigitalForeGroundColor( Color c )
    {
        getDigitalDisplay().setForeGroundColor( c );
    }

/**
 * Get the background color of digital display.
 * @return The background color.
 * @see DigitalDisplay#getBackGroundColor
 */    
    public Color getDigitalBackGroundColor()
    {
        return getDigitalDisplay().getBackGroundColor();
    }
/**
 * Set the background color of digital display.
 * @param c The new color.
 * @see DigitalDisplay#setBackGroundColor
 */
    public void setDigitalBackGroundColor( Color c )
    {
        getDigitalDisplay().setBackGroundColor( c );
    }

/**
 * Get the display style in digital mode.
 * @return The digital display style.
 * @see DigitalDisplay#getDisplayStyle
 */    
    public int getDigitalDisplayStyle()
    {
        return getDigitalDisplay().getDisplayStyle();
    }
/**
 * Set the display style in digital mode.
 * @param w The new digital display style.
 * @see DigitalDisplay#setDisplayStyle
 */
    public void setDigitalDisplayStyle( int w )
    {
        getDigitalDisplay().setDisplayStyle( w );
    }

    
/**
 * Get the display AMPM mode.
 * @return The AMPM mode.
 * @see DigitalDisplay#getDisplayAMPM
 */    
    public boolean getDigitalDisplayAMPM()
    {
        return getDigitalDisplay().getDisplayAMPM();
    }

/**
 * Set the displayAMPM mode
 * @param v the new mode
 * @see DigitalDisplay#setDisplayAMPM
 */ 
    public void setDigitalDisplayAMPM( boolean v )
    {
        getDigitalDisplay().setDisplayAMPM( v );
    }

/**
 * Get the Twelve Hour Mode
 * @return TwelveHour Clock Mode
 * @see DigitalDisplay#getTwelveHourMode
 */    
    public boolean getDigitalTwelveHourMode()
    {
        return getDigitalDisplay().getTwelveHourMode();
    }

/**
 * Set twelve hour mode.
 * @param v the new mode.
 * @see DigitalDisplay#setTwelveHourMode
 */
    public void setDigitalTwelveHourMode( boolean v )
    {
        getDigitalDisplay().setTwelveHourMode( v );
    }
}//end of Clock class definition
