/* @(#) Time.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class Time.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 *
 *
 */
package com.ibm.clock;

/**
 * Class to represent time of the day. The time of the hour is represented
 * by three fields -- hour, minute and second.
 */
public class Time
{
/**
 * Hours in a day.
 */
    public static final int HOURS_IN_DAY        = 24;

/**
 * Minutes in a hour.
 */
    public static final int MINUTES_IN_HOUR     = 60;

/**
 * Seconds in a minute.
 */
    public static final int SECONDS_IN_MINUTE   = 60;

    // useful constants for calculation purposes.    
    protected static final long MILLISECONDS_IN_DAY     = 86400000;
    protected static final long MILLISECONDS_IN_HOUR    = 3600000;
    protected static final long MILLISECONDS_IN_MINUTE  = 60000;
    protected static final long MILLISECONDS_IN_SECOND  = 1000;
    
    private long time = 0;

/**
 * Default constructor. The default time is taken to be zero hour, zero
 * minute, zero second.
 */    
    public Time()
    {
        this( 0 );
    }//end of c'tor

/**
 * Construct a time object with the specified time.
 * @param time The time in milliseconds. Must lie between 0 and 86399999.
 * @exception IllegalArgumentException If the argument is not proper.
 */    
    protected Time( long time )
    throws IllegalArgumentException
    {
        setInMillis( time );
    }//end of c'tor

/**
 * Construct a time field given the hour, minute and second fields.
 * @param hour The hour of the day. Must be between 0 and 23.
 * @param minute The minute of the hour. Must be between 0 and 59.
 * @param second The second of the minute. Must be between 0 and 59.
 * @exception IllegalArgumentException If any of the arguments is improper.
 */    
    public Time( int hour, int minute, int second )
    {
        if( !isProperHour( hour ) || !isProperMinute( minute ) 
                || !isProperSecond( second ) )
            throw new IllegalArgumentException();
            
        setInMillis( computeTime( hour, minute, second ) );
    }//end of c'tor

/**
 * Set the time of the day in millis.
 * @param time The time of the day in milli seconds. The time must lie between
 * 0 and 86399999.
 * @exception IllegalArgumentException If the argument is not proper.
 */    
    protected synchronized void setInMillis( long time )
    throws IllegalArgumentException
    {
        if( !isProperTime( time ) )
            throw new IllegalArgumentException();
            
        this.time = time;
    }//end of setInMillis

/**
 * Get the time of the day in milliseconds. This returns the time in milli
 * seconds from the begining of the day.
 * @return The time of the day in milliseconds.
 */    
    public long getInMillis()
    {
        return time;
    }//end of getInMillis

/**
 * Get the hour of the day.
 * @return The hour of the day as an integer lying between 0 and 23.
 */    
    public int getHour()
    {
        return (int)( getInMillis() / MILLISECONDS_IN_HOUR );
    }//end of getHour 
    
/**
 * Set the hour field.
 * @param hour The hour of the day. Must lie between 0 and 23.
 * @exception IllegalArgumentException If the argument is illegal.
 */    
    public synchronized void setHour( int hour )
    throws IllegalArgumentException
    {
        if( !isProperHour( hour ) )
            throw new IllegalArgumentException();
            
        int minute = getMinute();
        int second = getSecond();
        
        setInMillis( computeTime( hour, minute, second ) );
    }//end of setHour

/**
 * Get the minute field.
 * @return The minute field as an integer between 0 and 59.
 */
    public int getMinute()
    {
        long timeWithinHour = getInMillis() % MILLISECONDS_IN_HOUR;
        
        return (int)( timeWithinHour / MILLISECONDS_IN_MINUTE );
    }//end 
    
/**
 * Set the minute field.
 * @param minute The minute of the hour. Must lie between 0 and 59.
 * @exception IllegalArgumentException If the argument is illegal.
 */    
    public void setMinute( int minute )
    {
        if( !isProperMinute( minute ) )
            throw new IllegalArgumentException();
            
        int hour    = getHour();
        int second  = getSecond();
        
        setInMillis( computeTime( hour, minute, second ) );
    }//end of setMinute

/**
 * Get the second field.
 * @return The second field as an integer between 0 and 59.
 */
    public int getSecond()
    {
        long timeWithinMinute = getInMillis() % MILLISECONDS_IN_MINUTE;
        return ( int )( timeWithinMinute / MILLISECONDS_IN_SECOND );
    }//end 
    
/**
 * Set the second field.
 * @param second The second of the hour. Must lie between 0 and 59.
 * @exception IllegalArgumentException If the argument is illegal.
 */    
    public void setSecond( int second )
    {
        if( !isProperSecond( second ) )
            throw new IllegalArgumentException();
            
        int hour    = getHour();
        int minute  = getMinute();
        
        setInMillis( computeTime( hour, minute, second ) );
    }//end of setMinute    
    
    private boolean isProperTime( long time )
    {
        return ( time >= 0 && time < MILLISECONDS_IN_DAY );
    }//end of isProperTime
    
    private boolean isProperHour( int hour )
    {
        return ( hour >= 0 && hour < HOURS_IN_DAY );
    }//end of isProperHour
    
    private boolean isProperMinute( int minute )
    {
        return ( minute >= 0 && minute < MINUTES_IN_HOUR );
    }//end of isProperMinute
    
    private boolean isProperSecond( int second )
    {
        return ( second >= 0 && second < SECONDS_IN_MINUTE );
    }//end of isProperSecond
    
    private long computeTime( int hour, int minute, int second )
    {
        return ( ( hour * MILLISECONDS_IN_HOUR ) +
                    ( minute * MILLISECONDS_IN_MINUTE ) +
                    ( second * MILLISECONDS_IN_SECOND ) );
    }//end of computeTime
    
}//end of Time class definition
