package com.ibm.wtp.emf.workbench;
/*
 * Licensed Material - Property of IBM 
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 */

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.emf.ecore.resource.ResourceSet;

import com.ibm.wtp.emf.workbench.plugin.EMFWorkbenchPlugin;
import com.ibm.wtp.internal.emf.workbench.EMFWorkbenchContextFactory;

public class ResourceSetWorkbenchSynchronizer implements IResourceChangeListener {
	protected IProject project;
	protected ResourceSet resourceSet;
	/**Extenders that will be notified after a pre build resource change */
	protected List extenders;
	/**The delta for this project that will be broadcast to the extenders */
	protected IResourceDelta currentProjectDelta;
	private int currentEventType = -1;
	/**
	 * Constructor for ResourceSetWorkbenchSynchronizer.
	 */
	public ResourceSetWorkbenchSynchronizer(ResourceSet aResourceSet, IProject aProject) {
		resourceSet = aResourceSet;
		project = aProject;
		if (aResourceSet != null && aResourceSet instanceof ProjectResourceSet)
			((ProjectResourceSet) aResourceSet).setSynchronizer(this);
		initialize();
	}

	public IProject getProject() {
		return project;
	}

	/*
	 * @see IResourceChangeListener#resourceChanged(IResourceChangeEvent)
	 */
	public void resourceChanged(IResourceChangeEvent event) {
		currentEventType = event.getType();
		currentProjectDelta = null;
		if ((currentEventType == IResourceChangeEvent.PRE_CLOSE || currentEventType == IResourceChangeEvent.PRE_DELETE) && event.getResource().equals(getProject())) {
			release();
			notifyExtendersOfClose();
		}
	}

	protected void notifyExtendersIfNecessary() {
		if (currentEventType != IResourceChangeEvent.POST_CHANGE || extenders == null || currentProjectDelta == null)
			return;
		for (int i = 0; i < extenders.size(); i++) {
			ISynchronizerExtender extender = (ISynchronizerExtender) extenders.get(i);
			extender.projectChanged(currentProjectDelta);
		}
	}
	
	protected void notifyExtendersOfClose() {
		if (extenders != null && !extenders.isEmpty()) {
			for (int i = 0; i < extenders.size(); i++) {
				ISynchronizerExtender extender = (ISynchronizerExtender) extenders.get(i);
				extender.projectClosed();
			}
		}
	}


	protected IWorkspace getWorkspace() {
		if (getProject() == null)
			return ResourcesPlugin.getWorkspace();
		return getProject().getWorkspace();
	}
	protected void initialize() {
		getWorkspace().addResourceChangeListener(this, IResourceChangeEvent.PRE_CLOSE | IResourceChangeEvent.PRE_DELETE | IResourceChangeEvent.POST_CHANGE | IResourceChangeEvent.PRE_BUILD);
	}

	public void dispose() {
		getWorkspace().removeResourceChangeListener(this);
	}
	/**
	 * The project is going away so we need to cleanup ourself and the ResourceSet.
	 */
	protected void release() {
		if (EMFWorkbenchPlugin.isActivated()) {
			EMFWorkbenchContextFactory.INSTANCE.removeCachedProject(getProject());
			if (resourceSet instanceof ProjectResourceSet)
				 ((ProjectResourceSet) resourceSet).release();
			dispose();
		}
	}
	
	public void addExtender(ISynchronizerExtender extender) {
		if (extenders == null)
			extenders = new ArrayList(3);
		extenders.add(extender);
	}
	
	public void removeExtender(ISynchronizerExtender extender) {
		if (extenders == null)
			return;
		extenders.remove(extender);
	}
	
	public String toString() {
		return getClass().getName()+'('+((getProject() != null) ? getProject().getName(): "null")+')';  //$NON-NLS-1$
	}
	
	/**
	 * This method should be called prior to writing to an IFile from a MOF resource.
	 */
	public void preSave(IFile aFile) {
		//Default is do nothing
	}

}
