package com.ibm.wtp.emf.workbench;
/*
 * Licensed Material - Property of IBM 
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 */

import java.io.*;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.URIConverterImpl;

import com.ibm.wtp.emf.workbench.plugin.EMFWorkbenchPlugin;

/**
 * This converter is only necessary to resolve old ambiguous workbench URIs.
 * Creation date: (11/2/2000 9:36:36 AM)
 * @author: Administrator
 */
public class WorkbenchURIConverterImpl extends URIConverterImpl implements WorkbenchURIConverter {
	private final static IWorkspaceRoot WORKSPACE_ROOT = ResourcesPlugin.getWorkspace().getRoot();
	private static final String FILE_PROTOCOL = "file"; //$NON-NLS-1$
	private static final IPath INVALID_PATH = new Path("!!!!~!!!!"); //$NON-NLS-1$
	private static final IFile INVALID_FILE = WORKSPACE_ROOT.getFile(INVALID_PATH.append(INVALID_PATH));
	//Used to avoid trying to fixup the URI when getting the 
	//OutputStream
	protected boolean forceSaveRelative = false;
	protected List inputContainers;
	protected IContainer outputContainer;
	protected ResourceSetWorkbenchSynchronizer resourceSetSynchronizer;
	/**
	 * WorkbenchURIConverter constructor comment.
	 */
	public WorkbenchURIConverterImpl() {
		super();
	}
	/**
	 * WorkbenchURIConverter constructor comment.
	 */
	public WorkbenchURIConverterImpl(IContainer anInputContainer) {
		this(anInputContainer, (ResourceSetWorkbenchSynchronizer) null);
	}
	/**
	 * WorkbenchURIConverter constructor comment.
	 */
	public WorkbenchURIConverterImpl(IContainer aContainer, ResourceSetWorkbenchSynchronizer aSynchronizer) {
		this(aContainer, null, aSynchronizer);
	}
	/**
	 * WorkbenchURIConverter constructor comment.
	 */
	public WorkbenchURIConverterImpl(IContainer anInputContainer, IContainer anOutputContainer) {
		this(anInputContainer, anOutputContainer, null);
	}
	/**
	 * WorkbenchURIConverter constructor comment.
	 */
	public WorkbenchURIConverterImpl(IContainer anInputContainer, IContainer anOutputContainer, ResourceSetWorkbenchSynchronizer aSynchronizer) {
		addInputContainer(anInputContainer);
		setOutputContainer(anOutputContainer);
		resourceSetSynchronizer = aSynchronizer;
	}
	/**
	 * Add aContainer to the end of the list of inputContainers.
	 * @return java.util.List
	 */
	public void addInputContainer(IContainer aContainer) {
		if (aContainer != null && !getInputContainers().contains(aContainer))
			getInputContainers().add(aContainer);
	}

	/**
	 * @see com.ibm.wtp.emf.workbench.WorkbenchURIConverter#addInputContainers(List)
	 */
	public void addInputContainers(List containers) {
		for (int i = 0; i < containers.size(); i++) {
			addInputContainer((IContainer) containers.get(i));
		}
	}
	/**
	 * Remove aContainer from the list of inputContainers.
	 * @return java.util.List
	 */
	public boolean removeInputContainer(IContainer aContainer) {
		return getInputContainers().remove(aContainer);
	}
	/**
	 * @return java.util.List
	 */
	public List getInputContainers() {
		if (inputContainers == null)
			inputContainers = new ArrayList();
		return inputContainers;
	}
	/**
	 * Return the first container in the list of input containers.
	 * @return org.eclipse.core.resources.IContainer
	 */
	public IContainer getInputContainer() {
		if (!getInputContainers().isEmpty())
			return (IContainer) getInputContainers().get(0);
		else
			return null;
	}
	/**
	 * @return com.ibm.itp.core.api.resources.IContainer
	 */
	public IContainer getOutputContainer() {
		if (outputContainer == null)
			outputContainer = getInputContainer();
		return outputContainer;
	}
	/**
	 * Insert the method's description here.
	 * @param newOutputContainer com.ibm.itp.core.api.resources.IContainer
	 */
	public void setOutputContainer(IContainer newOutputContainer) {
		outputContainer = newOutputContainer;
	}
	/**
	 * Return an IFile for @aPath.
	 */
	public IFile getOutputFile(IPath aPath) {
		IFile file = null;
		if (getOutputContainer() != null) {
			if (forceSaveRelative)
				return primGetOutputFile(aPath);
			file = getOutputFileForPathWithContainerSegments(aPath);
			if (file != null)
				return file;
			else
				return primGetOutputFile(aPath);
		}
		return file;
	}

	protected IFile primGetOutputFile(IPath aPath) {
		return primGetFile(getOutputContainer(), aPath);
	}

	protected IFile getOutputFileForPathWithContainerSegments(IPath aPath) {
		IContainer out = getOutputContainer();
		return getFileForPathWithContainerSegments(aPath, out, false);
	}
	
	protected IFile getFileForPathWithContainerSegments(IPath aPath, IContainer container, boolean testExists) {
		IPath containerPath = null;
		IFile file = null;
		if (testExists) {
			containerPath = container.getProjectRelativePath();
			if (!containerPath.isEmpty()) {
				file = getFileForMatchingPath(aPath, containerPath, container);
				if (file != null && file.exists())
					return file;
			}
		}
		containerPath = container.getFullPath();
		file = getFileForMatchingPath(aPath, containerPath, container);
		return file;
	}

	protected IFile getFileForMatchingPath(IPath containerPath, IPath sourcePath, IContainer container) {
		int matches = 0;
		matches = containerPath.matchingFirstSegments(sourcePath);
		if (matches > 0 && matches == sourcePath.segmentCount()) {
			IPath loadPath = containerPath.removeFirstSegments(matches);
			return primGetFile(container, loadPath);
		}
		return null;
	}

	/**
	 * Check each of the inputContainers for the file.
	 */
	public IFile getFile(String uri) {
		return getFile(new Path(uri));
	}
	public IFile getFile(IPath path) {
		IFile file = null;
		if (getInputContainer() != null) {
			path = path.makeRelative();
			java.util.Iterator it = getInputContainers().iterator();
			while (it.hasNext()) {
				IContainer con = (IContainer) it.next();
				file = getInputFile(con, path);
				if (file != null && file.exists())
					return file;
			}
		}
		if (file == null)
			return INVALID_FILE;
		return file;
	}

	public IFile getOutputFile(String uri) {
		return getOutputFile(new Path(uri));
	}

	/**
	 * get the intput file from container.
	 */
	public IFile getInputFile(IContainer con, IPath path) {
		IFile file = null;
		if (WORKSPACE_ROOT.equals(con) && path.segmentCount() < 2)
			path = INVALID_PATH.append(path);
		file = primGetFile(con, path);
		if (file == null || !file.exists())
			file = getFileForPathWithContainerSegments(path, con, true);
		return file;
	}
	
	protected IFile primGetFile(IContainer container, IPath path) {
		try {
			return container.getFile(path);
		} catch (IllegalArgumentException ex) {}
		return null;
	}

	/**
	 * @see com.ibm.etools.emf.workbench.WorkbenchURIConverter#canGetUnderlyingResource(String)
	 */
	public boolean canGetUnderlyingResource(String aFileName) {
		IFile file = getFile(aFileName);
		return file != null && file.exists();
	}

	/**
	 * Returns the forceSaveRelative.
	 * @return boolean
	 */
	public boolean isForceSaveRelative() {
		return forceSaveRelative;
	}

	/**
	 * Sets the forceSaveRelative.
	 * @param forceSaveRelative The forceSaveRelative to set
	 */
	public void setForceSaveRelative(boolean forceSaveRelative) {
		this.forceSaveRelative = forceSaveRelative;
	}

	/**
	 * This converter is responsible for making the uri in the form of
	 * platform:/resource/[project name] so that we can take advantage of
	 * document relative HREFs.
	 */
	public URI normalize(URI uri) {
		URI result = uri;
		String fragment = null;
		if (uri.hasFragment()) {
			fragment = uri.fragment();
			result = uri.trimFragment();
		}
		result = getInternalURIMap().getURI(result);
		if (WorkbenchResourceHelperBase.isPlatformResourceURI(result))
			return appendFragment(result, fragment);
		if (WorkbenchResourceHelperBase.isPlatformPluginResourceURI(result)) {
			URI normalized = normalizePluginURI(result, fragment);
			return (normalized != null) ? normalized : uri;
		}
		String protocol = result.scheme();
		URI fileSearchURI = null;
		if (protocol == null) {
			fileSearchURI = normalizeEmptyProtocol(result, fragment);
			if (fileSearchURI != null)
				return fileSearchURI;
		} else if (FILE_PROTOCOL.equals(protocol)) {
			fileSearchURI = normalizeFileProtocol(result, fragment);
			if (fileSearchURI != null)
				return fileSearchURI;
		} else if (EMFWorkbenchPlugin.WORKSPACE_PROTOCOL.equals(protocol))
			return normalizeWorkspaceProtocol(result, fragment);
		return super.normalize(uri);
	}
	
	/*
     * Resolves a plugin format into the actual.
     */
	protected URI normalizePluginURI(URI uri, String fragment) {
		if (uri.segmentCount() < 2)
			return uri;	// Invalid, just let it go on.
		// See if already normalized.
		int u_scoreNdx = uri.segment(1).lastIndexOf('_');
		if (u_scoreNdx != -1) {
			// Not normalized. Remove the version to make it normalized.
			String[] segments = uri.segments();
			segments[1] = segments[1].substring(0, u_scoreNdx);
			return URI.createHierarchicalURI(uri.scheme(), uri.authority(), uri.device(), segments, uri.query(), fragment);
		} else
			return uri;
	}
	
	/*
	 * Remove the workspace protocol and create a platform URI.
	 */
	protected URI normalizeWorkspaceProtocol(URI aWorkspaceURI, String fragment) {
		URI result;
		String uriString = aWorkspaceURI.toString();
		uriString = uriString.substring(EMFWorkbenchPlugin.WORKSPACE_PROTOCOL.length() + 1);
		result = URI.createPlatformResourceURI(uriString);
		if (fragment != null)
			result = appendFragment(aWorkspaceURI, fragment);
		return result;
	}
	
	protected URI normalizeEmptyProtocol(URI aFileUri, String fragment) {
		//Make the relative path absolute and return a platform URI.
		IPath path = new Path(aFileUri.toString());
		return normalizeToWorkspaceURI(path, fragment);
	}
	
	private URI normalizeToWorkspaceURI(IPath path, String fragment) {
		URI result = null;
		IFile file = getFile(path);
		if (file == null || !file.exists())
			file = getOutputFile(path);
		if (file != null) {
			result = URI.createPlatformResourceURI(file.getFullPath().toString());
			result = appendFragment(result, fragment);
		}
		return result;
	}
	
	protected URI normalizeFileProtocol(URI aFileUri, String fragment) {
		URI result = null;
		//Make the relative path absolute and return a platform URI.
		String devicePath = aFileUri.devicePath();
		String rootLocation = workspaceRoot.getLocation().toString();
		//Test for workspace location.
		if (devicePath.startsWith(rootLocation) && devicePath.length() > rootLocation.length()) {
			//test for workspace location
			IPath path = new Path(devicePath.substring(rootLocation.length()));
			result = normalizeToWorkspaceURI(path, fragment);
		} else if (aFileUri.isRelative()) {
			result = normalizeToWorkspaceURI(new Path(aFileUri.toString()), fragment);
		} else {
			result = aFileUri;
		}
		return result;
	}
	
	protected URI appendFragment(URI result, String fragment) {
		if (fragment != null)
			return result.appendFragment(fragment);
		else
			return result;
	}

	public IFile getOutputFileWithMappingApplied(String uri) {
		URI converted = getInternalURIMap().getURI(URI.createURI(uri));
		return getOutputFile(new Path(converted.toString()));
	}

	public OutputStream createPlatformResourceOutputStream(String platformResourcePath) throws IOException {
		IFile file = WORKSPACE_ROOT.getFile(new Path(platformResourcePath));
		ProjectUtilities.ensureContainerNotReadOnly(file);
		return new WorkbenchByteArrayOutputStream(file, resourceSetSynchronizer);
	}

	
	protected URI getContainerRelativeURI(IFile aFile) {
		IPath path = WorkbenchResourceHelperBase.getPathFromContainers(inputContainers, aFile.getFullPath());
		if (path != null)
			return URI.createURI(path.toString());
		return null;
	}

	public InputStream createPlatformResourceInputStream(String platformResourcePath) throws IOException {
		IFile file = workspaceRoot.getFile(new Path(platformResourcePath));
		try {
			if (!file.isLocal(IResource.DEPTH_ONE) || !file.isSynchronized(IResource.DEPTH_ONE)) {
				try {
				    File iofile = file.getFullPath().toFile();
				    if(iofile.exists() || file.exists())
				        file.refreshLocal(IResource.DEPTH_ONE, null); 
				} catch (CoreException ce) {
					if (ce.getStatus().getCode() != IResourceStatus.WORKSPACE_LOCKED)
						throw ce;
				}
			}
			return file.getContents();
		} catch (CoreException exception) {
			throw new Resource.IOWrappedException(exception);
		}
	}

}
