/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*
 *  $RCSfile: TimerTests.java,v $
 *  $Revision: 1.4 $  $Date: 2004/08/18 14:59:52 $ 
 */
package com.ibm.wtp.common.util;

import java.text.NumberFormat;
import java.util.*;
 

/**
 * A utility class to help take timing steps and print them out.
 * @since 1.0.0
 */
public class TimerTests {
	
	/**
	 * Default TimerTests class to use when not using your own. It's a global.
	 */
	public static TimerTests basicTest = new TimerTests();
	
	/**
	 * ID to use as parent in <code>createStep(id, parentid)</code> when you want the parent to be
	 * whatever the current step is. It allows nesting without having to know who the parent should be.
	 * It should be used when you know the step will be completed before the parent could be created. In other
	 * words it should be used surrounding a step that will complete before returning. This way you will have 
	 * a valid nesting.
	 * 
	 * @see TimerTests#startStep(String, String)
	 */
	public static final String CURRENT_PARENT_ID = "current parent";
	
	protected static class TimerStep {
		protected String id;
		protected long startTime;
		protected long stopTime;
		protected List subSteps;
		protected TimerStep parentStep;
	}
	
	protected static class TimerCumulativeStep extends TimerStep {
		protected long totalTime;
		protected int count;
	}
	
	protected boolean testOn = false;
	protected List steps;
	protected Map stepMap;
	
	protected TimerStep currentParent;
	
	/**
	 * Start a step. 
	 * @param id
	 * @param parentId the id of the parent step this one is to be nested into. <code>null</code> if not nested.
	 * @return <code>true</code> if step added, <code>false</code> if for some reason it could not add it. (such as already started and not stopped).
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean startStep(String id, String parentId) {
		if (!testOn)
			return true;
		if (stepMap.containsKey(id)) {
			new IllegalStateException("Starting same step \""+id+"\" while previous is not completed.").printStackTrace();
			return false;
		}

		TimerStep step = createTimerStep(id, parentId, false);
		return step != null;
	}
	
	protected TimerStep createTimerStep(String id, String parentId, boolean cumulative) {		
		TimerStep parentStep = null;
		if (parentId == CURRENT_PARENT_ID)
			parentStep = currentParent;
		else if (parentId != null) {
			if ((parentStep = (TimerStep) stepMap.get(parentId)) == null) {
				new IllegalStateException("Starting step \""+id+"\" but parent step \""+parentId+"\" didn't exist or wasn't active.").printStackTrace();
				return null;
			}
		}
		
		TimerStep newStep = !cumulative ? new TimerStep() : new TimerCumulativeStep();
		stepMap.put(id, newStep);
		newStep.id = id;
		newStep.parentStep = parentStep;
		newStep.startTime = System.currentTimeMillis();
		if (parentStep != null) {
			if (parentStep.subSteps == null)
				parentStep.subSteps = new ArrayList();
			parentStep.subSteps.add(newStep);
		} else
			steps.add(newStep);
		
		currentParent = newStep;
		return newStep;
	}
	
	/**
	 * Stop a timer step.
	 * @param id
	 * @return <code>true</code> if timer stop, <code>false</code> if not stopped for some reason.
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean stopStep(String id) {
		if (!testOn)
			return true;
		TimerStep step = (TimerStep) stepMap.remove(id);
		if (step == null) {
			new IllegalStateException("Stopping step \""+id+"\" but was not started.").printStackTrace();
			return false;
		}
		
		step.stopTime = System.currentTimeMillis();
		if (step == currentParent)
			currentParent = step.parentStep;
		return true;
	}
	
	/**
	 * Start a brand new cumulative step. This will start accumulating. Use <code>startCumulativeStep(id)</code>
	 * for each individual step to accumulate. When <code>stopStep(id)</code> is called the avg of all of the
	 * steps start/stop cumulative for this id will be generated.
	 * 
	 * @param id
	 * @param parentId
	 * @return
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean startCumulativeStep(String id, String parentId) {
		if (!testOn)
			return true;
		
		if (stepMap.containsKey(id)) {
			return true;	// Just reuse the same step. We don't nest cumulative's of the same id.
		}

		TimerCumulativeStep step = (TimerCumulativeStep) createTimerStep(id, parentId, true);
		return step != null;
	}
	
	/**
	 * Start an individual cumulative step. This step with the matching <code>stopCumulativeStep</code>
	 * will be one step. All such steps for this id will be averaged together for the full step.
	 * 
	 * @param id
	 * @return <code>true</code> if step processed, <code>false</code> if for some reason it could not start it. (such as main accumulator step missing). 
	 * @since 1.0.0
	 */
	public synchronized boolean startCumulativeStep(String id) {
		if (!testOn)
			return true;
		
		TimerCumulativeStep step = (TimerCumulativeStep) stepMap.get(id);
		if (step == null)
			return false;
		
		step.startTime = System.currentTimeMillis();
		step.stopTime = 0;
		return true;
	}
	
	/**
	 * Stop an individual cumulating step.
	 * @param id
	 * @return <code>true</code> if step stopped, <code>false</code> if for some reason it could not stop it. (such as main accumulator step missing).
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean stopCumulativeStep(String id) {
		if (!testOn)
			return true;
		
		TimerCumulativeStep step = (TimerCumulativeStep) stepMap.get(id);
		if (step == null)
			return false;
		
		step.stopTime = System.currentTimeMillis();
		step.totalTime += (step.stopTime - step.startTime);
		step.count++;
		return true;
	}
	
	/**
	 * Start an exclusion period for a cumulative step. The time from the start of
	 * the current cumulative step til now we be part of the total for the step, but
	 * the time between now and stopExcludeCumulativeStep will be ignored.
	 * 
	 * @param id
	 * @return
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean startExcludeCumulativeStep(String id) {
		if (!testOn)
			return true;
		
		TimerCumulativeStep step = (TimerCumulativeStep) stepMap.get(id);
		if (step == null)
			return false;
		
		step.stopTime = System.currentTimeMillis();
		step.totalTime += (step.stopTime - step.startTime);
		return true;
	}
	
	/**
	 * Stop excluding time within a cumulative step. And start timing again for the step.
	 * 
	 * @param id
	 * @return
	 * 
	 * @since 1.0.0
	 */
	public synchronized boolean stopExcludeCumulativeStep(String id) {
		if (!testOn)
			return true;
		
		TimerCumulativeStep step = (TimerCumulativeStep) stepMap.get(id);
		if (step == null)
			return false;
		
		step.startTime = System.currentTimeMillis();
		step.stopTime = 0;
		return true;
	}
	
	private static final long PRINT_ALL = -1l;	// Time to use to print all steps no matter how short.
	
	/**
	 * Print out the current set of steps to sysout. Print all of them no matter how short.
	 * 
	 * 
	 * @since 1.0.0
	 */
	public synchronized void printIt() {
		if (!testOn)
			return;
		System.out.println("*** Timings Start Output:");
		printIt(steps, null, 3, PRINT_ALL, null);
		System.out.println("*** Timings Stop Output:");
	}
	
	/**
	 * Print out the current set of steps to sysout. Pass in the cut-off of only steps
	 * with a larger delta are printed.
	 * 
	 * @param minTime Don't bother printing steps that have less than this delta time.
	 * @since 1.0.0
	 */
	public synchronized void printIt(long minTime) {
		if (!testOn)
			return;
		System.out.println("*** Timings Start Output:");
		printIt(steps, null, 3, minTime, null);
		System.out.println("*** Timings Stop Output:");
	}

	protected static NumberFormat nf;
	protected long printIt(List steps, TimerStep outerStep, int indent, long cutoff, StringBuffer pending) {
		if (steps == null)
			return 0;
		TimerStep prevStep = outerStep;
		long latestEnd = 0;
		for (int i = 0; i < steps.size(); i++) {
			TimerStep step = (TimerStep) steps.get(i);
			StringBuffer strb = new StringBuffer(50);
			for (int j = 0; j < indent; j++)
				strb.append(' ');
			if (!(step instanceof TimerCumulativeStep)) {
				long stepTime = step.stopTime - step.startTime;
				if (stepTime >= cutoff) {
					// Want this one printed
					strb.append("\"");
					strb.append(step.id);
					strb.append("\" ");
					int headlen = strb.length();
					strb.append("time is ");
					strb.append(stepTime);
					if (prevStep != null && step.startTime != prevStep.stopTime) {
						long endDelta = step.startTime - prevStep.stopTime;
						if (endDelta > 0) {
							strb.append(" \t\t time since end of \"");
							strb.append(prevStep.id);
							strb.append("\" is ");
							strb.append(endDelta);
						} else {
							long startDelta = step.startTime - prevStep.startTime;
							if (startDelta > 0) {
								strb.append(" \t\t time since start of \"");
								strb.append(prevStep.id);
								strb.append("\" is ");
								strb.append(startDelta);
							}
						}
					}
					if (pending != null) {
						System.out.println(pending.toString());
						pending.setLength(0);
						pending = null;
					}
					System.out.println(strb);
					long latestSubEnd = printIt(step.subSteps, step, indent + 3, cutoff, null);
					if (latestSubEnd != 0 && latestSubEnd < step.stopTime) {
						strb.setLength(headlen);
						strb.append("time between last nested step and this end is ");
						strb.append(step.stopTime - latestSubEnd);
						System.out.println(strb);
					}
					prevStep = step;
					latestEnd = Math.max(latestEnd, step.stopTime);
				}
			} else {
				TimerCumulativeStep cumStep = (TimerCumulativeStep) step;
				strb.append("\"");
				strb.append(cumStep.id);
				if (cumStep.count > 0 && cumStep.totalTime >= cutoff) {
					strb.append("\" total time/count: ");
					strb.append(cumStep.totalTime);
					strb.append('/');
					strb.append(cumStep.count);
					strb.append(" avg: ");
					double avg = ((double) cumStep.totalTime) / cumStep.count;
					if (nf == null) {
						nf = NumberFormat.getNumberInstance();
						nf.setMaximumFractionDigits(3);
					}
					strb.append(nf.format(avg));
					if (pending != null) {
						System.out.println(pending.toString());
						pending.setLength(0);
						pending = null;
					}
					System.out.println(strb);
				} else if (step.subSteps != null) {
					if (pending != null) {
						strb.insert(0, System.getProperties().getProperty("line.separator"));
						strb.insert(0, pending.toString());
					}
					printIt(step.subSteps, step, indent + 3, cutoff, strb);
					if (pending != null && strb.length() == 0) {
						// We have a pending and we used the strb (so we used the pending) we can
						// now clear the pending.
						pending.setLength(0);
						pending = null;
					}
				}
			}
		}
		return latestEnd;
	}
	
	/**
	 * Clear the tests so that you can restart and do some more tests.
	 * 
	 * 
	 * @since 1.0.0
	 */
	public synchronized void clearTests() {
		if (!testOn)
			return;
		stepMap.clear();
		steps.clear();
		currentParent = null;
	}
	
	/**
	 * Turn this test on. If not turned on then all calls will quickly return with no errors. This allows
	 * the code to stay in place even when not debugging.
	 * <p>
	 * When turned off, it will clear the test.
	 * @param on
	 * 
	 * @since 1.0.0
	 */
	public synchronized void testState(boolean on) {
		if (on == testOn)
			return;
		if (on) {
			testOn = true;
			if (stepMap == null)
				stepMap = new HashMap();
			if (steps == null)
				steps = new ArrayList();
			currentParent = null;
		} else {
			testOn = false;
			stepMap = null;
			steps = null;
			currentParent = null;
		}
	}
}
