/* @(#) Timer.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class Timer.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 *
 *
 */
package com.ibm.clock;


class Timer
implements Runnable
{
/**
 * The default ticking frequency.
 */
    public static final long DEFAULT_TICKING_FREQUENCY  = 1000;
    
/**
 * The interval after which a single tick, i.e., a timeChanged event is 
 * generated.
 */    
    private long tickingFrequency = DEFAULT_TICKING_FREQUENCY;

/**
 * The support used to fire timeChangedEvents.
 */    
    private TimerSupport timerSupport = null;

/**
 * The timer thread.
 */    
    private Thread timerThread = null;
    private boolean stopped = false;
    
    private TimerSupport getTimerSupport()
    {
        if( timerSupport == null )
            timerSupport = new TimerSupport( this );
        
        return timerSupport;
    }//end of getTimerSupport

/**
 * Get the ticking frequency.
 */
    synchronized long getTickingFrequency()
    {
        return tickingFrequency;
    }//end of getTickingFrequency

/**
 * Set the ticking frequency.
 * @param tickingFrequency The frequency of ticking.Must be greater than
 * zero.
 * @exception IllegalArgumentException If the specified tickingFrequency has
 * an illegal value.
 */    
    synchronized void setTickingFrequency( long tickingFrequency )
    {
        if( tickingFrequency <= 0 )
            throw new IllegalArgumentException();
        
        this.tickingFrequency = tickingFrequency;
    }//end of setTickingFrequency
    
/**
 * Run the timer thread. This method starts the timer object. Once this
 * thread is started the listeners to the timer are notified
 */
    public void run()
    {
        while( !stopped )
        {
            try
            {
                Thread.sleep( getTickingFrequency() );
            }
            catch( InterruptedException e )
            {
            }
            getTimerSupport().fireTimeChangedEvent();
        }
    }//end of run

/**
 * Get the timer thread.
 */    
    private Thread getTimerThread()
    {
        if( timerThread == null )
        {
            timerThread = new Thread( this );
        }//end if
        
        return timerThread;
            
    }//end of getTimerThread

/**
 * Start timer ticking.
 */    
    public synchronized void startTicking()
    {
        try
        {
            Thread thread = getTimerThread();
            thread.start();
        }
        catch( IllegalThreadStateException e )
        {
            // already started thread do nothing.
            debug( e.toString() );
        }//end of try catch
    }//end of start

/**
 * End timer ticking.
 * @exception SecurityException If the current thread does not have access
 * rights for stopping the timer thread.
 */    
    public synchronized void stopTicking()
    throws SecurityException
    {
        stopped = true;
    }//end ticking
/**
 * Add a listener to the timer.
 * @param l The listener to be registered.
 */        
    public void addTimerListener( TimerListener l )
    {
        getTimerSupport().addTimerListener( l );
    }//end of addTimerListener

/**
 * Remove a registered listener.
 * @param l The listener to be deregistered.
 */    
    public void removeTimerListener( TimerListener l )
    {
        getTimerSupport().removeTimerListener( l );
    }//end of removeTimerListener

    private static final boolean DEBUG = true;
    private static final void debug( String s )
    {
        if( DEBUG )
            System.out.println( "Timer::" + s );
    }//end of debug
}//end of Timer
