/* @(#) DigitalDisplay.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class DigitalDisplay.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 * 01, April,1998                Initial Release
 * 08, July, 1998   Sunanda Bera Incorporated preceding 0 in hour string display. Change in
 *                               method formatTimePortion
 *
 */

package com.ibm.clock;

import java.awt.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.util.TimeZone;

/**
 * The clock has two modes of display -- Analog and Digital. The display can
 * be set to either analog or display. Both the display modes support a variety
 * of customizations. The Digital Display is responsible for the Digital Rendering
 * of the clock.
 * <p>
 * The following properties can be set on the DigitalDisplay itself.
 * <ul>
 * <li> <i>font</i> - the default font for the display of text in the
 * bean.
 * <li> <i>foregroundcolor</i> - the default foreground colour for the bean.
 * <li> <i>backgroundcolor</i> - the background colour for the bean.
 * <li> <i>displayStyle</i> - selects the display style for the
 * digital display mode. This can be one of FULL, LONG and SHORT.
 * <pre>
 * FULL :  Eg : 9:18:17 PM GMT 
 * LONG :  Eg : 9:18:17 PM
 * SHORT : Eg : 9:08 PM
 * </pre>
 * <li> <i>twelveHourMode</i> - <code>true</code> if the clock shows time
 * in a 12-hour format in the digital display mode and <code>false</code>
 * if the time is shown in a 24-hour format.
 * Note that in 24 hour mode, AMPM strings are not displayed
 * <li> <i>displayAMPM</i> - determines whether the AM/PM strings are
 * displayed in the digital display mode. Note that this is valid only
 * when the 12-hour format is selected for the digital display mode.
 * </ul> 
 *
 *
 */
public class DigitalDisplay 
extends Canvas implements ClockDisplay, Serializable, Cloneable
{

/**
* Constant representing the "full" display style
*/
    public final static int FULL = 0;             
/**
* Constant representing the "long" display style.    
*/
    public final static int LONG = 1;       
/**
* Constant representing the "short" display style.      
*/
    public final static int SHORT = 2;            
    
    private final static int ZEROHOUR = 12;

/**
* flag to display AMPM    
* set to true if the AM/PM string is to be displayed with time
*/
    private boolean displayAMPM = true; 
/**
* flag to display in twelve hour mode    
* set to true the clock should display in 12-hour mode
*/
    private boolean twelveHourMode = true; 
/**
* the current display style
* Default style is FULL mode
*/    
    private int     displayStyle ;   
/**
* the current font to display the numerals
* the default font is dialog
*/    
    private Font    numeralFont     = null; // current font for the numerals
    private Color   backGroundColor = null;
    private Color   foreGroundColor = null;    

    private final static int     DEFAULT_DISPLAY_STYLE          = DigitalDisplay.FULL;
    private final static boolean DEFAULT_DISPLAY_AMPM           = true;
    private final static boolean DEFAULT_DISPLAY_TWELVEHOURMODE = true;
    private final static Color   DEFAULT_FOREGROUND_COLOR       = Color.black;
    private final static Color   DEFAULT_BACKGROUND_COLOR       = Color.lightGray;
    private final static String        AMstr       = new String ( "AM" );  // The AM string for the the current locale. 
    private final static String        PMstr       = new String ( "PM" );  // The PM string for the the current locale. 
    private final static String        colonFormat = new String ( ":"  );
    private final static String justBlank   = " ";

    private transient int hour24 = 0;
    private transient int hour   = 0;                // Holds the current hour got from IClock.
    private transient int minute = 0;                // Holds the current minute got from IClock.
    private transient int second = 0;                // Holds the current second got from Timer IClock.
    private transient int xcoord = 0;
    private transient int ycoord = 0;
    private transient boolean isAMPM = true;
    

    private transient String        timeZoneStr;            // The string representation of the current timezone.It is used in FULL style display.
    private transient String        formattedMinute  ;      // holds the formatted minute
    private transient String        formattedSecond  ;      // holds the formatted second

    private transient String        AMPMstr     = new String ( " "  );     
    private transient StringBuffer  displayStr  = null;      // Holds the string representaion of the current time

    private TimeZone  currentTimeZone = null;
    private transient PropertyChangeSupport changesNotifier = null;    
    private transient Image offscreen = null;
    private transient Image digitalImage   = null;
/**
* The default constructor for the class 
* Default values are
* <ul>
* <li><code>displayStyle</code> default value <code>FULL</code>
* <li><code>foreGroundColor</code> default value <code>color.black</code>
* <li><code>backGroundColor</code> default value <code>color.lightGray</code>
* <li><code>numeralFont</code> default value <code>Dialog</code>
* <li><code>displayAMPM</code> default value <code>true</code>
* <li><code>twelveHourMode</code> default value <code>true</code>
* </ul>
*/
    public DigitalDisplay( )
    {   
    }   // End  default Constructor .

/**
* The constructor with arguments
* @param displayStyle the displayStyle for the bean
* @param foreGroundColor the fore ground color for the bean
* @param backGroundColor the back ground color for the bean
* @param numeralFont the font for displaying the clock numerals
* @param displayAMPM to specify whether AMPM display is required
* @param twelHourMode to specify whether the twelveHourMode display is required
*/
    public DigitalDisplay(
    int      displayStyle,
    Color    foreGroundColor,
    Color    backGroundColor,
    Font     numeralFont,
    boolean  displayAMPM,
    boolean  twelveHourMode )       
    {
        this();
        setDisplayStyle   ( displayStyle );
        setForeGroundColor( foreGroundColor );
        setBackGroundColor( backGroundColor );
        setNumeralFont    ( numeralFont );
        setDisplayAMPM    ( displayAMPM );
        setTwelveHourMode ( twelveHourMode );    
    }
/**
* Get the font for the numerals.
* @return the current font.
*/
    public Font getNumeralFont()
    {
        if( numeralFont == null )
            numeralFont = new Font ( "Dialog", Font.PLAIN, 18 );
        return numeralFont;
    }//end of getNumeralFont    

/**
* Set the Numeral font.
* @param tfont The font to be used.
* @exception NullPointerException If a null argument is passed.
*/    
    public void setNumeralFont( Font newNumeralFont )
    throws NullPointerException
    {
        if( newNumeralFont == null )
            throw new NullPointerException();
        Font oldNumeralFont = numeralFont;    
        numeralFont = newNumeralFont;
        repaint( );
        getPropertyChangeNotifier().firePropertyChange( "numeralFont", 
                    oldNumeralFont, 
                    newNumeralFont);
        
    }//end of setNumeralFont

/**
* Set the Time
* @param hours the current hour 
* @param minutes the current minute
* @param seconds the current seconds
*/   
    public void setTime( int hours, int minutes, int seconds)
    {
        if ( checkTime( hours, minutes, seconds ))
        {
        hour   = hours;
        minute = minutes;
        second = seconds;
        isAMPM   = true;
        repaint();
        }
        else
        {
        throw new IllegalArgumentException();
        }
    }   // End method setTime.

    private boolean checkTime( int hour, int minute, int second )
    {
        if ( hour < 0 || hour > 23 )
        {
            return false;
        }
        if ( minute < 0 || minute > 59 )
        {
            return false;
        }
        if ( second < 0 || second > 59 )
        {
            return false;
        }
        return true;
    }        
    
/**
* Get the Preferred Size
* @return Returns the preferred size
*/
    public Dimension getPreferredSize( )
    {
        Dimension   d = new Dimension( );
        FontMetrics fm = getFontMetrics( getNumeralFont( ));

        d.width  = fm.stringWidth( displayStrFormatter( ).toString( ));
        d.height = fm.getMaxAscent( ) + fm.getMaxDescent( );
                          
        return d;
               
    }   // End method getPreferredSize. 
                
         
/**
* The following method formats the time string
* ( hour, minute, second and AMPMstr) as per  
* the customization of the properties that 
* affect the format in which  the "displayStr" is    
* displayed.
* @return the formatted string to be displayed
*/  
    private StringBuffer displayStrFormatter( ) 
    { 

        getDisplayStr( ).setLength( 0 );
        formatTimePortion();
        formatAMPMPortion();
        return( displayStr);
                
    }   // End method timeFormatter .
                                    

 
/**
 * Overridden Invalidate for double buffering purposes.
 */
    public void invalidate()
    {
        super.invalidate();
        offscreen = null;
    }//end invalidate
    
/**
 * Overridden update for double buffering purposes.
 */
    public void update( Graphics g )
    {
        paint( g );
    }    
                                    
/**
* Repaints the bean
* @param g The graphics context
*/                                       
public void paint( Graphics og)
    {
      if(offscreen == null) {
             offscreen = createImage(getSize().width, getSize().height);
      }

      Graphics g = offscreen.getGraphics();
      g.setClip(0,0,getSize().width, getSize().height);
      
      //super.paint(og);
      Image image = getDigitalImage();
      g.drawImage(image, 0, 0, null);

      g.setFont( getNumeralFont( ));
      
      FontMetrics fontMet = getFontMetrics( getNumeralFont( ));
      int temp1 = fontMet.stringWidth( displayStrFormatter().toString( ));
      int temp2 = fontMet.getMaxAscent( ) - fontMet.getMaxDescent( );
     
      Rectangle rbounds = getBounds();
      g.setColor ( getBackGroundColor ( ) );
      g.fillRect(rbounds.x, rbounds.y, rbounds.width-1, rbounds.height-1);
      g.setColor( getForeGroundColor ( ) );
     
      xcoord = (rbounds.width - temp1)/2 ;
      ycoord = (rbounds.height + temp2) / 2;
      g.drawString( displayStrFormatter().toString(),xcoord,ycoord);                        
      og.drawImage( offscreen,0,0,null);
      
      g.dispose();

      }   // End method paint.  
        
public Image getDigitalImage() {
      if ( digitalImage == null )
      {
      digitalImage = createImage(getSize().width, getSize().height);
      Graphics g = digitalImage.getGraphics();
                
      g.setFont( getNumeralFont( ));
      FontMetrics fontMet = getFontMetrics( getNumeralFont( ));
      int temp1 = fontMet.stringWidth( displayStrFormatter().toString( ));
      int temp2 = fontMet.getMaxAscent( ) - fontMet.getMaxDescent( );
     
      Rectangle rbounds = getBounds();
      g.setColor ( getBackGroundColor ( ) );
      g.fillRect(rbounds.x, rbounds.y, rbounds.width-1, rbounds.height-1);
      g.setColor( getForeGroundColor ( ) );
     
      xcoord = (rbounds.width - temp1)/2 ;
      ycoord = (rbounds.height + temp2) / 2;
      
    //  g.drawString( displayStrFormatter().toString(),xcoord,ycoord);                
      }
      return digitalImage;        
} // end of digitalImage
      

        


/**
 * Set the Foreground color.
 * @param color The new Foreground color.
 * @exception IllegalArgumentException If the argument passed is null.
 */     
    public void setForeGroundColor(Color newForeGroundColor)
    {
        if( newForeGroundColor == null )
                throw new IllegalArgumentException();
        Color oldForeGroundColor = foreGroundColor;              
        foreGroundColor = newForeGroundColor;
        repaint( );
        getPropertyChangeNotifier().firePropertyChange( "foreGroundColor", 
                    oldForeGroundColor, 
                    newForeGroundColor);


    }

/**
 * Set the Background color.
 * @param color The new Background color.
 * @exception IllegalArgumentException If the argument passed is null.
 */     
    public void setBackGroundColor(Color newBackGroundColor)
    {
        if( newBackGroundColor == null )
                throw new IllegalArgumentException();
        Color oldBackGroundColor = backGroundColor;                  
        backGroundColor = newBackGroundColor;
        repaint( );
        getPropertyChangeNotifier().firePropertyChange( "backGroundColor", 
                    oldBackGroundColor, 
                    newBackGroundColor);


    }

/**
 * Get the current Foreground color. The default color is black.
 * @return the current Foreground color.
 */
    public Color getForeGroundColor()
    {
        if ( foreGroundColor == null )
        foreGroundColor = DEFAULT_FOREGROUND_COLOR;
        return foreGroundColor;
    }

/**
 * Get the current Background color. The default color is black.
 * @return the current Background color.
 */
    public  Color getBackGroundColor()
    {
        if ( backGroundColor == null )
        backGroundColor = DEFAULT_BACKGROUND_COLOR;
        return backGroundColor;
    }

/**
* Set the Twelve Hour Mode
* @param newMode The new Mode to be set
*/          
    public void setTwelveHourMode( boolean newTwelveHourMode)
    {
        boolean oldTwelveHourMode = twelveHourMode;
        twelveHourMode = newTwelveHourMode;
        repaint( );            
        getPropertyChangeNotifier().firePropertyChange( "twelveHourMode", 
                     new Boolean(oldTwelveHourMode), 
                     new Boolean(newTwelveHourMode));
        

        
        
    }   //End method settwelveHourMode .




/**
* Get the Twelve Hour Mode
* @return TwelveHour Clock Mode
*/          
    public boolean getTwelveHourMode( )
    {
        return twelveHourMode;
    }   //End method getTwelveHourMode.
                   
/**
* Set the displayAMPM mode
* @param newValue The AMPM value to be set
*/ 
    public void setDisplayAMPM( boolean  newDisplayAMPM)
    {
        boolean oldDisplayAMPM = displayAMPM;        
        displayAMPM = newDisplayAMPM;
        repaint( );        
        getPropertyChangeNotifier().firePropertyChange( "displayAMPM", 
                    new Boolean(oldDisplayAMPM), 
                    new Boolean(newDisplayAMPM));

    }   // End method setDisplayAMPM.  


/**
* Get the displayAMPM mode
* @return displayAMPM mode
*/ 
    public boolean getDisplayAMPM( )
    {
        return displayAMPM;
    }   // End method getdisplayAMPM.
        
/**
* Set the current display style. This can be one of the following :
* DigitalDisplay.FULL  : time along with ampm and timezone
* DigitalDisplay.LONG  : time with ampm
* DigitalDisplay.SHORT : time without seconds and with ampm
* @param newStyle The new display style to be used.
*/    
    public void setDisplayStyle( int newDisplayStyle)
    {
        int oldDisplayStyle = displayStyle;
        displayStyle = newDisplayStyle;
        repaint( );
        getPropertyChangeNotifier().firePropertyChange( "displayStyle", 
                    new Integer( oldDisplayStyle), 
                    new Integer( newDisplayStyle));
    }   //End method setDisplayStyle .


/**
* Get the current Display Style
* @return displayStyle
*/        
    public int getDisplayStyle( )
    {
        if ( ! (displayStyle==0 | displayStyle==1 | displayStyle==2) )
        displayStyle = FULL;
        return displayStyle;
    }   //End method getDisplayStyle. 

/**
* Formats the time, with all the necessary customizatiions
* checks for twelvehour mode and formats the time
*/
    private void formatTimePortion()
    {
        if( minute < 10)
        {
            formattedMinute = colonFormat + "0"+ minute;
        }   //End if .
        else
        {
            formattedMinute = colonFormat + minute;
        }   //End else .    
               
        if( second < 10)
        {
            formattedSecond = colonFormat + "0" + second;
        }   //End if .
        else
        {
            formattedSecond = colonFormat + second;
        }   //End else .    
                      
        if ( isAMPM == true )
        {                                     
        if ( twelveHourMode )
        {   
            if ( hour == ZEROHOUR )
            {
            AMPMstr  = PMstr;
            }
            else
            if ( hour < ZEROHOUR )
            {
            AMPMstr = AMstr;
            }
            else
            if ( hour > ZEROHOUR )
            {
            hour    = hour - ZEROHOUR;
            AMPMstr = PMstr;
            }
            
        }
        isAMPM = false ;
        }
        
        // CF - by Sunanda Bera 8-th July, 1998.
        // Change was done to incorporate preceding 0 in displayed hour
        // in the twenty four hour mode
        String formattedHour = "" + hour;
        if( !twelveHourMode && hour < 10 )
        {
        formattedHour = "0" + formattedHour;
        }
           
        getDisplayStr( ).append( formattedHour);
        
        // old code
        // getDisplayStr().append( hour );
        // CF
        
        getDisplayStr( ).append( formattedMinute);
        if (!( displayStyle == SHORT ))
        {         
        getDisplayStr( ).append( formattedSecond);
        }
    
    }

/**
* formats the time with the AMPM string
*/    
   
    private void formatAMPMPortion()
    {
        if( (  twelveHourMode) && ( displayAMPM))
        {
            getDisplayStr( ).append( justBlank ).append( getAMPMstr( ));
        }   // End if 
        if( displayStyle == FULL)
        {
            getDisplayStr( ).append( justBlank ).append( getTimeZoneStr());
        }   // End if 
   
    }    

/**
* Sets the current timezone
* @param tz The new timezone to be set
*/    
    public void setTimeZone( TimeZone tz )
    throws IllegalArgumentException
    {
        if( tz == null )
                throw new IllegalArgumentException(); 
        if ( tz.equals(currentTimeZone) )
                return; // no change if same
        currentTimeZone = tz;
        timeZoneStr = currentTimeZone.getID( );        
        repaint();        
    }

/**
* Get the current TimeZone
* @return TimeZone
*/
    private TimeZone getTimeZone( )
    {
        if( currentTimeZone == null )
            currentTimeZone = TimeZone.getDefault();
        return currentTimeZone;    
    }   

    private StringBuffer getDisplayStr()
    {
        if ( displayStr == null )
        displayStr = new StringBuffer();
        return displayStr;
    }
    
    private String getTimeZoneStr()
    {
        if ( timeZoneStr == null )
        {
            timeZoneStr = TimeZone.getDefault().getID();
        }
        return timeZoneStr;
    }

    private String getAMPMstr( )
    {
        if ( AMPMstr == null )
        {
        AMPMstr = "AM";
        }
        return AMPMstr;
    }

    private PropertyChangeSupport getPropertyChangeNotifier()
    {
        if( changesNotifier == null )
            changesNotifier = new PropertyChangeSupport( this );
        return changesNotifier;
    }//getPropertyChangeNotifier

/**
 * Add a property change listener.
 */
    public void addPropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().addPropertyChangeListener( pcl);
    }   


/**
 * Remove a property change listener.
 */    
    public void removePropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().removePropertyChangeListener( pcl);
    }
        
    private static final boolean DEBUG = false;
    private static final void debug( String s )
    {
        if( DEBUG )
            System.out.println( "DigitalDisplay::" + s );
    }
}   // End class DigitalDisplay.





