package com.ibm.wtp.internal.emf.workbench;
/*
 * Licensed Material - Property of IBM 
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 */

import java.util.*;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.ecore.resource.ResourceSet;

import com.ibm.wtp.common.RegistryReader;
import com.ibm.wtp.common.logger.proxy.Logger;
import com.ibm.wtp.emf.workbench.*;
import com.ibm.wtp.emf.workbench.nature.EMFNature;
import com.ibm.wtp.emf.workbench.plugin.EMFWorkbenchPlugin;
import com.ibm.wtp.internal.emf.workbench.nls.EMFWorkbenchResourceHandler;


public class EMFWorkbenchContextFactory  {
	public static final EMFWorkbenchContextFactory INSTANCE;
	
	static {
		INSTANCE = createFactoryInstance();
	}
	private final Class CONTRIBUTOR_CLASS = IEMFContextContributor.class;
	protected Map emfContextCache = new WeakHashMap();

	
	private static EMFWorkbenchContextFactory createFactoryInstance() {
		EMFWorkbenchContextFactory factory = createFactoryInstanceFromExtension();
		if (factory == null)
			factory = new EMFWorkbenchContextFactory();
		return factory;
	}
	
	private static EMFWorkbenchContextFactory createFactoryInstanceFromExtension() {
		final EMFWorkbenchContextFactory[] factoryHolder = new EMFWorkbenchContextFactory[1];
		RegistryReader reader = new RegistryReader(EMFWorkbenchPlugin.ID, "internalWorkbenchContextFactory") { //$NON-NLS-1$
			public boolean readElement(IConfigurationElement element) {
				if (element.getName().equals("factoryClass")) //$NON-NLS-1$
					try {
						factoryHolder[0] = (EMFWorkbenchContextFactory)element.createExecutableExtension("name"); //$NON-NLS-1$
						return true;
					} catch (CoreException e) {
						Logger.getLogger().logError(e);
					}				
				return false;
			}
		};
		reader.readRegistry();
		return factoryHolder[0];
	}

	/**
	 * Constructor for EMFNatureFactory.
	 */
	protected EMFWorkbenchContextFactory() {
		super();

	}


	protected void cacheEMFContext(IProject aProject, EMFWorkbenchContextBase emfContext) {
		if (aProject != null && emfContext != null)
			emfContextCache.put(aProject, emfContext);
	}

	protected EMFWorkbenchContextBase getCachedEMFContext(IProject aProject) {
		if (aProject != null)
			return (EMFWorkbenchContextBase) emfContextCache.get(aProject);
		return null;
	}

	/**
	 * <code>aProject</code> is either being closed or deleted so we need to cleanup our cache.
	 */
	public void removeCachedProject(IProject aProject) {
		if (aProject != null) 
			emfContextCache.remove(aProject); 
		
	}
	/**
	 * Return a new or existing EMFNature on <code>aProject</code>. Allow the <code>contributor</code>
	 * to contribute to the new or existing nature prior to returning.
	 */
	public EMFWorkbenchContextBase createEMFContext(IProject aProject, IEMFContextContributor contributor) {
		if (aProject == null)
			throw new IllegalStateException("[EMFWorkbenchContextBase]" + EMFWorkbenchResourceHandler.getString("EMFWorkbenchContextFactory_UI_0")); //$NON-NLS-1$ //$NON-NLS-2$
		if (!aProject.isAccessible())
			throw new IllegalStateException("[EMFWorkbenchContextBase]" + EMFWorkbenchResourceHandler.getString("EMFWorkbenchContextFactory_UI_1", new Object[]{aProject.getName()})); //$NON-NLS-1$ //$NON-NLS-2$
		EMFWorkbenchContextBase context = getCachedEMFContext(aProject);
		if (context == null) {
			context = primCreateEMFContext(aProject);
			cacheEMFContext(aProject, context);
			if (contributor == null)
				initializeEMFContextFromContributors(aProject, context);
		}
		if (contributor != null && context != null)
			contributor.primaryContributeToContext(context);
		return context;
	}

	protected void initializeEMFContextFromContributors(IProject aProject, EMFWorkbenchContextBase emfContext) {
		if (aProject == null || emfContext == null)
			return;
		List runtimes = EMFNature.getRegisteredRuntimes(aProject);
		for (int i = 0; i < runtimes.size(); i++) {
			IProjectNature nature = (IProjectNature) runtimes.get(i);
			if (nature != null && CONTRIBUTOR_CLASS.isInstance(nature))
				 ((IEMFContextContributor) nature).primaryContributeToContext(emfContext);
		}
	}

	protected boolean isNatureEnabled(IProject aProject, String natureId) {
		try {
			return aProject.isNatureEnabled(natureId);
		} catch (CoreException e) {
			return false;
		}
	}

	protected String[] getNatureIds(IProject aProject) {
		try {
			if (aProject.isAccessible())
				return aProject.getDescription().getNatureIds();
		} catch (CoreException e) {
		}
		return null;
	}

	protected IProjectNature getNature(IProject aProject, String natureId) {
		try {
			return aProject.getNature(natureId);
		} catch (CoreException e) {
			return null;
		}
	}

	protected EMFWorkbenchContextBase primCreateEMFContext(IProject aProject) {
		return new EMFWorkbenchContextBase(aProject);
	}
	/**
	 * Return an existing EMFNature on <code>aProject</code>.
	 */
	public EMFWorkbenchContextBase getEMFContext(IProject aProject) {
		return getCachedEMFContext(aProject);
	}

	public ResourceSetWorkbenchSynchronizer createSynchronizer(ResourceSet aResourceSet, IProject aProject) {
		return new ResourceSetWorkbenchSynchronizer(aResourceSet, aProject);
	}

}
