package com.ibm.calendar;

/**************************************************************************
 * File: Holiday.java
 * Version 2.0
 *
 *  Date               Author                  Changes
 *  Apr. 11,2001       Lajos Kiskeri           Created	
 * 
 * Copyright (c) 2001, IBM Corporation
 * All Rights Reserved
 ***************************************************************************/

/**
 * Holiday class holds 4 numbers, representing a holiday.      
 * Holiday can be
 * <UL>
 * <LI>a day of a month, or
 * <LI>a day on a specified week counted from the beginning or the end of a month
 * </UL>
 * Each holiday can be set as 
 * <UL>
 * <LI>a yearly repeatable holiday, or 
 * <LI>a unique holiday (celebrated just in one year).
 * </UL>
 * Holiday can be set or get as a coded string described at setHoliday of the Calendar bean.
 *
 * @see java.util.Calendar
 */
public class Holiday implements java.io.Serializable{
	
	/**
	 * Holds the year.
	 */ 
	private int year = 0;
	
	/**
	 * Holds the number of the month (January = 1)
	 */
	private int month = 0;
	
	/**
	 * Holds day information for holiday. If date is specified as a day of a month, it holds the number of the day.
	 * If date is specified as a day of a week, it holds a number representing the day in week (Sunday = 1)
	 */
	private int day = 0;
	
	/**
	 * Holds the number of the week. If this in 0, day represents a day of a month.
	 * If week is positive, day represents a day of a week counted from the beginning of the month.
	 * If week is negative, day represents a day of a week counted from the end of the month.
	 */
	private int week;

	/**
	 * Holds names of the months.
	 */
	public java.lang.String[] monthnames;
	
	/**
	 * Holds names of days.
	 */
	private java.lang.String[] daynames;
	
/**
 * Holiday constructor.
 */
public Holiday() {
	super();
	initialize();
}

/**
 * Holiday constructor. Parameter is a coded string, representing a holiday. The object holds this holiday.
 * @param s java.lang.String
 */
public Holiday(String s) {
  super();
  initialize();
  parse(s);
}

/**
 * Checks whether the parameter suites the holiday holded by the instance.
 * @return boolean
 * @param cal java.util.Calendar
 */
public boolean equals(java.util.Calendar cal) {
	if (getYear() != 0 && getYear() != cal.get(java.util.Calendar.YEAR))
	  return false;

	if (getWeek() == 0) {
	  return (getMonth()-1 == cal.get(java.util.Calendar.MONTH) && getDay() == cal.get(java.util.Calendar.DAY_OF_MONTH));
	}
	else { // if week is set
	  java.util.Calendar tempCal = (java.util.Calendar) cal.clone();
	  if (getYear() != 0)
	 	tempCal.set (java.util.Calendar.YEAR, getYear());
	  tempCal.set (java.util.Calendar.MONTH, getMonth()-1);
	  tempCal.set (java.util.Calendar.DAY_OF_WEEK, getDay()+1);
	  tempCal.set (java.util.Calendar.DAY_OF_WEEK_IN_MONTH, getWeek());
	  return (tempCal.equals(cal));
	} // else

}

/**
 * Returns the day number.
 * @return int
 */
public int getDay() {
	return day;
}

/**
 * Returns the number of the month.
 * @return int
 */
public int getMonth() {
	return month;
}

/**
 * Returns a string describing the holiday in long form.
 * @return java.lang.String
 */
public String getText() {
	String r = new String();
	if (week == 0) {
	  r = ordinals[day] + " of " + monthnames[month];  
	}
	else {
	  r = daynames[day] + " on " + ordinals[java.lang.Math.abs(week)] + " week from the " ;
	  if (week < 0) {
		r += "end of ";
	  }
	  else {
		r += "beginning of ";
	  }
	  r += monthnames[month];
	}
	if (year == 0) {
		r += " every year";
	  }
	  else {
		r += " in " + year;
	}
	return r;
}

/**
 * Returns the number of the week.
 * @return int
 */
public int getWeek() {
	return week;
}

/**
 * Returns the year.
 * @return int
 */
public int getYear() {
	return year;
}

/**
 * Initializes the daynames array.
 */
private void initDayNames() {
  daynames[0] = "Sunday";
  daynames[1] = "Monday";
  daynames[2] = "Tuesday";
  daynames[3] = "Wednesday";
  daynames[4] = "Thursday";
  daynames[5] = "Friday";
  daynames[6] = "Saturday";
}

/**
 * Creates and initializes needed arrays.
 */
public void initialize() {

  monthnames = new String[13];
  daynames = new String[7];
  ordinals = new String[32];
   
  initMonthNames();
  initOrdinals();
  initDayNames();  
}

/**
 * Initializes the monthnames array.
 */
private void initMonthNames() {
  monthnames [1] = "January";
  monthnames [2] = "February";
  monthnames [3] = "March";
  monthnames [4] = "April";
  monthnames [5] = "May";
  monthnames [6] = "June";
  monthnames [7] = "July";
  monthnames [8] = "August";
  monthnames [9] = "September";	
  monthnames [10] = "October";
  monthnames [11] = "November";
  monthnames [12] = "December";	
}

/**
 * Parses a coded string and sets the fields of the object.
 * @param s java.lang.String
 */
public void parse(String s) {
  int i1, i2;
 
  i1 = s.indexOf('/');
  if (i1 > 0) 
	year = Integer.parseInt (s.substring(0, i1));
  i1++;
  i2 = s.indexOf('/', i1);
  month = Integer.parseInt (s.substring(i1, i2));
  i1 = i2+1;
  i2 = s.indexOf('/', i1);
  if (i2 == -1) {
	day = Integer.parseInt (s.substring (i1));
  }
  else {
	day = Integer.parseInt (s.substring(i1, i2));
 	i1 = i2+1;
 	if (i1 < s.length()) {
	  week = Integer.parseInt (s.substring(i1));
 	}
  }

} // parse()

/**
 * Sets day number.
 * @param d int
 */
public void setDay(int d) {
  day = d;	
}

/**
 * Sets number of month.
 * @param m int
 */
public void setMonth(int m) {
  month = m;	
}

/**
 * Sets number of week.
 * @param w int
 */
public void setWeek(int w) {
  week = w;	
}

/**
 * Sets year field.
 * @param y int
 */
public void setYear(int y) {
  year = y;	
}

/**
 * Returns the coded string form of the held holiday.
 * @return java.lang.String
 */
public String toString() {
	String r = new String("");
	if (year != 0) r += year; 		
	r += "/" + month + "/" + day ;
	if (week != 0) r += "/" + week;
	return r;
}

	 /**
	 * Holds English ordinar numbers (1-31).
	 */
	private String[] ordinals;
	
/**
 * Initializes the ordinals array.
 */
private void initOrdinals() {
  int i;
	
  ordinals [1] = "1st";
  ordinals [2] = "2nd";
  ordinals [3] = "3rd";
 
  for (i = 4; i < 21; i++) {
	ordinals [i] = i + "th";
  } 
  
  ordinals [21] = "21st";
  ordinals [22] = "22nd";
  ordinals [23] = "23rd";

  for (i = 24; i < 31; i++) {
	ordinals [i] = i + "th";
  } 
  
  ordinals [31] = "31st";
}

}