/* @(#) AnalogDisplay.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class AnalogDisplay.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 * 01, April    Sunanda Bera    Initial Release.
 * 12, March    Sunanda Bera    Added hour, minute and second hand length
 *                              computation in getDialImage.
 * 08, July, 1998 Sunanda Bera  Changed paint and getDialImage method to reflect changes
 *                              in background color of container.
 *
 */
package com.ibm.clock;

import java.awt.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;

/**
 * The analog display class.
 * The features that can be customized are :
 * <ul>
 * <li> <i>borderWidth</i> - the width of the clock border.
 * <li> <i>borderColor</i> - the border color.
 * <li> <i>dialColor</i> - the color of the clock dial.
 * <li> <i>tickColor</i> - the color of the ticks.
 * <li> <i>tickStyle</i> - the tick style. The tick styles supported are all
 * ticks( one tick for each minute/second), major( one tick for each hour ),
 * minimal( one tick for each quarter hour ) and none.
 * <li> <i>numeralStyle</i> - the numeral style. The supported styles are plain( arabic
 * numerals ) and roman.
 * <li> <i>numeralColor</i> - the color of the numerals.
 * <li> <i>numeralFont</i> - the font of the numerals.
 * <li> <i>hourHandColor</i> - the hour hand color.
 * <li> <i>hourHandVisible</i> - the hour hand visibility.
 * <li> <i>minuteHandColor</i> - the minute hand color.
 * <li> <i>minuteHandVisible</i> - the minute hand visibility.
 * <li> <i>secondHandColor</i> - the second hand color.
 * <li> <i>secondHandVisible</i> - the second hand visibility.
 * </ul>
 * See the respective getters and setters for details.
 */
public class AnalogDisplay
extends Canvas implements ClockDisplay, Serializable, Cloneable
{
/**
 * tick style all.
 * @see #setTickStyle
 * @see #getTickStyle
 */
    public static final int TICK_STYLE_ALL      = 0;
/**
 * tick style major.
 * @see #setTickStyle
 * @see #getTickStyle
 */
    public static final int TICK_STYLE_MAJOR    = 1;
/**
 * tick style minimal.
 * @see #setTickStyle
 * @see #getTickStyle
 */
    public static final int TICK_STYLE_MINIMAL  = 2;
/**
 * tick style none.
 * @see #setTickStyle
 * @see #getTickStyle
 */
    public static final int TICK_STYLE_NONE     = 3;

/**
 * Roman numeral style.
 * @see #setNumeralStyle
 * @see #getNumeralStyle
 */    
    public static final int NUMERAL_ROMAN       = 0;
/**
 * Plain numeral style.
 * @see #setNumeralStyle
 * @see #getNumeralStyle
 */
    public static final int NUMERAL_PLAIN       = 1;
    
    private static final int DEFAULT_BORDER_WIDTH       = 6;
    private static final Color DEFAULT_BORDER_COLOR     = Color.black;
    private static final Color DEFAULT_DIAL_COLOR       = Color.white;
    
    private static final Color DEFAULT_TICK_COLOR   = Color.green;
    private static final int DEFAULT_TICK_STYLE     = TICK_STYLE_MAJOR;
    private static final int DEFAULT_TICK_WIDTH     = 3;
    
    private static final int DEFAULT_NUMERAL_STYLE      = NUMERAL_PLAIN;
    private static final Color DEFAULT_NUMERAL_COLOR    = Color.black;

    private static final Color DEFAULT_HOUR_HAND_COLOR      = Color.black;
    private static final Color DEFAULT_MINUTE_HAND_COLOR    = Color.black;
    private static final Color DEFAULT_SECOND_HAND_COLOR    = Color.black;
    
    private static final int DEFAULT_HOUR_HAND_THICKNESS    = 6;
    private static final int DEFAULT_MINUTE_HAND_THICKNESS  = 6;
    private static final int DEFAULT_SECOND_HAND_THICKNESS  = 1;
    
    private static final int DEFAULT_NUMERAL_INSET  = 8;
    private static final int DEFAULT_DIAL_RADIUS    = 100;
    
  
    
    private static final String [] plainNumerals =
                {  "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "1", "2" };
    private static final String [] romanNumerals =
                { "III", "IV", "V", "VI", "VII", "VIII", "IX", "X", "XI", "XII", "I", "II" };
                    
    private transient ClockHand hourHand   = null;
    private transient ClockHand minuteHand = null;
    private transient ClockHand secondHand = null;

    private Color borderColor   = null;    
    private int borderWidth     = DEFAULT_BORDER_WIDTH;
    private Color dialColor     = null;
    private Color tickColor     = null;
    private int tickStyle       = -1;    
    private int numeralStyle    = -1;    
    private Color numeralColor  = null;
    private Font numeralFont    = null;
    
    private Color hourHandColor         = null;
    private boolean hourHandVisible     = true;
    private Color minuteHandColor       = null;
    private boolean minuteHandVisible   = true;
    private Color secondHandColor       = null;
    private boolean secondHandVisible   = true;
   
    private int tickWidth       = DEFAULT_TICK_WIDTH;      
    
    private transient Image dialImage   = null;
    private transient int hour      = 0;
    private transient int minute    = 0;
    private transient int second    = 0;
    
    private transient double [] sinTheta  = null;
    private transient double [] cosTheta  = null;
    
    private transient int sideLength = 0;
    private transient PropertyChangeSupport changesNotifier = null;
    private transient Image offscreen = null;
/**
 * Default constructor for analog display. The default values for the customizable
 * properties are listed below :
 * <ul>
 * <li> <i>borderWidth</i> - 6.
 * <li> <i>borderColor</i> - <code>Color.black</code>
 * <li> <i>dialColor</i> - <code>Color.white</code>
 * <li> <i>tickColor</i> - <code>Color.green</code>
 * <li> <i>tickStyle</i> - <code>AnalogDisplay.TICK_STYLE_MAJOR</code>
 * <li> <i>numeralStyle</i> - <code>AnalogDisplay.NUMERAL_PLAIN</code>
 * <li> <i>numeralColor</i> - <code>Color.black</code>
 * <li> <i>numeralFont</i> - the default component font.
 * <li> <i>hourHandColor</i> - <code>Color.black</code>
 * <li> <i>hourHandVisibility</i> - <code>true</code>
 * <li> <i>minuteHandColor</i> - <code>Color.black</code>
 * <li> <i>minuteHandVisibility</i> - <code>true</code>
 * <li> <i>secondHandColor</i> - <code>Color.black</code>
 * <li> <i>secondHandVisibility</i> - <code>true</code>
 * </ul>
 */
    public AnalogDisplay()
    {
    }//end of c'tor

/**
 * Constructor which takes all the customization features as parameter.
 * @param borderColor The border color.
 * @param borderWidth The width of the border.
 * @param dialColor The dial color.
 * @param tickColor The tick color.
 * @param tickStyle The tick style.
 * @param numeralStyle The numeral style.
 * @param numeralColor The numeral color.
 * @param numeralFont The numeral font.
 * @param hourHandColor The hour hand color.
 * @param hourHandVisibile The hour hand visibility.
 * @param minuteHandColor The minute hand color.
 * @param minuteHandVisibile The minute hand visibility.
 * @param secondHandColor The second hand color.
 * @param secondHandVisible The second hand visibility.
 * @exception IllegalArgumentException if any of arguments have an illegal value.
 */    
    public AnalogDisplay( Color borderColor, int borderWidth, Color dialColor,
                            Color tickColor, int tickStyle, int numeralStyle,
                            Color numeralColor, Font numeralFont, Color hourHandColor,
                            boolean hourHandVisible, Color minuteHandColor,
                            boolean minuteHandVisible, Color secondHandColor,
                            boolean secondHandVisible )
    throws IllegalArgumentException
    {
        this();
        if( borderColor == null || dialColor == null || tickColor == null 
            || numeralColor == null || numeralFont == null 
            || hourHandColor == null  || minuteHandColor == null
            || secondHandColor == null )
            throw new IllegalArgumentException();
        if( !isProperBorderWidth( borderWidth ) || !isProperTickStyle( tickStyle )
            || !isProperNumeralStyle( numeralStyle ) )
            throw new IllegalArgumentException();
        
        this.borderColor        = borderColor;
        this.borderWidth        = borderWidth;
        this.dialColor          = dialColor;
        this.tickColor          = tickColor;
        this.tickStyle          = tickStyle;
        this.numeralStyle       = numeralStyle;
        this.numeralColor       = numeralColor;
        this.numeralFont        = numeralFont;
        this.hourHandColor      = hourHandColor;
        this.hourHandVisible    = hourHandVisible;
        this.minuteHandColor    = minuteHandColor;
        this.minuteHandVisible  = minuteHandVisible;
        this.secondHandColor    = secondHandColor;
        this.secondHandVisible  = secondHandVisible;
    }//end of AnalogDisplay
    
/**
 * Get the font for the numerals.
 * @return the current font.
 */
    public Font getNumeralFont()
    {
        if( numeralFont == null )
            numeralFont = getFont();
        return numeralFont;
    }//end of getNumeralFont    

/**
 * Set the numeral font.
 * @param font The font to be used.
 * @exception IllegalArgumentException If a null argument is passed.
 */    
    public void setNumeralFont( Font newNumeralFont )
    throws IllegalArgumentException
    {
        if( newNumeralFont == null )
            throw new IllegalArgumentException();
        Font oldNumeralFont = this.numeralFont;    
        this.numeralFont = newNumeralFont;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange( "numeralFont", 
                    oldNumeralFont, 
                    newNumeralFont);

    }//end of setNumeralFont
/**
 * Get the current hour hand color. The default color is black.
 * @return the current hour hand color.
 */
    public Color getHourHandColor()
    {
        if( hourHandColor == null )
            setHourHandColor( DEFAULT_HOUR_HAND_COLOR );
        return hourHandColor;
    }//end of getHourHandColor

/**
 * Set the hour hand color.
 * @param color The new hour hand color.
 * @exception IllegalArgumentException If the argument passed is null.
 */    
    public void setHourHandColor( Color newHourHandColor )
    throws IllegalArgumentException
    {
        if( newHourHandColor == null )
            throw new IllegalArgumentException();
        Color oldHourHandColor = this.hourHandColor;           
        this.hourHandColor     = newHourHandColor;
        getHourHand().setColor( newHourHandColor );
        repaint();
        
        getPropertyChangeNotifier().firePropertyChange( "hourHandColor", 
                    oldHourHandColor, 
                    newHourHandColor);
    }//end of setHourHandColor

/**
 * Get the current minute hand color. The default color is black.
 * @return the current minute hand color.
 */
    public Color getMinuteHandColor()
    {
        if( minuteHandColor == null )
            setMinuteHandColor( DEFAULT_HOUR_HAND_COLOR );
        return minuteHandColor;
    }//end of getMinuteHandColor

/**
 * Set the minute hand color.
 * @param color The new minute hand color.
 * @exception IllegalArgumentException If the argument passed is null.
 */    
    public void setMinuteHandColor( Color newMinuteHandColor )
    throws IllegalArgumentException
    {
        if( newMinuteHandColor == null )
            throw new IllegalArgumentException();
            
        Color oldMinuteHandColor = this.minuteHandColor;
        this.minuteHandColor     = newMinuteHandColor;
        getMinuteHand().setColor( newMinuteHandColor );
        repaint();
        getPropertyChangeNotifier().firePropertyChange( "minuteHandColor", 
                    oldMinuteHandColor, 
                    newMinuteHandColor);
    }//end of setMinuteHandColor

/**
 * Get the current second hand color. The default color is black.
 * @return the current second hand color.
 */
    public Color getSecondHandColor()
    {
        if( secondHandColor == null )
            setSecondHandColor( DEFAULT_HOUR_HAND_COLOR );
        return secondHandColor;
    }//end of getSecondHandColor

/**
 * Set the second hand color.
 * @param color The new second hand color.
 * @exception IllegalArgumentException If the argument passed is null.
 */    
    public void setSecondHandColor( Color newSecondHandColor )
    throws IllegalArgumentException
    {
        if( newSecondHandColor == null )
            throw new IllegalArgumentException();
        Color oldSecondHandColor = this.secondHandColor;    
        this.secondHandColor     = newSecondHandColor;
        getSecondHand().setColor( newSecondHandColor );
        repaint();
        getPropertyChangeNotifier().firePropertyChange("secondHandColor",
                                    oldSecondHandColor,
                                    newSecondHandColor);
    }//end of setSecondHandColor

/**
 * Is the hour hand visible ?
 * @return true if visible, false otherwise.
 */    
    public boolean isHourHandVisible()
    {
        return hourHandVisible;
    }//end isHourHandVisible

/**
 * Make the hour hand visible / invisible.
 * @param true => the hour hand will be rendered,
 *        false => the hour hand will not be rendered.
 */    
    public void setHourHandVisible( boolean newVisible )
    {
        boolean oldVisible = this.hourHandVisible;        
        this.hourHandVisible = newVisible;
        getHourHand().setVisible( newVisible );
        repaint();
        getPropertyChangeNotifier().firePropertyChange("hourHandVisible",
                                    new Boolean(oldVisible),
                                    new Boolean(newVisible));
    }//end of setHourHandVisible
    
/**
 * Is the minute hand visible ?
 * @return true if visible, false otherwise.
 */    
    public boolean isMinuteHandVisible()
    {
        return minuteHandVisible;
    }//end isMinuteHandVisible

/**
 * Make the minute hand visible / invisible.
 * @param true => the minute hand will be rendered,
 *        false => the minute hand will not be rendered.
 */    
    public void setMinuteHandVisible( boolean newVisible )
    {
        boolean oldVisible = this.minuteHandVisible;
        this.minuteHandVisible = newVisible;
        getMinuteHand().setVisible( newVisible );
        repaint();
        getPropertyChangeNotifier().firePropertyChange("minuteHandVisible",
                                                        new Boolean(oldVisible),
                                                        new Boolean(newVisible));
    }//end of setMinuteHandVisible

/**
 * Is the hour hand visible ?
 * @return true if visible, false otherwise.
 */    
    public boolean isSecondHandVisible()
    {
        return secondHandVisible;
    }//end isSecondHandVisible

/**
 * Make the second hand visible / invisible.
 * @param true => the second hand will be rendered,
 *        false => the second hand will not be rendered.
 */    
    public void setSecondHandVisible( boolean newVisible )
    {
        boolean oldVisible = this.secondHandVisible;
        this.secondHandVisible = newVisible;
        getSecondHand().setVisible( newVisible );
        repaint();
        getPropertyChangeNotifier().firePropertyChange("secondHandVisible",
                                                        new Boolean(oldVisible),
                                                        new Boolean(newVisible));
    }//end of setSecondHandVisible    
    
    private int computeHourHandLength()
    {
        int radius = computeMinuteHandLength();
        radius -= getTickWidth();
        radius = (int) ( 0.75 * radius );
        if( radius < 0 )
            radius = 0;
        return radius;
    }//end of computeHourHandLength
    
    private int computeSecondHandLength()
    {
        int radius = getSideLength() / 2;
        radius -= getBorderWidth();
        radius -= getBorderInset();
        if( radius < 0 )
            radius = 0;
        return radius;
    }//end of computeSecondHandLength
    
    private int computeMinuteHandLength()
    {
        int radius = computeSecondHandLength();
        radius -= getNumeralInset();
        radius -= getMinuteHand().getWidth();
        radius -= 1;
        if( radius < 0 )
            radius = 0;
        return radius;
    }//end of computeMinuteHandLength
    
    private void setHour( int hour )
    {
        this.hour = hour;
    }//end of setHour
    
    private int getHour()
    {
        return hour;
    }//end of getHour
    
    private void setMinute( int minute )
    {
        this.minute = minute;
    }//end of setMinute
    
    private int getMinute()
    {
        return minute;
    }//end of getMinute
    
    private void setSecond( int second )
    {
        this.second = second;
    }// end of setSecond
    
    private int getSecond()
    {
        return second;
    }//end of getSecond
    
/**
 * Get the current tick style.
 * @return The current tick style.
 * @see #setTickStyle
 */    
    public int getTickStyle()
    {
        if( tickStyle < 0 )
            tickStyle = DEFAULT_TICK_STYLE;
        return tickStyle;
    }//end of getTickStyle

/**
 * Set the current tick style. This can be one of the following :
 * <ol><li> AnalogDisplay.TICK_STYLE_ALL : all ticks.
 *     <li> AnalogDisplay.TICK_STYLE_MAJOR : one tick for each hour.
 *     <li> AnalogDisplay.TICK_STYLE_MINIMAL : only for 3, 6, 9 and 12.
 *     <li> AnalogDisplay.TICK_STYLE_NONE : no ticks.
 * </ol>
 * @param style The tick style to be used.
 * @exception IllegalArgumentException If the tick style is not one of the
 * above mentioned.
 */    
    public void setTickStyle( int newStyle )
    throws IllegalArgumentException
    {
        if( !isProperTickStyle( newStyle ) )
            throw new IllegalArgumentException();
        int oldStyle = this.tickStyle;    
        this.tickStyle = newStyle;
        invalidateDialImage();
        repaint();
        
        getPropertyChangeNotifier().firePropertyChange("tickStyle",
                                                        new Integer(oldStyle),
                                                        new Integer(newStyle));
    }//end of setTickStyle

/**
 * Is this tick style supported ?
 * @param style The specified style.
 * @return true if supported, false otherwise.
 * @see #setTickStyle
 */    
    protected boolean isProperTickStyle( int style )
    {
        boolean isProper = false;
        switch( style )
        {
            case TICK_STYLE_ALL:
            case TICK_STYLE_MAJOR:
            case TICK_STYLE_MINIMAL:
            case TICK_STYLE_NONE:
                isProper = true;
                break;
            default:
        }//end switch
        
        return isProper;
    }//end of isProperTickStyle

/**
 * Get the current numeral style.
 * @return The current numeral style.
 * @see #setNumeralStyle
 */    
    public int getNumeralStyle()
    {
        if( numeralStyle < 0 )
            numeralStyle = DEFAULT_NUMERAL_STYLE;
        return numeralStyle;
    }//end of

/**
 * Set the numeral style. This can be either NUMERAL_PLAIN( 1, 2, 3, ... ) or
 * NUMERAL_ROMAN( I, II, III, IV, ... ).
 * @param style The style to be set.
 * @exception IllegalArgumentException If the specified numeral style is not
 * one of those mentioned above.
 */    
    public void setNumeralStyle( int newStyle )
    {
        if( !isProperNumeralStyle( newStyle ) )
            throw new IllegalArgumentException();
        int oldStyle = this.numeralStyle;    
        this.numeralStyle = newStyle;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("numeralStyle",
                                                        new Integer(oldStyle),
                                                        new Integer(newStyle));
    }//end of setNumeralStyle

/**
 * Is the numeral style supported ?
 * @param style The specified numeral style.
 * @return true if supported, false otherwise.
 */    
    protected boolean isProperNumeralStyle( int style )
    {
        boolean isProper = false;
        switch( style )
        {
            case NUMERAL_PLAIN:
            case NUMERAL_ROMAN:
                isProper = true;
                break;
            default:
        }//end switch
        
        return isProper;
    }//end of isProperNumeralStyle

/**
 * setBounds. overridden method.
 */    
    public void setBounds( int x, int y, int width, int height )
    {
        super.setBounds( x, y, width, height );
        
        if( width <= height )
        {
            sideLength = width;
        }
        else
        {
            sideLength = height;
        }//end if-else

        getHourHand().setLength( computeHourHandLength() );
        getMinuteHand().setLength( computeMinuteHandLength() );
        getSecondHand().setLength( computeSecondHandLength() );
        invalidateDialImage();
    }//end of reshape
    

    private void invalidateDialImage()
    {
        dialImage = null;
    }//end of invalidateDialImage
    
    private int getSideLength()
    {
        return sideLength;
    }//end of getSideLength

/**
 * Get number string i.
 * i = 0 corresponds for hour number 3, i=1 for 4,...., i=11 for 12.
 */   
    protected String getNumberString( int i )
    {
        switch( getNumeralStyle() )
        {
            case NUMERAL_PLAIN:
                return plainNumerals[ i ];
            case NUMERAL_ROMAN:
                return romanNumerals[ i ];
            default:
        }//end switch
        return null;
    }//end of getNumberString

/**
 * Get the numeral color.
 * @return The current numeral color.
 */    
    public Color getNumeralColor()
    {
        if( numeralColor == null )
            numeralColor = DEFAULT_NUMERAL_COLOR;
        
        return numeralColor;
    }//end of getNumberColor

/**
 * Set the numeral color.
 * @param color The new numeral color.
 * @exception IllegalArgumentException If the argument passed is null.
 */    
    public void setNumeralColor( Color newColor )
    throws IllegalArgumentException
    {
        if( newColor == null )
            throw new IllegalArgumentException();
        Color oldColor = this.numeralColor;    
        this.numeralColor = newColor;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("numeralColor",
                                   oldColor,
                                   newColor);
        
    }//end of setNumeralColor
    
    private double getSinTheta( int i )
    {
        if( sinTheta == null )
        {
            sinTheta = new double[60];
            for( int j = 0; j < 60; j++ )
            {
                sinTheta[j] = Math.sin( j * ( Math.PI / 30 ) );
            }//end for
        }//end if 
        
        return sinTheta[i];
    }//end of getSinTheta
    
    private double getCosTheta( int i )
    {
        if( cosTheta == null )
        {
            cosTheta = new double[60];
            for( int j = 0; j < 60 ; j++ )
            {
                cosTheta[j] = Math.cos( j * ( Math.PI / 30 ) );
            }//end for
        }//end if
        
        return cosTheta[i];
    }//end of getCosTheta

/**
 * Get the tick color.
 * @return The color in which ticks are drawn.
 */
    public Color getTickColor()
    {
        if( tickColor == null )
            tickColor = DEFAULT_TICK_COLOR;
        return tickColor;
    }//end of getTickColor

/**
 * Set the color in which the ticks must be drawn.
 * @param color The tick color. This cannot be null.
 * @throws IllegalArgumentException If the specified color is null.
 */    
    public void setTickColor( Color newColor )
    throws IllegalArgumentException
    {
        if( newColor == null )
            throw new IllegalArgumentException();
        Color oldColor = this.tickColor;
        this.tickColor = newColor;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("tickColor",
                                   oldColor,
                                   newColor);
    }//end of setTickColor

/**
 * Get the current border color. By default this is black.
 * @return The current border color. 
 */
    public Color getBorderColor()
    {
        if( borderColor == null )
            borderColor = DEFAULT_BORDER_COLOR;
        return borderColor;
    }//end of getBorderColor

/**
 * Set the border color.
 * @param color The new color.
 * @exception IllegalArgumentException if the specified argument is null.
 */
    public void setBorderColor( Color newColor )
    {
        if( newColor == null )
            throw new IllegalArgumentException();
        Color oldColor = this.borderColor;    
        this.borderColor = newColor;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("borderColor",
                                   oldColor,
                                   newColor);
    }//end

/**
 * Get the color of the dial. By default this is white.
 * @return The current dial color.
 */
    public Color getDialColor()
    {
        if( dialColor == null )
            dialColor = DEFAULT_DIAL_COLOR;
        return dialColor;
    }//end of getDialColor

/**
 * Set the dial color.
 * @param color The new dial color.
 * @exception IllegalArgumentException If the argument passed is null.
 */
    public void setDialColor( Color newColor )
    throws IllegalArgumentException
    {
        if( newColor == null )
            throw new IllegalArgumentException();
        Color oldColor = this.dialColor;    
        this.dialColor = newColor;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("dialColor",
                                                        oldColor,
                                                        newColor);
    }//end of setDialColor

/**
 * Get the border width. By default this is 6 pixels.
 * @return The current border width.
 */
    public int getBorderWidth()
    {
        return borderWidth;
    }//end of getBorderWidth
    
/**
 * Set the border width.
 * @param width The new border width.
 * @exception IllegalArgumentException If the width specified is not proper.
 * @see #isProperBorderWidth
 */
    public void setBorderWidth( int newWidth )
    {
        if( !isProperBorderWidth( newWidth ) )
            throw new IllegalArgumentException();
        int oldWidth = this.borderWidth;    
        this.borderWidth = newWidth;
        invalidateDialImage();
        repaint();
        getPropertyChangeNotifier().firePropertyChange("borderWidth",
                                                        new Integer(oldWidth),
                                                        new Integer(newWidth));
        
        
    }//end of setBorderWidth

/**
 * Is the specified width proper ?
 * @param width The specified width.
 * @return true if proper, false otherwise.
 * @see #setBorderWidth
 */
    protected boolean isProperBorderWidth( int width )
    {
        if( width < 1 )
            return false;
        return true;
    }//end of isProperBorderWidth
    
    private int getTickWidth()
    {
        return tickWidth;
    }//end of getTickWidth

/**
 * Set the time to be shown. This calls repaint.
 * @param hour The hour of the day.
 * @param minute The minute of the hour.
 * @param second The second of the minute.
 * throws IllegalArgumentException
 */    
    public synchronized void setTime( int hour, int minute, int second )
    {

        if ( checkTime(hour, minute, second))
        {     
        setHour( hour );
        setMinute( minute );
        setSecond( second );
        repaint();
        }
        else
        {
        throw new IllegalArgumentException();
        }
    }//end of setTime
    
private boolean checkTime( int hour, int minute, int second )
{
        if ( hour < 0 || hour > 23 )
        {
            return false;
        }
        if ( minute < 0 || minute > 59 )
        {
            return false;
        }
        if ( second < 0 || second > 59 )
        {
            return false;
        }
        return true;
}// end of method checkTime        
            
                                

/**
 * Get the preferred size of this component.
 * @return The preffered size.
 */    
    public Dimension getPreferredSize()
    {
        int radius = 0;
        radius += getBorderWidth();
        radius += getBorderInset();
        radius += getNumeralInset();
        radius += getTickWidth();
        radius += getPreferredDialRadius();
        
        return new Dimension( radius, radius );
    }//end of getPreferredSize
    
    private int getPreferredDialRadius()
    {
        return DEFAULT_DIAL_RADIUS;
    }//end getPreferredDialRadius

/**
 * Overridden Invalidate for double buffering purposes.
 */
    public void invalidate()
    {
        super.invalidate();
        offscreen = null;
        invalidateDialImage();
    }//end invalidate
    
/**
 * Overridden update for double buffering purposes.
 */
    public void update( Graphics g )
    {
        paint( g );
    }//end of update
    
/**
 * Paint.
 * @param og The graphics context.
 */    
    public void paint( Graphics og )
    {
        if( offscreen == null )
            offscreen = createImage( getSize().width, getSize().height );
            
        Graphics g = offscreen.getGraphics();
        // CF -- Sunanda Bera 08 July 1998. So that background changes of 
        // container of IClock bean is reflected.
        Color oldColor = g.getColor();
        g.setColor( getBackground() );
        g.fillRect( 0, 0, getSize().width, getSize().height );
        g.setColor( oldColor );
        // CF
        g.setClip( 0, 0, getSize().width, getSize().height );
        
        Image image = getDialImage();
        g.drawImage( image, 0, 0, null );
        
        Rectangle r = getBounds();
        int correctedSecond = ( getSecond() + 45 ) % 60;
        int correctedMinute = ( getMinute() + 45 ) % 60;
        int correctedHour   = ( getHour() + 9 ) % 12;
        
        double secondAngle  = correctedSecond  * ( Math.PI / 30 );
        double minuteAngle  = correctedMinute * ( Math.PI / 30 );
        double hourAngle    = correctedHour * ( Math.PI / 6 );
        
        // correct hourAngle for minute part.
        hourAngle += ( ( getMinute() / 12 ) * ( Math.PI / 30 ) );
        
        int x = getSideLength() / 2;
        int y = getSideLength() / 2;
        getHourHand().paint( g, x, y, hourAngle );
        getMinuteHand().paint( g, x, y, minuteAngle );
        getSecondHand().paint( g, x, y, secondAngle );
        
        og.drawImage( offscreen, 0, 0, null );
        g.dispose();
    }//end of paint
    
    private int getBorderInset()
    {
        FontMetrics fm = getFontMetrics( getNumeralFont() );

        int fontHeight = fm.getHeight();

        int width1 = fm.stringWidth( getNumberString( 0 ) );
        int width2 = fm.stringWidth( getNumberString( 6 ) );
        int inset = ( width1 > width2 )? width1 : width2;
        inset = ( inset > fontHeight )? inset : fontHeight;
        
        inset = ( inset + 1 ) / 2;
        return inset;
    }//end of getBorderInset
    
    private int getNumeralInset()
    {
        return getBorderInset();
    }//end of getTickInset
    
    private Image getDialImage()
    {
        if( dialImage == null )
        {
            int x       = 0;
            int y       = 0;
            int length  = getSideLength();
            
            dialImage = createImage( length, length );
            
            Graphics graphics = dialImage.getGraphics();
            // CF -- Sunanda Bera 08 July 1998. So that background changes of
            // container of IClock bean is reflected.
            Color oldColor = graphics.getColor();
            graphics.setColor( getBackground() );
            graphics.fillRect( 0, 0, length, length );
            graphics.setColor( oldColor );
            // CF
            drawBorder( graphics, x, y, length );
            
            int borderWidth = getBorderWidth()+ getBorderInset();
            x += borderWidth;
            y += borderWidth;
            length -= 2*borderWidth;
            
            drawNumbers( graphics, x, y, length );
            
            int tickWidth = getTickWidth() + getNumeralInset();
            x +=  tickWidth;
            y +=  tickWidth;
            length  -= 2*tickWidth;
            
            drawTicks( graphics, x, y, length );
            
            // recompute hand lengths
            getHourHand().setLength( computeHourHandLength() );
            getMinuteHand().setLength( computeMinuteHandLength() );
            getSecondHand().setLength( computeSecondHandLength() );
        }//end if
        
        return dialImage;
    }//end of getDialImage
    
    private void drawBorder( Graphics g, int x, int y, int length )
    {
        Color oldColor = g.getColor();
        
        g.setColor( getBorderColor() );
        g.fillOval( x, y, length, length );
        
        int borderWidth = getBorderWidth();
        x += borderWidth;
        y += borderWidth;
        length -= 2*borderWidth;
        g.setColor( getDialColor() );
        g.fillOval( x, y, length, length );
        
        g.setColor( oldColor );
    }//end of drawBorder

    private void drawTicks( Graphics g, int x, int y, int length )
    {
        switch( getTickStyle() )
        {
            case TICK_STYLE_NONE:
                break;
            case TICK_STYLE_ALL:
                drawTicks( g, x, y, length, 60 );
                break;
            case TICK_STYLE_MAJOR:
                drawTicks( g, x, y, length, 12 );
                break;
            case TICK_STYLE_MINIMAL:
                drawTicks( g, x, y, length, 4 );
                break;
            default:
        }//end switch
        
        return;
    }//end of drawTicks
    
    private void drawTicks( Graphics g, int x, int y, int length, int number )
    {
        int originX = x + length / 2;
        int originY = y + length / 2;
        
        g.translate( originX, originY );
        Color oldColor = g.getColor();
        
        int tickWidth = getTickWidth();
        
        int radius = length / 2;
        
        g.setColor( getTickColor() );
        
        int xi = 0;
        int yi = 0;
        int increment = 60 / number;
        for( int i = 0; i < number; i++ )
        {
            double cosTheta = getCosTheta( i*increment );
            double sinTheta = getSinTheta( i*increment );
            
            xi = ( int )( radius * cosTheta );
            yi = ( int )( radius * sinTheta );
            
            xi -= ( tickWidth / 2 );
            yi -= ( tickWidth / 2 );
            
            g.fillOval( xi, yi, tickWidth, tickWidth );
        }//end for
        
        g.setColor( oldColor );
        g.translate( -originX, -originY );
    }//end of drawAllTicks


    
    private void drawNumbers( Graphics g, int x, int y, int length )
    {
        int originX = x + length / 2;
        int originY = y + length / 2;
        
        g.translate( originX, originY );
        Color oldColor = g.getColor();
        Font oldFont   = g.getFont();
        
      
        int radius = length / 2;
        
        g.setColor( getNumeralColor() );
        g.setFont( getNumeralFont() );
        
        int xi = 0;
        int yi = 0;
        FontMetrics fm = g.getFontMetrics();
                
        for( int i = 0; i < 12; i++ )
        {
            double cosTheta = getCosTheta( 5*i );
            double sinTheta = getSinTheta( 5*i );
            
            xi = ( int )( radius * cosTheta );
            yi = ( int )( radius * sinTheta );
            
            int width = fm.stringWidth( getNumberString( i ) );
            int height = fm.getAscent() + fm.getDescent();
            
            xi -= ( width / 2 );
            yi += ( height / 2 );

            g.drawString( getNumberString( i ), xi, yi );
        }//end for
        
        g.setColor( oldColor );
        g.setFont( oldFont );
        g.translate( -originX, -originY );
    }//end of drawNumbers
    
    
    private ClockHand getHourHand()
    {
        if( hourHand == null )
        {
            hourHand = new ClockHand();
            hourHand.setWidth( DEFAULT_HOUR_HAND_THICKNESS );
        }
        return hourHand;
    }//end of getHourHand
    
    private ClockHand getMinuteHand()
    {
        if( minuteHand == null )
        {
            minuteHand = new ClockHand();
            minuteHand.setWidth( DEFAULT_MINUTE_HAND_THICKNESS );
        }
        return minuteHand;
    }//end of getMinuteHand
    
    private ClockHand getSecondHand()
    {
        if( secondHand == null )
        {
            secondHand = new ClockHand();
            secondHand.setWidth( DEFAULT_SECOND_HAND_THICKNESS );
        }
        return secondHand;
    }//end of getSecondHand
    
    private PropertyChangeSupport getPropertyChangeNotifier()
    {
        if( changesNotifier == null )
            changesNotifier = new PropertyChangeSupport( this );
        return changesNotifier;
    }//getPropertyChangeNotifier

/**
 * Add a property change listener.
 */
    public void addPropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().addPropertyChangeListener( pcl);
    }   


/**
 * Remove a property change listener.
 */    
    public void removePropertyChangeListener( PropertyChangeListener pcl) 
    {
        getPropertyChangeNotifier().removePropertyChangeListener( pcl);
    }
        

    private static final boolean DEBUG = false ;
    private void debug( String s )
    {
        if( DEBUG )
            System.out.println( getClass().getName() + "::" + s );
    }
}//end of AnalogDisplay class definition.
