/* @(#) ClockHand.java        Ver 1.0    01,April,1998
 *
 * Copyright (c) 1998 International Business Machines.
 * All Rights Reserved.
 *
 * Author : Sunanda Bera & P.Sushma
 * Last Modified : 01,April,1998
 *
 * Purpose : Defines class ClockHand.
 *
 *
 * Revision History 
 * ======== ======= 
 *
 * Date        By            Description
 * ----        --            -----------
 *
 *
 */
package com.ibm.clock;

import java.awt.Color;
import java.awt.Graphics;

class ClockHand
{
    public static final int DEFAULT_LENGTH  = 30;
    public static final int MINIMUM_WIDTH   = 1;
   
    
    private boolean visible = true;
    private int length      = DEFAULT_LENGTH;
    private Color color     = null;
    private int width       = MINIMUM_WIDTH;

/**
 * The current width of the clock hand.
 * @return The current width in pixels.
 */    
    public int getWidth()
    {
        return width;
    }//end of getWidth

/**
 * Set the width of the clock hand. This must be greater than 1.
 * @param width The new width in pixels.
 * @exception IllegalArgumentException if the specified width is less than 1.
 */    
    public void setWidth( int width )
    throws IllegalArgumentException
    {
        if( !isProperWidth( width ) )
            throw new IllegalArgumentException( " width =" + width );
        this.width = width;
    }//end of setWidth

/**
 * Is the width supported ?
 */
    protected boolean isProperWidth( int width )
    {
        if( width < 1 )
            return false;
        return true;
    }//end of width

/**
 * Get the color of the clock hand.
 * @return The color of the clock hand.
 */    
    public Color getColor()
    {
        if( color == null )
            color = Color.black;
        return color;
    }//end of getColor

/**
 * Set the color of the clock hand.
 * @param c The new color.
 * @exception NullPointerException If a null value is passed.
 */    
    public void setColor( Color c )
    {
        if( c == null )
            throw new NullPointerException();
        this.color = c;
    }//end of setColor

/**
 * Get the length of the clock hand.
 * @return The current length.
 */    
    public int getLength()
    {
        return length;
    }//end of getLength

/**
 * Set the length of the clock hand.
 * @param length The new length in pixels. Must be greater than zero.
 */    
    public void setLength( int length )
    throws IllegalArgumentException
    {
        if( length < 0 )
            throw new IllegalArgumentException( " bad length = " + length );
        this.length = length;
    }//end of setLength

/**
 * Is the hand visible ? An invisible hand will not render itself. 
 */
    public boolean isVisible()
    {
        return visible;
    }//end of isVisible

/**
 * Set the visibility parameter.
 * @param visible true  => The hand must render itself
 *                false => The hand must not render itself.
 */    
    public void setVisible( boolean visible )
    {
        this.visible = visible;
    }//end of setVisible
    
    protected boolean isProperAngle( double angle )
    {
        if( angle < 0.0 || angle > ( 2 * Math.PI ) )
            return false;
        return true;
    }//end of isProperAngle
    

    
    
    public void paint( Graphics g, int x, int y, double angle )
    {
        if( !isVisible() )
            return;
        Color oldColor = g.getColor();
        
        g.setColor( getColor() );
        g.translate( x, y );
        drawLine( g, angle, getWidth(), getLength() );
        drawTop( g, angle, getWidth(), getLength(), getWidth() / 2 );
        drawCentre( g, getWidth() );
        
        g.translate( -x, -y );
        g.setColor( oldColor );
    }//end of paint
    
    private void drawLine( Graphics g, double angle, int width, int length )
    {
        int [] xArr = new int[4];
        int [] yArr = new int[4];
        
        double sinTheta = Math.sin( angle );
        double cosTheta = Math.cos( angle );
        

        int x = ( int )Math.round( length * cosTheta );
        int y = ( int )Math.round( length * sinTheta );
        
        if( width != 1 )
        {
            int deltaX = ( int )Math.round( ( width * sinTheta ) / 2 );
            int deltaY = ( int )Math.round( ( width * cosTheta ) / 2 );
            
            xArr[0] = deltaX;
            yArr[0] = -deltaY;
            
            xArr[1] = -deltaX;
            yArr[1] = +deltaY;
            
            xArr[2] = x - deltaX;
            yArr[2] = y + deltaY;
            
            xArr[3] = x + deltaX;
            yArr[3] = y - deltaY;
            
            g.fillPolygon( xArr, yArr, 4 );
        }
        else
        {
            g.drawLine( 0, 0, x, y );
        }
        

    }//end drawLine
    
    private void drawTop( Graphics g, double angle, int width, int length, int topLength )
    {
        if( width <= 2 )
            return;// no need to draw a top
        int [] xArr = new int[3];
        int [] yArr = new int[3];
        
        double sinTheta = Math.sin( angle );
        double cosTheta = Math.cos( angle );
        

        int x = ( int )Math.round( length * cosTheta );
        int y = ( int )Math.round( length * sinTheta );
        
        int deltaX = ( int )Math.round( ( width * sinTheta ) / 2 );
        int deltaY = ( int )Math.round( ( width * cosTheta ) / 2 );
        
        xArr[0] = x + deltaX;
        yArr[0] = y - deltaY;
        
        xArr[1] = x - deltaX;
        yArr[1] = y + deltaY;
        
        xArr[2] = ( int )Math.round( ( length + topLength ) * cosTheta );
        yArr[2] = ( int )Math.round( ( length + topLength ) * sinTheta );
        
        g.fillPolygon( xArr, yArr, 3 );
        
    }//end of drawTop
    
    private void drawCentre( Graphics g, int width )
    {
        int x0 =  -width;
        int y0 =  -width;
        
        g.fillOval( x0, y0, width * 2, width * 2 );
    }
    
    
}//end of Hand
