/**
 * Copyright (c) 2004-2015, Istvan David, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Istvan David - initial API and implementation
 */
package org.eclipse.viatra.cep.core.engine.compiler.builders;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.cep.core.engine.compiler.PermutationsHelper;
import org.eclipse.viatra.cep.core.engine.compiler.builders.BuilderPrimitives;
import org.eclipse.viatra.cep.core.metamodels.automaton.Automaton;
import org.eclipse.viatra.cep.core.metamodels.automaton.AutomatonFactory;
import org.eclipse.viatra.cep.core.metamodels.automaton.Guard;
import org.eclipse.viatra.cep.core.metamodels.automaton.NegativeTransition;
import org.eclipse.viatra.cep.core.metamodels.automaton.Parameter;
import org.eclipse.viatra.cep.core.metamodels.automaton.State;
import org.eclipse.viatra.cep.core.metamodels.automaton.Transition;
import org.eclipse.viatra.cep.core.metamodels.automaton.TypedTransition;
import org.eclipse.viatra.cep.core.metamodels.events.AND;
import org.eclipse.viatra.cep.core.metamodels.events.ComplexEventOperator;
import org.eclipse.viatra.cep.core.metamodels.events.ComplexEventPattern;
import org.eclipse.viatra.cep.core.metamodels.events.EventPattern;
import org.eclipse.viatra.cep.core.metamodels.events.EventPatternReference;
import org.eclipse.viatra.cep.core.metamodels.events.EventsFactory;
import org.eclipse.viatra.cep.core.metamodels.events.FOLLOWS;
import org.eclipse.viatra.cep.core.metamodels.events.NEG;
import org.eclipse.viatra.cep.core.metamodels.events.OR;
import org.eclipse.viatra.cep.core.metamodels.trace.TraceFactory;
import org.eclipse.viatra.cep.core.metamodels.trace.TraceModel;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class ComplexMappingUtils {
  @Extension
  protected final AutomatonFactory automatonFactory = AutomatonFactory.eINSTANCE;
  
  @Extension
  protected final TraceFactory traceFactory = TraceFactory.eINSTANCE;
  
  @Extension
  private BuilderPrimitives builderPrimitives;
  
  private TraceModel traceModel;
  
  public ComplexMappingUtils(final TraceModel traceModel) {
    this.traceModel = traceModel;
    BuilderPrimitives _builderPrimitives = new BuilderPrimitives(traceModel);
    this.builderPrimitives = _builderPrimitives;
  }
  
  /**
   * Builds a path of {@link Transition}s and {@link State}s for the referred {@link EventPattern} with a
   * {@link FOLLOWS} operator.
   */
  public void buildFollowsPath(final Automaton automaton, final ComplexEventPattern eventPattern, final State preState, final State postState) {
    EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
    this.buildFollowsPath(automaton, _containedEventPatterns, preState, postState);
    this.builderPrimitives.initializeTimewindow(automaton, eventPattern, preState, postState);
  }
  
  /**
   * Builds a path of {@link Transition}s and {@link State}s for the referred {@link EventPatternReference}s combined
   * under an {@link OR} operator.
   */
  private State buildFollowsPath(final Automaton automaton, final List<EventPatternReference> eventPatternReferences, final State preState, final State postState) {
    State _xblockexpression = null;
    {
      State firstCreatedState = null;
      State nextState = null;
      for (final EventPatternReference eventPatternReference : eventPatternReferences) {
        boolean _equals = Objects.equal(nextState, null);
        if (_equals) {
          State _mapWithMultiplicity = this.builderPrimitives.mapWithMultiplicity(eventPatternReference, automaton, preState);
          nextState = _mapWithMultiplicity;
          firstCreatedState = nextState;
        } else {
          State _mapWithMultiplicity_1 = this.builderPrimitives.mapWithMultiplicity(eventPatternReference, automaton, nextState);
          nextState = _mapWithMultiplicity_1;
        }
      }
      this.builderPrimitives.createEpsilon(nextState, postState);
      _xblockexpression = firstCreatedState;
    }
    return _xblockexpression;
  }
  
  public boolean unfoldFollowsPath(final Automaton automaton, final ComplexEventPattern eventPattern, final TypedTransition transition) {
    boolean _xblockexpression = false;
    {
      EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
      State _preState = transition.getPreState();
      State _postState = transition.getPostState();
      final State firstCreatedState = this.buildFollowsPath(automaton, _containedEventPatterns, _preState, _postState);
      _xblockexpression = this.builderPrimitives.alignTimewindow(automaton, eventPattern, transition, firstCreatedState);
    }
    return _xblockexpression;
  }
  
  /**
   * Builds a path of {@link Transition}s and {@link State}s for the referred {@link EventPattern} with an {@link OR}
   * operator.
   */
  public void buildOrPath(final Automaton automaton, final ComplexEventPattern eventPattern, final State preState, final State postState) {
    final State lastState = this.automatonFactory.createState();
    EList<State> _states = automaton.getStates();
    _states.add(lastState);
    EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
    for (final EventPatternReference eventPatternReference : _containedEventPatterns) {
      this.builderPrimitives.mapWithMultiplicity(eventPatternReference, automaton, preState, lastState);
    }
    this.builderPrimitives.createEpsilon(lastState, postState);
  }
  
  public void unfoldOrPath(final Automaton automaton, final ComplexEventPattern eventPattern, final TypedTransition transition) {
    State _preState = transition.getPreState();
    State _postState = transition.getPostState();
    this.buildOrPath(automaton, eventPattern, _preState, _postState);
    this.builderPrimitives.alignTimewindow(automaton, eventPattern, transition);
  }
  
  /**
   * Builds a path of {@link Transition}s and {@link State}s for the referred {@link EventPattern} with an
   * {@link AND} operator.
   */
  public void buildAndPath(final Automaton automaton, final ComplexEventPattern eventPattern, final State preState, final State postState) {
    PermutationsHelper<EventPatternReference> _permutationsHelper = new PermutationsHelper<EventPatternReference>();
    EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
    List<List<EventPatternReference>> _all = _permutationsHelper.getAll(_containedEventPatterns);
    for (final List<EventPatternReference> permutation : _all) {
      this.buildFollowsPath(automaton, permutation, preState, postState);
    }
  }
  
  public void unfoldAndPath(final Automaton automaton, final ComplexEventPattern eventPattern, final TypedTransition transition) {
    State _preState = transition.getPreState();
    State _postState = transition.getPostState();
    this.buildAndPath(automaton, eventPattern, _preState, _postState);
    this.builderPrimitives.alignTimewindow(automaton, eventPattern, transition);
  }
  
  /**
   * Builds a path of {@link Transition}s and {@link State}s for the referred {@link EventPattern} with a {@link NOT}
   * operator.
   */
  public void buildNotPath(final Automaton automaton, final EventPattern eventPattern, final State preState, final State postState) {
    NegativeTransition transition = this.automatonFactory.createNegativeTransition();
    Guard guard = this.automatonFactory.createGuard();
    guard.setEventType(eventPattern);
    EList<Guard> _guards = transition.getGuards();
    _guards.add(guard);
    transition.setPreState(preState);
    transition.setPostState(postState);
  }
  
  /**
   * Unfolds a {@link Transition} guarded by a {@link ComplexEventPattern} with a {@link NEG} operator.
   */
  public Boolean unfoldNotPath(final Automaton automaton, final ComplexEventPattern eventPattern, final NegativeTransition transition) {
    boolean _switchResult = false;
    ComplexEventOperator _operator = eventPattern.getOperator();
    boolean _matched = false;
    if (!_matched) {
      if (_operator instanceof NEG) {
        _matched=true;
        boolean _xblockexpression = false;
        {
          State _preState = transition.getPreState();
          State _postState = transition.getPostState();
          final TypedTransition newTransition = this.builderPrimitives.newTransition(_preState, _postState);
          EList<Guard> _guards = newTransition.getGuards();
          EList<Guard> _guards_1 = transition.getGuards();
          Iterables.<Guard>addAll(_guards, _guards_1);
          EList<Parameter> _parameters = newTransition.getParameters();
          EList<Parameter> _parameters_1 = transition.getParameters();
          _xblockexpression = Iterables.<Parameter>addAll(_parameters, _parameters_1);
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (_operator instanceof FOLLOWS) {
        _matched=true;
        State _preState = transition.getPreState();
        State _postState = transition.getPostState();
        final NegativeTransition firstNegBranch = this.builderPrimitives.newNegTransition(_preState, _postState);
        EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
        EventPatternReference _head = IterableExtensions.<EventPatternReference>head(_containedEventPatterns);
        this.builderPrimitives.addGuard(firstNegBranch, _head);
        EList<EventPatternReference> _containedEventPatterns_1 = eventPattern.getContainedEventPatterns();
        EventPatternReference _head_1 = IterableExtensions.<EventPatternReference>head(_containedEventPatterns_1);
        this.builderPrimitives.handleTransitionParameters(_head_1, firstNegBranch);
        EList<EventPatternReference> _containedEventPatterns_2 = eventPattern.getContainedEventPatterns();
        int _size = _containedEventPatterns_2.size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          EList<EventPatternReference> _containedEventPatterns_3 = eventPattern.getContainedEventPatterns();
          int _size_1 = _containedEventPatterns_3.size();
          boolean _equals = (_size_1 == 2);
          Preconditions.checkArgument(_equals);
          EList<EventPatternReference> _containedEventPatterns_4 = eventPattern.getContainedEventPatterns();
          EventPatternReference _head_2 = IterableExtensions.<EventPatternReference>head(_containedEventPatterns_4);
          State _preState_1 = transition.getPreState();
          final State secondNegBranchState = this.builderPrimitives.transitionToNewState(automaton, _head_2, _preState_1);
          State _postState_1 = transition.getPostState();
          final NegativeTransition secondNegBranch = this.builderPrimitives.newNegTransition(secondNegBranchState, _postState_1);
          EList<EventPatternReference> _containedEventPatterns_5 = eventPattern.getContainedEventPatterns();
          EventPatternReference _get = _containedEventPatterns_5.get(1);
          this.builderPrimitives.addGuard(secondNegBranch, _get);
          EList<EventPatternReference> _containedEventPatterns_6 = eventPattern.getContainedEventPatterns();
          EventPatternReference _last = IterableExtensions.<EventPatternReference>last(_containedEventPatterns_6);
          this.builderPrimitives.handleTransitionParameters(_last, secondNegBranch);
        }
      }
    }
    if (!_matched) {
      if (_operator instanceof OR) {
        _matched=true;
        State _preState = transition.getPreState();
        State _postState = transition.getPostState();
        final NegativeTransition newTransition = this.builderPrimitives.newNegTransition(_preState, _postState);
        EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
        final Procedure1<EventPatternReference> _function = new Procedure1<EventPatternReference>() {
          @Override
          public void apply(final EventPatternReference ref) {
            EventPattern _eventPattern = ref.getEventPattern();
            ComplexMappingUtils.this.builderPrimitives.addGuard(newTransition, _eventPattern);
          }
        };
        IterableExtensions.<EventPatternReference>forEach(_containedEventPatterns, _function);
      }
    }
    if (!_matched) {
      if (_operator instanceof AND) {
        _matched=true;
        PermutationsHelper<EventPatternReference> _permutationsHelper = new PermutationsHelper<EventPatternReference>();
        EList<EventPatternReference> _containedEventPatterns = eventPattern.getContainedEventPatterns();
        List<List<EventPatternReference>> _all = _permutationsHelper.getAll(_containedEventPatterns);
        for (final List<EventPatternReference> permutation : _all) {
          {
            final ComplexEventPattern surrogateFollowsPattern = EventsFactory.eINSTANCE.createComplexEventPattern();
            EList<EventPatternReference> _containedEventPatterns_1 = surrogateFollowsPattern.getContainedEventPatterns();
            Iterables.<EventPatternReference>addAll(_containedEventPatterns_1, permutation);
            FOLLOWS _createFOLLOWS = EventsFactory.eINSTANCE.createFOLLOWS();
            surrogateFollowsPattern.setOperator(_createFOLLOWS);
            State _preState = transition.getPreState();
            State _postState = transition.getPostState();
            this.builderPrimitives.transitionBetween(surrogateFollowsPattern, _preState, _postState);
          }
        }
        throw new UnsupportedOperationException();
      }
    }
    if (!_matched) {
      throw new IllegalArgumentException();
    }
    return Boolean.valueOf(_switchResult);
  }
}
