/**
 * Copyright (c) 2004-2014, Istvan David, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Istvan David - initial API and implementation
 */
package org.eclipse.viatra.cep.core.engine.runtime;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import java.util.Map;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.emf.EMFScope;
import org.eclipse.incquery.runtime.evm.api.RuleSpecification;
import org.eclipse.incquery.runtime.evm.specific.ConflictResolvers;
import org.eclipse.incquery.runtime.evm.specific.resolver.FixedPriorityConflictResolver;
import org.eclipse.viatra.cep.core.api.events.ParameterizableEventInstance;
import org.eclipse.viatra.cep.core.api.patterns.ObservedComplexEventPattern;
import org.eclipse.viatra.cep.core.engine.IEventModelManager;
import org.eclipse.viatra.cep.core.engine.runtime.EnabledNegativeTransitionMatch;
import org.eclipse.viatra.cep.core.engine.runtime.EnabledNegativeTransitionMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.EnabledTransitionMatch;
import org.eclipse.viatra.cep.core.engine.runtime.EnabledTransitionMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.FinishedAutomatonMatch;
import org.eclipse.viatra.cep.core.engine.runtime.FinishedAutomatonMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.RuntimePatterns;
import org.eclipse.viatra.cep.core.engine.runtime.TokenEntersTimedZoneMatch;
import org.eclipse.viatra.cep.core.engine.runtime.TokenEntersTimedZoneMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.TokenInTrapStateMatch;
import org.eclipse.viatra.cep.core.engine.runtime.TokenInTrapStateMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.TokenLeavesTimedZoneMatch;
import org.eclipse.viatra.cep.core.engine.runtime.TokenLeavesTimedZoneMatcher;
import org.eclipse.viatra.cep.core.engine.runtime.util.EnabledNegativeTransitionQuerySpecification;
import org.eclipse.viatra.cep.core.engine.runtime.util.EnabledTransitionQuerySpecification;
import org.eclipse.viatra.cep.core.engine.runtime.util.FinishedAutomatonQuerySpecification;
import org.eclipse.viatra.cep.core.engine.runtime.util.TokenEntersTimedZoneQuerySpecification;
import org.eclipse.viatra.cep.core.engine.runtime.util.TokenInTrapStateQuerySpecification;
import org.eclipse.viatra.cep.core.engine.runtime.util.TokenLeavesTimedZoneQuerySpecification;
import org.eclipse.viatra.cep.core.engine.timing.TimingTable;
import org.eclipse.viatra.cep.core.evm.CepRealm;
import org.eclipse.viatra.cep.core.logging.LoggerUtils;
import org.eclipse.viatra.cep.core.metamodels.automaton.Automaton;
import org.eclipse.viatra.cep.core.metamodels.automaton.AutomatonFactory;
import org.eclipse.viatra.cep.core.metamodels.automaton.EventToken;
import org.eclipse.viatra.cep.core.metamodels.automaton.FinalState;
import org.eclipse.viatra.cep.core.metamodels.automaton.NegativeTransition;
import org.eclipse.viatra.cep.core.metamodels.automaton.Parameter;
import org.eclipse.viatra.cep.core.metamodels.automaton.ParameterBinding;
import org.eclipse.viatra.cep.core.metamodels.automaton.ParameterTable;
import org.eclipse.viatra.cep.core.metamodels.automaton.TimedZone;
import org.eclipse.viatra.cep.core.metamodels.automaton.TrapState;
import org.eclipse.viatra.cep.core.metamodels.automaton.TypedTransition;
import org.eclipse.viatra.cep.core.metamodels.events.Event;
import org.eclipse.viatra.emf.runtime.rules.EventDrivenTransformationRuleGroup;
import org.eclipse.viatra.emf.runtime.rules.eventdriven.EventDrivenTransformationRule;
import org.eclipse.viatra.emf.runtime.rules.eventdriven.EventDrivenTransformationRuleFactory;
import org.eclipse.viatra.emf.runtime.transformation.eventdriven.EventDrivenTransformation;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

@SuppressWarnings("all")
public class RuntimeRules {
  @Extension
  private final EventDrivenTransformationRuleFactory ruleFactory = new EventDrivenTransformationRuleFactory();
  
  @Extension
  private final RuntimePatterns evaluationPatterns = new Function0<RuntimePatterns>() {
    public RuntimePatterns apply() {
      try {
        RuntimePatterns _instance = RuntimePatterns.instance();
        return _instance;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  @Extension
  private final Logger logger = LoggerUtils.getInstance().getLogger();
  
  @Accessors
  private IEventModelManager eventModelManager;
  
  @Accessors
  private Map<RuleSpecification<?>, Integer> modelHandlers;
  
  public RuntimeRules(final IEventModelManager eventModelManager) {
    this.eventModelManager = eventModelManager;
  }
  
  public EventDrivenTransformationRuleGroup getRules() {
    return new EventDrivenTransformationRuleGroup(
      this.enabledTransitionRule, 
      this.enabledNegativeTransitionRule, 
      this.finishedAutomatonRule, 
      this.tokenInTrapStateRule, 
      this.tokenEntersTimedZoneRule, 
      this.tokenLeavesTimedZoneRule);
  }
  
  public EventDrivenTransformation registerRulesWithCustomPriorities() {
    try {
      EventDrivenTransformation _xblockexpression = null;
      {
        final FixedPriorityConflictResolver fixedPriorityResolver = ConflictResolvers.createFixedPriorityResolver();
        RuleSpecification<EnabledNegativeTransitionMatch> _ruleSpecification = this.enabledNegativeTransitionRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification, 150);
        RuleSpecification<EnabledTransitionMatch> _ruleSpecification_1 = this.enabledTransitionRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification_1, 100);
        RuleSpecification<FinishedAutomatonMatch> _ruleSpecification_2 = this.finishedAutomatonRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification_2, 50);
        RuleSpecification<TokenInTrapStateMatch> _ruleSpecification_3 = this.tokenInTrapStateRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification_3, 10);
        RuleSpecification<TokenEntersTimedZoneMatch> _ruleSpecification_4 = this.tokenEntersTimedZoneRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification_4, 5);
        RuleSpecification<TokenLeavesTimedZoneMatch> _ruleSpecification_5 = this.tokenLeavesTimedZoneRule.getRuleSpecification();
        fixedPriorityResolver.setPriority(_ruleSpecification_5, 1);
        ResourceSet _resourceSet = this.eventModelManager.getResourceSet();
        EMFScope _eMFScope = new EMFScope(_resourceSet);
        EventDrivenTransformation.EventDrivenTransformationBuilder _forScope = EventDrivenTransformation.forScope(_eMFScope);
        EventDrivenTransformationRuleGroup _rules = this.getRules();
        EventDrivenTransformation.EventDrivenTransformationBuilder _addRules = _forScope.addRules(_rules);
        EventDrivenTransformation.EventDrivenTransformationBuilder _setConflictResolver = _addRules.setConflictResolver(fixedPriorityResolver);
        _xblockexpression = _setConflictResolver.build();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private final EventDrivenTransformationRule<EnabledTransitionMatch, EnabledTransitionMatcher> enabledTransitionRule = new Function0<EventDrivenTransformationRule<EnabledTransitionMatch, EnabledTransitionMatcher>>() {
    public EventDrivenTransformationRule<EnabledTransitionMatch, EnabledTransitionMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledTransitionMatch, EnabledTransitionMatcher> _createRule = RuntimeRules.this.ruleFactory.<EnabledTransitionMatch, EnabledTransitionMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledTransitionMatch, EnabledTransitionMatcher> _name = _createRule.name("enabled transition rule");
        EnabledTransitionQuerySpecification _enabledTransition = RuntimeRules.this.evaluationPatterns.getEnabledTransition();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledTransitionMatch, EnabledTransitionMatcher> _precondition = _name.precondition(_enabledTransition);
        final IMatchProcessor<EnabledTransitionMatch> _function = new IMatchProcessor<EnabledTransitionMatch>() {
          @Override
          public void process(final EnabledTransitionMatch it) {
            Automaton _automaton = it.getAutomaton();
            String _eventPatternId = _automaton.getEventPatternId();
            Automaton _automaton_1 = it.getAutomaton();
            String _format = String.format("Enabled transition in %s (automaton: %s)", _eventPatternId, _automaton_1);
            RuntimeRules.this.logger.debug(_format);
            TypedTransition _transition = it.getTransition();
            EventToken _eventToken = it.getEventToken();
            RuntimeRules.this.eventModelManager.handleEvent(_transition, _eventToken);
            Event _event = it.getEvent();
            if ((_event instanceof ParameterizableEventInstance)) {
              TypedTransition _transition_1 = it.getTransition();
              EList<Parameter> _parameters = _transition_1.getParameters();
              for (final Parameter parameter : _parameters) {
                {
                  Event _event_1 = it.getEvent();
                  int _position = parameter.getPosition();
                  final Object parameterValueToBind = ((ParameterizableEventInstance) _event_1).getParameter(_position);
                  EventToken _eventToken_1 = it.getEventToken();
                  ParameterTable _parameterTable = _eventToken_1.getParameterTable();
                  EList<ParameterBinding> _parameterBindings = _parameterTable.getParameterBindings();
                  final Function1<ParameterBinding, Boolean> _function = new Function1<ParameterBinding, Boolean>() {
                    @Override
                    public Boolean apply(final ParameterBinding binding) {
                      String _symbolicName = binding.getSymbolicName();
                      String _symbolicName_1 = parameter.getSymbolicName();
                      return Boolean.valueOf(_symbolicName.equalsIgnoreCase(_symbolicName_1));
                    }
                  };
                  final ParameterBinding existingBinding = IterableExtensions.<ParameterBinding>findFirst(_parameterBindings, _function);
                  boolean _equals = Objects.equal(existingBinding, null);
                  if (_equals) {
                    final ParameterBinding newBinding = AutomatonFactory.eINSTANCE.createParameterBinding();
                    String _symbolicName = parameter.getSymbolicName();
                    newBinding.setSymbolicName(_symbolicName);
                    newBinding.setValue(parameterValueToBind);
                    EventToken _eventToken_2 = it.getEventToken();
                    ParameterTable _parameterTable_1 = _eventToken_2.getParameterTable();
                    EList<ParameterBinding> _parameterBindings_1 = _parameterTable_1.getParameterBindings();
                    _parameterBindings_1.add(newBinding);
                  } else {
                    Object _value = existingBinding.getValue();
                    boolean _equals_1 = _value.equals(parameterValueToBind);
                    boolean _not = (!_equals_1);
                    if (_not) {
                      return;
                    }
                  }
                }
              }
            }
            TypedTransition _transition_2 = it.getTransition();
            EventToken _eventToken_1 = it.getEventToken();
            RuntimeRules.this.eventModelManager.fireTransition(_transition_2, _eventToken_1);
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledTransitionMatch, EnabledTransitionMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<EnabledTransitionMatch, EnabledTransitionMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  private final EventDrivenTransformationRule<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> enabledNegativeTransitionRule = new Function0<EventDrivenTransformationRule<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher>>() {
    public EventDrivenTransformationRule<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> _createRule = RuntimeRules.this.ruleFactory.<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> _name = _createRule.name("enabled negative transition rule");
        EnabledNegativeTransitionQuerySpecification _enabledNegativeTransition = RuntimeRules.this.evaluationPatterns.getEnabledNegativeTransition();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> _precondition = _name.precondition(_enabledNegativeTransition);
        final IMatchProcessor<EnabledNegativeTransitionMatch> _function = new IMatchProcessor<EnabledNegativeTransitionMatch>() {
          @Override
          public void process(final EnabledNegativeTransitionMatch it) {
            Automaton _automaton = it.getAutomaton();
            String _eventPatternId = _automaton.getEventPatternId();
            Automaton _automaton_1 = it.getAutomaton();
            String _format = String.format("Enabled negative transition in %s (automaton: %s)", _eventPatternId, _automaton_1);
            RuntimeRules.this.logger.debug(_format);
            NegativeTransition _transition = it.getTransition();
            EventToken _eventToken = it.getEventToken();
            RuntimeRules.this.eventModelManager.handleEvent(_transition, _eventToken);
            Event _event = it.getEvent();
            if ((_event instanceof ParameterizableEventInstance)) {
              NegativeTransition _transition_1 = it.getTransition();
              EList<Parameter> _parameters = _transition_1.getParameters();
              for (final Parameter parameter : _parameters) {
                {
                  Event _event_1 = it.getEvent();
                  int _position = parameter.getPosition();
                  final Object parameterValueToBind = ((ParameterizableEventInstance) _event_1).getParameter(_position);
                  EventToken _eventToken_1 = it.getEventToken();
                  ParameterTable _parameterTable = _eventToken_1.getParameterTable();
                  EList<ParameterBinding> _parameterBindings = _parameterTable.getParameterBindings();
                  final Function1<ParameterBinding, Boolean> _function = new Function1<ParameterBinding, Boolean>() {
                    @Override
                    public Boolean apply(final ParameterBinding binding) {
                      String _symbolicName = binding.getSymbolicName();
                      String _symbolicName_1 = parameter.getSymbolicName();
                      return Boolean.valueOf(_symbolicName.equalsIgnoreCase(_symbolicName_1));
                    }
                  };
                  final ParameterBinding existingBinding = IterableExtensions.<ParameterBinding>findFirst(_parameterBindings, _function);
                  boolean _equals = Objects.equal(existingBinding, null);
                  if (_equals) {
                    final ParameterBinding newBinding = AutomatonFactory.eINSTANCE.createParameterBinding();
                    String _symbolicName = parameter.getSymbolicName();
                    newBinding.setSymbolicName(_symbolicName);
                    newBinding.setValue(parameterValueToBind);
                    EventToken _eventToken_2 = it.getEventToken();
                    ParameterTable _parameterTable_1 = _eventToken_2.getParameterTable();
                    EList<ParameterBinding> _parameterBindings_1 = _parameterTable_1.getParameterBindings();
                    _parameterBindings_1.add(newBinding);
                  } else {
                    Object _value = existingBinding.getValue();
                    boolean _equals_1 = _value.equals(parameterValueToBind);
                    boolean _not = (!_equals_1);
                    if (_not) {
                      return;
                    }
                  }
                }
              }
            }
            NegativeTransition _transition_2 = it.getTransition();
            EventToken _eventToken_1 = it.getEventToken();
            RuntimeRules.this.eventModelManager.fireTransition(_transition_2, _eventToken_1);
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<EnabledNegativeTransitionMatch, EnabledNegativeTransitionMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  private final EventDrivenTransformationRule<FinishedAutomatonMatch, FinishedAutomatonMatcher> finishedAutomatonRule = new Function0<EventDrivenTransformationRule<FinishedAutomatonMatch, FinishedAutomatonMatcher>>() {
    public EventDrivenTransformationRule<FinishedAutomatonMatch, FinishedAutomatonMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<FinishedAutomatonMatch, FinishedAutomatonMatcher> _createRule = RuntimeRules.this.ruleFactory.<FinishedAutomatonMatch, FinishedAutomatonMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<FinishedAutomatonMatch, FinishedAutomatonMatcher> _name = _createRule.name("finished automaton rule");
        FinishedAutomatonQuerySpecification _finishedAutomaton = RuntimeRules.this.evaluationPatterns.getFinishedAutomaton();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<FinishedAutomatonMatch, FinishedAutomatonMatcher> _precondition = _name.precondition(_finishedAutomaton);
        final IMatchProcessor<FinishedAutomatonMatch> _function = new IMatchProcessor<FinishedAutomatonMatch>() {
          @Override
          public void process(final FinishedAutomatonMatch it) {
            Automaton _automaton = it.getAutomaton();
            String _eventPatternId = _automaton.getEventPatternId();
            Automaton _automaton_1 = it.getAutomaton();
            String _format = String.format("Finished automaton of pattern %s (automaton: %s)", _eventPatternId, _automaton_1);
            RuntimeRules.this.logger.debug(_format);
            Automaton _automaton_2 = it.getAutomaton();
            EList<FinalState> _finalStates = _automaton_2.getFinalStates();
            int _size = _finalStates.size();
            boolean _equals = (_size == 1);
            Preconditions.checkArgument(_equals);
            Automaton _automaton_3 = it.getAutomaton();
            EList<EventToken> _eventTokens = _automaton_3.getEventTokens();
            EventToken _eventToken = it.getEventToken();
            _eventTokens.remove(_eventToken);
            Automaton _automaton_4 = it.getAutomaton();
            EventToken _eventToken_1 = it.getEventToken();
            ObservedComplexEventPattern observedPattern = new ObservedComplexEventPattern(_automaton_4, _eventToken_1);
            CepRealm _cepRealm = RuntimeRules.this.eventModelManager.getCepRealm();
            _cepRealm.forwardObservedEventPattern(observedPattern);
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<FinishedAutomatonMatch, FinishedAutomatonMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<FinishedAutomatonMatch, FinishedAutomatonMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  private final EventDrivenTransformationRule<TokenInTrapStateMatch, TokenInTrapStateMatcher> tokenInTrapStateRule = new Function0<EventDrivenTransformationRule<TokenInTrapStateMatch, TokenInTrapStateMatcher>>() {
    public EventDrivenTransformationRule<TokenInTrapStateMatch, TokenInTrapStateMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenInTrapStateMatch, TokenInTrapStateMatcher> _createRule = RuntimeRules.this.ruleFactory.<TokenInTrapStateMatch, TokenInTrapStateMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenInTrapStateMatch, TokenInTrapStateMatcher> _name = _createRule.name("trap state rule");
        TokenInTrapStateQuerySpecification _tokenInTrapState = RuntimeRules.this.evaluationPatterns.getTokenInTrapState();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenInTrapStateMatch, TokenInTrapStateMatcher> _precondition = _name.precondition(_tokenInTrapState);
        final IMatchProcessor<TokenInTrapStateMatch> _function = new IMatchProcessor<TokenInTrapStateMatch>() {
          @Override
          public void process(final TokenInTrapStateMatch it) {
            Automaton _automaton = it.getAutomaton();
            String _eventPatternId = _automaton.getEventPatternId();
            Automaton _automaton_1 = it.getAutomaton();
            String _format = String.format("Event token found in the trap state for pattern %s (automaton: %s).", _eventPatternId, _automaton_1);
            RuntimeRules.this.logger.debug(_format);
            Automaton _automaton_2 = it.getAutomaton();
            EList<EventToken> _eventTokens = _automaton_2.getEventTokens();
            EventToken _eventToken = it.getEventToken();
            _eventTokens.remove(_eventToken);
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenInTrapStateMatch, TokenInTrapStateMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<TokenInTrapStateMatch, TokenInTrapStateMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  private final EventDrivenTransformationRule<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> tokenEntersTimedZoneRule = new Function0<EventDrivenTransformationRule<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher>>() {
    public EventDrivenTransformationRule<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> _createRule = RuntimeRules.this.ruleFactory.<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> _name = _createRule.name("token enters timed zone rule");
        TokenEntersTimedZoneQuerySpecification _tokenEntersTimedZone = RuntimeRules.this.evaluationPatterns.getTokenEntersTimedZone();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> _precondition = _name.precondition(_tokenEntersTimedZone);
        final IMatchProcessor<TokenEntersTimedZoneMatch> _function = new IMatchProcessor<TokenEntersTimedZoneMatch>() {
          @Override
          public void process(final TokenEntersTimedZoneMatch it) {
            TimingTable _instance = TimingTable.getInstance();
            TimedZone _timedZone = it.getTimedZone();
            EventToken _eventToken = it.getEventToken();
            _instance.enterTimedZone(_timedZone, _eventToken);
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<TokenEntersTimedZoneMatch, TokenEntersTimedZoneMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  private final EventDrivenTransformationRule<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> tokenLeavesTimedZoneRule = new Function0<EventDrivenTransformationRule<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher>>() {
    public EventDrivenTransformationRule<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> apply() {
      try {
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> _createRule = RuntimeRules.this.ruleFactory.<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher>createRule();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> _name = _createRule.name("token leaves timed zone rule");
        TokenLeavesTimedZoneQuerySpecification _tokenLeavesTimedZone = RuntimeRules.this.evaluationPatterns.getTokenLeavesTimedZone();
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> _precondition = _name.precondition(_tokenLeavesTimedZone);
        final IMatchProcessor<TokenLeavesTimedZoneMatch> _function = new IMatchProcessor<TokenLeavesTimedZoneMatch>() {
          @Override
          public void process(final TokenLeavesTimedZoneMatch it) {
            TimingTable _instance = TimingTable.getInstance();
            TimedZone _timedZone = it.getTimedZone();
            EventToken _eventToken = it.getEventToken();
            final boolean canLeave = _instance.leaveTimedZone(_timedZone, _eventToken);
            if ((!canLeave)) {
              EventToken _eventToken_1 = it.getEventToken();
              TrapState _trapState = it.getTrapState();
              _eventToken_1.setCurrentState(_trapState);
            }
          }
        };
        EventDrivenTransformationRuleFactory.EventDrivenTransformationBuilder<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> _action = _precondition.action(_function);
        EventDrivenTransformationRule<TokenLeavesTimedZoneMatch, TokenLeavesTimedZoneMatcher> _build = _action.build();
        return _build;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }.apply();
  
  @Pure
  public IEventModelManager getEventModelManager() {
    return this.eventModelManager;
  }
  
  public void setEventModelManager(final IEventModelManager eventModelManager) {
    this.eventModelManager = eventModelManager;
  }
  
  @Pure
  public Map<RuleSpecification<?>, Integer> getModelHandlers() {
    return this.modelHandlers;
  }
  
  public void setModelHandlers(final Map<RuleSpecification<?>, Integer> modelHandlers) {
    this.modelHandlers = modelHandlers;
  }
}
