/**
 * Copyright (c) 2004-2014, Istvan David, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Istvan David - initial API and implementation
 */
package org.eclipse.viatra.cep.core.eventprocessingstrategy;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.cep.core.engine.IEventModelManager;
import org.eclipse.viatra.cep.core.eventprocessingstrategy.IEventProcessingStrategy;
import org.eclipse.viatra.cep.core.metamodels.automaton.Automaton;
import org.eclipse.viatra.cep.core.metamodels.automaton.AutomatonFactory;
import org.eclipse.viatra.cep.core.metamodels.automaton.EventToken;
import org.eclipse.viatra.cep.core.metamodels.automaton.FinalState;
import org.eclipse.viatra.cep.core.metamodels.automaton.InitState;
import org.eclipse.viatra.cep.core.metamodels.automaton.InternalModel;
import org.eclipse.viatra.cep.core.metamodels.automaton.State;
import org.eclipse.viatra.cep.core.metamodels.automaton.Transition;
import org.eclipse.viatra.cep.core.metamodels.events.Event;
import org.eclipse.viatra.cep.core.utils.AutomatonUtils;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Common ancestor of the of concrete strategies. Defines how the {@link EventToken}s should be fired on the
 * {@link Transition}s and how the new {@link EventToken}s in the {@link InitState}s should be produced in general.
 * The inheriting strategies may override this behavior.
 * 
 * @author Istvan David
 */
@SuppressWarnings("all")
public abstract class AbstractStrategy implements IEventProcessingStrategy {
  @Accessors
  private IEventModelManager eventModelManager;
  
  public AbstractStrategy(final IEventModelManager eventModelManager) {
    this.eventModelManager = eventModelManager;
  }
  
  @Override
  public void handleEvent(final Transition transition, final EventToken eventTokenToMove) {
    InternalModel _model = this.eventModelManager.getModel();
    Event _latestEvent = _model.getLatestEvent();
    this.addProcessedEvent(eventTokenToMove, _latestEvent);
  }
  
  @Override
  public void fireTransition(final Transition transition, final EventToken eventTokenToMove) {
    boolean _notEquals = (!Objects.equal(transition, null));
    Preconditions.checkArgument(_notEquals);
    boolean _notEquals_1 = (!Objects.equal(eventTokenToMove, null));
    Preconditions.checkArgument(_notEquals_1);
    State _currentState = eventTokenToMove.getCurrentState();
    EList<EventToken> _eventTokens = _currentState.getEventTokens();
    final Function1<EventToken, Boolean> _function = new Function1<EventToken, Boolean>() {
      @Override
      public Boolean apply(final EventToken et) {
        Event _lastProcessed = et.getLastProcessed();
        return Boolean.valueOf(Objects.equal(_lastProcessed, null));
      }
    };
    Iterable<EventToken> _filter = IterableExtensions.<EventToken>filter(_eventTokens, _function);
    final Procedure1<EventToken> _function_1 = new Procedure1<EventToken>() {
      @Override
      public void apply(final EventToken eventToken) {
        InternalModel _model = AbstractStrategy.this.eventModelManager.getModel();
        Event _latestEvent = _model.getLatestEvent();
        AbstractStrategy.this.addProcessedEvent(eventToken, _latestEvent);
      }
    };
    IterableExtensions.<EventToken>forEach(_filter, _function_1);
    final State preState = transition.getPreState();
    if ((preState instanceof FinalState)) {
      return;
    }
    final State nextState = transition.getPostState();
    eventTokenToMove.setCurrentState(nextState);
    this.eventModelManager.callbackOnFiredToken(transition, eventTokenToMove);
  }
  
  protected void addProcessedEvent(final EventToken eventToken, final Event event) {
    EList<Event> _recordedEvents = eventToken.getRecordedEvents();
    _recordedEvents.add(event);
    eventToken.setLastProcessed(event);
  }
  
  @Override
  public void handleInitTokenCreation(final InternalModel model, final AutomatonFactory factory) {
    EList<Automaton> _automata = model.getAutomata();
    final Procedure1<Automaton> _function = new Procedure1<Automaton>() {
      @Override
      public void apply(final Automaton automaton) {
        InitState _initialState = automaton.getInitialState();
        boolean _isEmpty = AutomatonUtils.isEmpty(_initialState);
        if (_isEmpty) {
          InitState _initialState_1 = automaton.getInitialState();
          AutomatonUtils.newEventToken(automaton, _initialState_1);
        }
      }
    };
    IterableExtensions.<Automaton>forEach(_automata, _function);
  }
  
  @Pure
  public IEventModelManager getEventModelManager() {
    return this.eventModelManager;
  }
  
  public void setEventModelManager(final IEventModelManager eventModelManager) {
    this.eventModelManager = eventModelManager;
  }
}
