/**
 * Copyright (c) 2004-2013, Abel Hegedus, Zoltan Ujhelyi and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus, Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.emf.runtime.rules.batch;

import com.google.common.base.Objects;
import java.util.HashSet;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.evm.api.ActivationLifeCycle;
import org.eclipse.incquery.runtime.evm.api.Job;
import org.eclipse.incquery.runtime.evm.api.RuleSpecification;
import org.eclipse.incquery.runtime.evm.api.event.EventFilter;
import org.eclipse.incquery.runtime.evm.api.event.EventType;
import org.eclipse.incquery.runtime.evm.specific.Jobs;
import org.eclipse.incquery.runtime.evm.specific.Rules;
import org.eclipse.incquery.runtime.evm.specific.event.IncQueryActivationStateEnum;
import org.eclipse.incquery.runtime.evm.specific.event.IncQueryEventTypeEnum;
import org.eclipse.incquery.runtime.evm.specific.event.IncQuerySinglePatternMatchEventFilter;
import org.eclipse.incquery.runtime.evm.specific.lifecycle.DefaultActivationLifeCycle;
import org.eclipse.incquery.runtime.evm.specific.lifecycle.UnmodifiableActivationLifeCycle;
import org.eclipse.viatra.emf.runtime.rules.ITransformationRule;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function0;

/**
 * Wrapper class for transformation rule definition to hide EVM specific internals.
 * 
 * Subclasses can simply override the abstract precondition and model manipulation methods.
 * 
 * @author Abel Hegedus, Zoltan Ujhelyi
 */
@SuppressWarnings("all")
public class BatchTransformationRule<Match extends IPatternMatch, Matcher extends IncQueryMatcher<Match>> implements ITransformationRule<Match, Matcher> {
  /**
   * Lifecycle for a rule that does not store the list of fired activations; thus allows re-firing the same activation again.
   */
  public final static ActivationLifeCycle STATELESS_RULE_LIFECYCLE = new Function0<ActivationLifeCycle>() {
    public ActivationLifeCycle apply() {
      UnmodifiableActivationLifeCycle _xblockexpression = null;
      {
        final ActivationLifeCycle cycle = ActivationLifeCycle.create(IncQueryActivationStateEnum.INACTIVE);
        cycle.addStateTransition(IncQueryActivationStateEnum.INACTIVE, IncQueryEventTypeEnum.MATCH_APPEARS, IncQueryActivationStateEnum.APPEARED);
        cycle.addStateTransition(IncQueryActivationStateEnum.APPEARED, EventType.RuleEngineEventType.FIRE, IncQueryActivationStateEnum.APPEARED);
        cycle.addStateTransition(IncQueryActivationStateEnum.APPEARED, IncQueryEventTypeEnum.MATCH_DISAPPEARS, IncQueryActivationStateEnum.INACTIVE);
        _xblockexpression = UnmodifiableActivationLifeCycle.copyOf(cycle);
      }
      return _xblockexpression;
    }
  }.apply();
  
  /**
   * Lifecycle for a rule that stores the list of fired activations; thus effectively forbids re-firing the same activation.
   */
  public final static DefaultActivationLifeCycle STATEFUL_RULE_LIFECYCLE = DefaultActivationLifeCycle.DEFAULT_NO_UPDATE_AND_DISAPPEAR;
  
  protected String ruleName;
  
  private final ActivationLifeCycle lifecycle;
  
  private RuleSpecification<Match> ruleSpec;
  
  private final IQuerySpecification<Matcher> precondition;
  
  private final IMatchProcessor<Match> action;
  
  private final EventFilter<Match> filter;
  
  protected BatchTransformationRule() {
    this("", null, BatchTransformationRule.STATELESS_RULE_LIFECYCLE, null);
  }
  
  public BatchTransformationRule(final String rulename, final IQuerySpecification<Matcher> matcher, final ActivationLifeCycle lifecycle, final IMatchProcessor<Match> action) {
    this(rulename, matcher, lifecycle, action, IncQuerySinglePatternMatchEventFilter.<Match>createFilter(((Match) matcher.newEmptyMatch().toImmutable())));
  }
  
  public BatchTransformationRule(final String rulename, final IQuerySpecification<Matcher> matcher, final ActivationLifeCycle lifecycle, final IMatchProcessor<Match> action, final EventFilter<Match> filter) {
    this.ruleName = rulename;
    this.precondition = matcher;
    this.action = action;
    this.lifecycle = lifecycle;
    this.filter = filter;
  }
  
  public BatchTransformationRule(final BatchTransformationRule<Match, Matcher> rule, final EventFilter<Match> filter) {
    this.ruleName = rule.ruleName;
    this.precondition = rule.precondition;
    this.action = rule.action;
    this.lifecycle = rule.lifecycle;
    this.filter = filter;
  }
  
  /**
   * @deprecated Use {#getName) instead
   */
  @Deprecated
  public String getRuleName() {
    return this.getName();
  }
  
  @Override
  public String getName() {
    return this.ruleName;
  }
  
  /**
   * Returns a RuleSpecification that can be added to a rule engine.
   */
  @Override
  public RuleSpecification<Match> getRuleSpecification() {
    RuleSpecification<Match> _xblockexpression = null;
    {
      boolean _equals = Objects.equal(this.ruleSpec, null);
      if (_equals) {
        final IQuerySpecification<Matcher> querySpec = this.precondition;
        final Job<Match> job = Jobs.<Match>newStatelessJob(IncQueryActivationStateEnum.APPEARED, this.action);
        HashSet<Job<Match>> _newHashSet = CollectionLiterals.<Job<Match>>newHashSet(job);
        RuleSpecification<Match> _newMatcherRuleSpecification = Rules.<Match>newMatcherRuleSpecification(querySpec, this.lifecycle, _newHashSet);
        this.ruleSpec = _newMatcherRuleSpecification;
      }
      _xblockexpression = this.ruleSpec;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the query specification representing the pattern used as a precondition.
   */
  @Override
  public IQuerySpecification<Matcher> getPrecondition() {
    return this.precondition;
  }
  
  /**
   * Return an IMatchProcessor representing the model manipulation executed by the rule.
   */
  public IMatchProcessor<Match> getAction() {
    return this.action;
  }
  
  @Override
  public EventFilter<? super Match> getFilter() {
    return this.filter;
  }
}
