/**
 * Copyright (c) 2004-2013, Abel Hegedus, Zoltan Ujhelyi and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus, Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.emf.runtime.rules.batch;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import java.util.HashSet;
import java.util.Set;
import org.eclipse.incquery.runtime.api.GenericPatternGroup;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQueryGroup;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.evm.api.Activation;
import org.eclipse.incquery.runtime.evm.api.Context;
import org.eclipse.incquery.runtime.evm.api.RuleEngine;
import org.eclipse.incquery.runtime.evm.api.RuleSpecification;
import org.eclipse.incquery.runtime.evm.api.event.EventFilter;
import org.eclipse.incquery.runtime.evm.api.resolver.ScopedConflictSet;
import org.eclipse.viatra.emf.runtime.filters.MatchParameterFilter;
import org.eclipse.viatra.emf.runtime.rules.BatchTransformationRuleGroup;
import org.eclipse.viatra.emf.runtime.rules.batch.BatchTransformationRule;
import org.eclipse.viatra.emf.runtime.transformation.batch.BatchTransformation;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Utility class for simple rule usage
 * 
 * @author Abel Hegedus, Zoltan Ujhelyi
 */
@SuppressWarnings("all")
public class BatchTransformationStatements {
  private final IncQueryEngine iqEngine;
  
  private final RuleEngine ruleEngine;
  
  private final Context context;
  
  public BatchTransformationStatements(final BatchTransformation transformation) {
    RuleEngine _ruleEngine = transformation.getRuleEngine();
    this.ruleEngine = _ruleEngine;
    Context _context = transformation.getContext();
    this.context = _context;
    IncQueryEngine _iqEngine = transformation.getIqEngine();
    this.iqEngine = _iqEngine;
  }
  
  public BatchTransformationStatements(final RuleEngine ruleEngine, final IncQueryEngine iqEngine, final Context context) {
    this.ruleEngine = ruleEngine;
    this.context = context;
    this.iqEngine = iqEngine;
  }
  
  /**
   * Executes the selected rule with the selected filter as long as there
   * are possible matches of its preconditions and the break condition is
   * not fulfilled. The matches are executed one-by-one, in case of conflicts
   * only one of the conflicting matches will cause an execution.
   */
  public <Match extends IPatternMatch> boolean fireUntil(final BatchTransformationRule<Match, ?> rule, final Predicate<Match> breakCondition) {
    boolean _xblockexpression = false;
    {
      RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
      final EventFilter<Match> filter = _ruleSpecification.createEmptyFilter();
      RuleSpecification<Match> _ruleSpecification_1 = rule.getRuleSpecification();
      _xblockexpression = this.<Match>fireUntil(_ruleSpecification_1, breakCondition, filter);
    }
    return _xblockexpression;
  }
  
  /**
   * Executes the selected rule with the selected filter as long as there
   * are possible matches of its precondition and the break condition is
   * not fulfilled. The matches are executed one-by-one, in case of conflicts
   * only one of the conflicting matches will cause an execution.
   */
  public <Match extends IPatternMatch> boolean fireUntil(final BatchTransformationRule<Match, ?> rule, final Predicate<Match> breakCondition, final Pair<String, Object>... filterParameters) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    MatchParameterFilter _matchParameterFilter = new MatchParameterFilter(filterParameters);
    return this.<Match>fireUntil(_ruleSpecification, breakCondition, _matchParameterFilter);
  }
  
  /**
   * Executes the selected rule with the selected filter as long as there
   * are possible matches of its precondition and the break condition is
   * not fulfilled. The matches are executed one-by-one, in case of conflicts
   * only one of the conflicting matches will cause an execution.
   */
  public <Match extends IPatternMatch> boolean fireUntil(final BatchTransformationRule<Match, ?> rule, final Predicate<Match> breakCondition, final EventFilter<? super Match> filter) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    return this.<Match>fireUntil(_ruleSpecification, breakCondition, filter);
  }
  
  /**
   * Executes the selected rules with the selected filter as long as there
   * are possible matches of any of their preconditions and the break condition is
   * not fulfilled. The matches are executed one-by-one, in case of conflicts
   * only one of the conflicting matches will cause an execution.
   */
  public void fireUntil(final BatchTransformationRuleGroup rules, final Predicate<IPatternMatch> breakCondition) {
    this.registerRules(rules);
    Multimap<RuleSpecification<?>, EventFilter<?>> _filteredRuleMap = rules.getFilteredRuleMap();
    final ScopedConflictSet conflictSet = this.ruleEngine.createScopedConflictSet(_filteredRuleMap);
    this.<IPatternMatch>fireUntil(conflictSet, breakCondition);
    conflictSet.dispose();
  }
  
  private <Match extends IPatternMatch> boolean fireUntil(final RuleSpecification<Match> ruleSpecification, final Predicate<Match> breakCondition, final EventFilter<? super Match> filter) {
    boolean _xblockexpression = false;
    {
      this.<Match>registerRule(ruleSpecification, filter);
      final ScopedConflictSet conflictSet = this.ruleEngine.<Match>createScopedConflictSet(ruleSpecification, filter);
      this.<Match>fireUntil(conflictSet, breakCondition);
      conflictSet.dispose();
      _xblockexpression = this.<Match>disposeRule(ruleSpecification, filter);
    }
    return _xblockexpression;
  }
  
  private <Match extends IPatternMatch> void fireUntil(final ScopedConflictSet conflictSet, final Predicate<Match> breakCondition) {
    Activation<Match> act = null;
    boolean continue_ = true;
    while ((continue_ && (!Objects.equal((act = ((Activation<Match>) conflictSet.getNextActivation())), null)))) {
      {
        this.fireActivation(act);
        Match _atom = act.getAtom();
        boolean _apply = breakCondition.apply(_atom);
        boolean _not = (!_apply);
        continue_ = _not;
      }
    }
  }
  
  /**
   * Executes the selected rule with the selected filter as long as there
   * are possible matches of its precondition. The matches are executed
   * one-by-one, in case of conflicts only one of the conflicting matches
   * will cause an execution.
   */
  public <Match extends IPatternMatch> boolean fireWhilePossible(final BatchTransformationRule<Match, ?> rule) {
    final Predicate<Match> _function = new Predicate<Match>() {
      @Override
      public boolean apply(final Match it) {
        return false;
      }
    };
    return this.<Match>fireUntil(rule, _function);
  }
  
  /**
   * Executes the selected rule with the selected filter as long as there
   * are possible matches of its preconditions. The matches are executed
   * one-by-one, in case of conflicts only one of the conflicting matches
   * will cause an execution.
   */
  public <Match extends IPatternMatch> boolean fireWhilePossible(final BatchTransformationRule<Match, ?> rule, final EventFilter<? super Match> filter) {
    final Predicate<Match> _function = new Predicate<Match>() {
      @Override
      public boolean apply(final Match it) {
        return false;
      }
    };
    return this.<Match>fireUntil(rule, _function, filter);
  }
  
  /**
   * Executes the selected rules with the selected filter as long as there
   * are possible matches of any of their preconditions. The matches are
   * executed one-by-one, in case of conflicts only one of the conflicting
   * matches will cause an execution.
   */
  public void fireWhilePossible(final BatchTransformationRuleGroup rules) {
    final Predicate<IPatternMatch> _function = new Predicate<IPatternMatch>() {
      @Override
      public boolean apply(final IPatternMatch it) {
        return false;
      }
    };
    this.fireUntil(rules, _function);
  }
  
  /**
   * Executes the selected rule with the selected filter on its current match set of the precondition.
   */
  public <Match extends IPatternMatch> void fireAllCurrent(final BatchTransformationRule<Match, ?> rule) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    final EventFilter<Match> filter = _ruleSpecification.createEmptyFilter();
    RuleSpecification<Match> _ruleSpecification_1 = rule.getRuleSpecification();
    this.<Match>fireAllCurrent(_ruleSpecification_1, filter);
  }
  
  /**
   * Executes the selected rule with the selected filter on its current match set of the precondition.
   */
  public <Match extends IPatternMatch> void fireAllCurrent(final BatchTransformationRule<Match, ?> rule, final Pair<String, Object>... parameterFilter) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    MatchParameterFilter _matchParameterFilter = new MatchParameterFilter(parameterFilter);
    this.<Match>fireAllCurrent(_ruleSpecification, _matchParameterFilter);
  }
  
  /**
   * Executes the selected rule with the selected filter on its current match set of the precondition.
   */
  public <Match extends IPatternMatch> void fireAllCurrent(final BatchTransformationRule<Match, ?> rule, final EventFilter<? super Match> filter) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    this.<Match>fireAllCurrent(_ruleSpecification, filter);
  }
  
  private <Match extends IPatternMatch> void fireAllCurrent(final RuleSpecification<Match> ruleSpecification, final EventFilter<? super Match> filter) {
    this.<Match>registerRule(ruleSpecification, filter);
    final ScopedConflictSet conflictSet = this.ruleEngine.<Match>createScopedConflictSet(ruleSpecification, filter);
    Set<Activation<?>> _conflictingActivations = conflictSet.getConflictingActivations();
    final ImmutableSet<Activation<?>> conflictingActivations = ImmutableSet.<Activation<?>>copyOf(_conflictingActivations);
    for (final Activation<?> act : conflictingActivations) {
      this.fireActivation(((Activation<Match>) act));
    }
    this.<Match>disposeRule(ruleSpecification, filter);
    conflictSet.dispose();
  }
  
  public <Match extends IPatternMatch> boolean registerRule(final RuleSpecification<Match> ruleSpecification) {
    EventFilter<Match> _createEmptyFilter = ruleSpecification.createEmptyFilter();
    return this.<Match>registerRule(ruleSpecification, _createEmptyFilter);
  }
  
  public <Match extends IPatternMatch> boolean registerRule(final RuleSpecification<Match> ruleSpecification, final EventFilter<? super Match> filter) {
    return this.ruleEngine.<Match>addRule(ruleSpecification, filter);
  }
  
  public void registerRules(final BatchTransformationRuleGroup rules) {
    try {
      Iterable<BatchTransformationRule<?, ?>> _filterNull = IterableExtensions.<BatchTransformationRule<?, ?>>filterNull(rules);
      final Function1<BatchTransformationRule<?, ?>, IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>> _function = new Function1<BatchTransformationRule<?, ?>, IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>>() {
        @Override
        public IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> apply(final BatchTransformationRule<?, ?> it) {
          return it.getPrecondition();
        }
      };
      final Iterable<IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>> notNullPreconditions = IterableExtensions.<BatchTransformationRule<?, ?>, IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>>map(_filterNull, _function);
      HashSet<IQuerySpecification<?>> _newHashSet = Sets.<IQuerySpecification<?>>newHashSet(notNullPreconditions);
      IQueryGroup _of = GenericPatternGroup.of(_newHashSet);
      _of.prepare(this.iqEngine);
      Iterable<BatchTransformationRule<?, ?>> _filterNull_1 = IterableExtensions.<BatchTransformationRule<?, ?>>filterNull(rules);
      final Procedure1<BatchTransformationRule<?, ?>> _function_1 = new Procedure1<BatchTransformationRule<?, ?>>() {
        @Override
        public void apply(final BatchTransformationRule<?, ?> it) {
          RuleSpecification<? extends IPatternMatch> _ruleSpecification = it.getRuleSpecification();
          EventFilter<?> _filter = it.getFilter();
          BatchTransformationStatements.this.ruleEngine.addRule(_ruleSpecification, ((EventFilter<IPatternMatch>) _filter));
        }
      };
      IterableExtensions.<BatchTransformationRule<?, ?>>forEach(_filterNull_1, _function_1);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public <Match extends IPatternMatch> boolean disposeRule(final RuleSpecification<Match> ruleSpecification) {
    EventFilter<Match> _createEmptyFilter = ruleSpecification.createEmptyFilter();
    return this.<Match>disposeRule(ruleSpecification, _createEmptyFilter);
  }
  
  public <Match extends IPatternMatch> boolean disposeRule(final RuleSpecification<Match> ruleSpecification, final EventFilter<? super Match> filter) {
    return this.ruleEngine.<Match>removeRule(ruleSpecification, filter);
  }
  
  public void disposeRules(final BatchTransformationRuleGroup rules) {
    Iterable<BatchTransformationRule<?, ?>> _filterNull = IterableExtensions.<BatchTransformationRule<?, ?>>filterNull(rules);
    final Procedure1<BatchTransformationRule<?, ?>> _function = new Procedure1<BatchTransformationRule<?, ?>>() {
      @Override
      public void apply(final BatchTransformationRule<?, ?> it) {
        RuleSpecification<? extends IPatternMatch> _ruleSpecification = it.getRuleSpecification();
        EventFilter<?> _filter = it.getFilter();
        BatchTransformationStatements.this.ruleEngine.removeRule(_ruleSpecification, ((EventFilter<IPatternMatch>) _filter));
      }
    };
    IterableExtensions.<BatchTransformationRule<?, ?>>forEach(_filterNull, _function);
  }
  
  /**
   * Selects and fires an activation of the selected rule with a corresponding filter.</p>
   * 
   * <p><strong>Warning</strong>: the selection criteria is not specified - it is neither random nor controllable
   */
  public <Match extends IPatternMatch> void fireOne(final BatchTransformationRule<Match, ?> rule) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    final EventFilter<Match> filter = _ruleSpecification.createEmptyFilter();
    RuleSpecification<Match> _ruleSpecification_1 = rule.getRuleSpecification();
    this.<Match>fireOne(_ruleSpecification_1, filter);
  }
  
  /**
   * Selects and fires an activation of the selected rule with a corresponding filter.</p>
   * 
   * <p><strong>Warning</strong>: the selection criteria is not specified - it is neither random nor controllable
   */
  public <Match extends IPatternMatch> void fireOne(final BatchTransformationRule<Match, ?> rule, final EventFilter<? super Match> filter) {
    RuleSpecification<Match> _ruleSpecification = rule.getRuleSpecification();
    this.<Match>fireOne(_ruleSpecification, filter);
  }
  
  private <Match extends IPatternMatch> void fireOne(final RuleSpecification<Match> ruleSpecification, final EventFilter<? super Match> filter) {
    this.<Match>registerRule(ruleSpecification, filter);
    final ScopedConflictSet conflictSet = this.ruleEngine.<Match>createScopedConflictSet(ruleSpecification, filter);
    Set<Activation<?>> _conflictingActivations = conflictSet.getConflictingActivations();
    Activation<?> _head = IterableExtensions.<Activation<?>>head(_conflictingActivations);
    final Activation<Match> act = ((Activation<Match>) _head);
    this.fireActivation(act);
    conflictSet.dispose();
  }
  
  private void fireActivation(final Activation<?> act) {
    boolean _and = false;
    boolean _notEquals = (!Objects.equal(act, null));
    if (!_notEquals) {
      _and = false;
    } else {
      boolean _isEnabled = act.isEnabled();
      _and = _isEnabled;
    }
    if (_and) {
      act.fire(this.context);
    }
    return;
  }
}
