/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.incquery.querybasedfeatures.tooling;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.ClassType;
import org.eclipse.incquery.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.incquery.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.incquery.patternlanguage.patternLanguage.Annotation;
import org.eclipse.incquery.patternlanguage.patternLanguage.AnnotationParameter;
import org.eclipse.incquery.patternlanguage.patternLanguage.BoolValue;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.patternlanguage.patternLanguage.StringValue;
import org.eclipse.incquery.patternlanguage.patternLanguage.Type;
import org.eclipse.incquery.patternlanguage.patternLanguage.ValueReference;
import org.eclipse.incquery.patternlanguage.patternLanguage.Variable;
import org.eclipse.incquery.patternlanguage.patternLanguage.VariableReference;
import org.eclipse.incquery.patternlanguage.patternLanguage.VariableValue;
import org.eclipse.incquery.querybasedfeatures.runtime.QueryBasedFeatureKind;
import org.eclipse.incquery.querybasedfeatures.tooling.DerivedFeatureSourceCodeUtil;
import org.eclipse.incquery.querybasedfeatures.tooling.ModelCodeBasedGenerator;
import org.eclipse.incquery.querybasedfeatures.tooling.QueryBasedFeatureParameters;
import org.eclipse.incquery.querybasedfeatures.tooling.SettingDelegateBasedGenerator;
import org.eclipse.incquery.tooling.core.generator.ExtensionData;
import org.eclipse.incquery.tooling.core.generator.ExtensionGenerator;
import org.eclipse.incquery.tooling.core.generator.fragments.IGenerationFragment;
import org.eclipse.incquery.tooling.core.generator.genmodel.IEiqGenmodelProvider;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.generator.IFileSystemAccess;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.w3c.dom.Element;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryBasedFeatureGenerator implements IGenerationFragment {
  @Inject
  protected IEiqGenmodelProvider provider;
  
  @Inject
  protected Logger logger;
  
  @Inject
  protected IErrorFeedback errorFeedback;
  
  @Inject
  @Extension
  protected ExtensionGenerator exGen;
  
  @Inject
  @Extension
  protected DerivedFeatureSourceCodeUtil codeGen;
  
  protected static String ANNOTATION_LITERAL = "QueryBasedFeature";
  
  protected static String DERIVED_EXTENSION_POINT = "org.eclipse.incquery.runtime.base.wellbehaving.derived.features";
  
  protected static String DERIVED_ERROR_CODE = "org.eclipse.incquery.runtime.querybasedfeature.error";
  
  protected static String ECORE_ANNOTATION = "http://www.eclipse.org/emf/2002/Ecore";
  
  protected static String SETTING_DELEGATES_KEY = "settingDelegates";
  
  protected static String SETTING_DELEGATES_VALUE = "org.eclipse.incquery.querybasedfeature";
  
  protected static String DERIVED_EXTENSION_PREFIX = "extension.derived.";
  
  protected static Map<String, QueryBasedFeatureKind> kinds = CollectionLiterals.<String, QueryBasedFeatureKind>newHashMap(
    Pair.<String, QueryBasedFeatureKind>of("single", QueryBasedFeatureKind.SINGLE_REFERENCE), 
    Pair.<String, QueryBasedFeatureKind>of("many", QueryBasedFeatureKind.MANY_REFERENCE), 
    Pair.<String, QueryBasedFeatureKind>of("sum", QueryBasedFeatureKind.SUM), 
    Pair.<String, QueryBasedFeatureKind>of("iteration", QueryBasedFeatureKind.ITERATION));
  
  private ModelCodeBasedGenerator modelCodeBasedGenerator = new ModelCodeBasedGenerator(this);
  
  private SettingDelegateBasedGenerator delegateBasedGenerator = new SettingDelegateBasedGenerator(this);
  
  public IPath[] getAdditionalBinIncludes() {
    return ((IPath[])Conversions.unwrapArray(CollectionLiterals.<IPath>newArrayList(), IPath.class));
  }
  
  public String[] getProjectDependencies() {
    return ((String[])Conversions.unwrapArray(CollectionLiterals.<String>newArrayList(), String.class));
  }
  
  public String getProjectPostfix() {
    return null;
  }
  
  public void generateFiles(final Pattern pattern, final IFileSystemAccess fsa) {
    this.processAnnotations(pattern, true);
  }
  
  public void cleanUp(final Pattern pattern, final IFileSystemAccess fsa) {
    this.processAnnotations(pattern, false);
  }
  
  private void processAnnotations(final Pattern pattern, final boolean generate) {
    Collection<Annotation> _annotationsByName = CorePatternLanguageHelper.getAnnotationsByName(pattern, QueryBasedFeatureGenerator.ANNOTATION_LITERAL);
    for (final Annotation annotation : _annotationsByName) {
      {
        final ValueReference useModelCode = CorePatternLanguageHelper.getFirstAnnotationParameter(annotation, "generateIntoModelCode");
        boolean _and = false;
        boolean _notEquals = (!Objects.equal(useModelCode, null));
        if (!_notEquals) {
          _and = false;
        } else {
          boolean _isValue = ((BoolValue) useModelCode).isValue();
          boolean _equals = (_isValue == true);
          _and = _equals;
        }
        if (_and) {
          this.modelCodeBasedGenerator.processJavaFiles(pattern, annotation, generate);
        } else {
          this.delegateBasedGenerator.updateAnnotations(pattern, annotation, generate);
        }
      }
    }
  }
  
  public Iterable<ExtensionData> extensionContribution(final Pattern pattern) {
    final ArrayList<QueryBasedFeatureParameters> parameterList = CollectionLiterals.<QueryBasedFeatureParameters>newArrayList();
    Collection<Annotation> _annotationsByName = CorePatternLanguageHelper.getAnnotationsByName(pattern, QueryBasedFeatureGenerator.ANNOTATION_LITERAL);
    for (final Annotation annotation : _annotationsByName) {
      try {
        QueryBasedFeatureParameters _processDerivedFeatureAnnotation = this.processDerivedFeatureAnnotation(pattern, annotation, false);
        parameterList.add(_processDerivedFeatureAnnotation);
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          String _message = e.getMessage();
          this.logger.error(_message);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    boolean _isEmpty = parameterList.isEmpty();
    if (_isEmpty) {
      return CollectionLiterals.<ExtensionData>newArrayList();
    }
    String _derivedContributionId = this.derivedContributionId(pattern);
    final Procedure1<Element> _function = new Procedure1<Element>() {
      public void apply(final Element it) {
        final Procedure1<QueryBasedFeatureParameters> _function = new Procedure1<QueryBasedFeatureParameters>() {
          public void apply(final QueryBasedFeatureParameters parameters) {
            final Procedure1<Element> _function = new Procedure1<Element>() {
              public void apply(final Element it) {
                String _nsURI = parameters.ePackage.getNsURI();
                QueryBasedFeatureGenerator.this.exGen.contribAttribute(it, "package-nsUri", _nsURI);
                String _name = parameters.source.getName();
                QueryBasedFeatureGenerator.this.exGen.contribAttribute(it, "classifier-name", _name);
                String _name_1 = parameters.feature.getName();
                QueryBasedFeatureGenerator.this.exGen.contribAttribute(it, "feature-name", _name_1);
              }
            };
            QueryBasedFeatureGenerator.this.exGen.contribElement(it, "wellbehaving-derived-feature", _function);
          }
        };
        IterableExtensions.<QueryBasedFeatureParameters>forEach(parameterList, _function);
      }
    };
    ExtensionData _contribExtension = this.exGen.contribExtension(_derivedContributionId, QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT, _function);
    final ArrayList<ExtensionData> wellbehaving = CollectionLiterals.<ExtensionData>newArrayList(_contribExtension);
    return wellbehaving;
  }
  
  public Collection<Pair<String, String>> getRemovableExtensions() {
    Pair<String, String> _of = Pair.<String, String>of(QueryBasedFeatureGenerator.DERIVED_EXTENSION_PREFIX, QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT);
    return CollectionLiterals.<Pair<String, String>>newArrayList(_of);
  }
  
  public Iterable<Pair<String, String>> removeExtension(final Pattern pattern) {
    String _derivedContributionId = this.derivedContributionId(pattern);
    Pair<String, String> _of = Pair.<String, String>of(_derivedContributionId, QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT);
    return CollectionLiterals.<Pair<String, String>>newArrayList(_of);
  }
  
  protected String derivedContributionId(final Pattern pattern) {
    String _fullyQualifiedName = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
    return (QueryBasedFeatureGenerator.DERIVED_EXTENSION_PREFIX + _fullyQualifiedName);
  }
  
  protected QueryBasedFeatureParameters processDerivedFeatureAnnotation(final Pattern pattern, final Annotation annotation, final boolean feedback) {
    final QueryBasedFeatureParameters parameters = new QueryBasedFeatureParameters();
    parameters.pattern = pattern;
    parameters.annotation = annotation;
    String sourceTmp = "";
    String targetTmp = "";
    String featureTmp = "";
    String kindTmp = "";
    boolean keepCacheTmp = true;
    EList<Variable> _parameters = pattern.getParameters();
    int _size = _parameters.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      if (feedback) {
        this.errorFeedback.reportError(pattern, "Pattern has less than 2 parameters!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus = ("Query-based feature pattern " + _fullyQualifiedName);
      String _plus_1 = (_plus + " has less than 2 parameters!");
      throw new IllegalArgumentException(_plus_1);
    }
    EList<AnnotationParameter> _parameters_1 = annotation.getParameters();
    for (final AnnotationParameter ap : _parameters_1) {
      String _name = ap.getName();
      boolean _matches = _name.matches("source");
      if (_matches) {
        ValueReference _value = ap.getValue();
        VariableReference _value_1 = ((VariableValue) _value).getValue();
        String _var = _value_1.getVar();
        sourceTmp = _var;
      } else {
        String _name_1 = ap.getName();
        boolean _matches_1 = _name_1.matches("target");
        if (_matches_1) {
          ValueReference _value_2 = ap.getValue();
          VariableReference _value_3 = ((VariableValue) _value_2).getValue();
          String _var_1 = _value_3.getVar();
          targetTmp = _var_1;
        } else {
          String _name_2 = ap.getName();
          boolean _matches_2 = _name_2.matches("feature");
          if (_matches_2) {
            ValueReference _value_4 = ap.getValue();
            String _value_5 = ((StringValue) _value_4).getValue();
            featureTmp = _value_5;
          } else {
            String _name_3 = ap.getName();
            boolean _matches_3 = _name_3.matches("kind");
            if (_matches_3) {
              ValueReference _value_6 = ap.getValue();
              String _value_7 = ((StringValue) _value_6).getValue();
              kindTmp = _value_7;
            } else {
              String _name_4 = ap.getName();
              boolean _matches_4 = _name_4.matches("keepCache");
              if (_matches_4) {
                ValueReference _value_8 = ap.getValue();
                boolean _isValue = ((BoolValue) _value_8).isValue();
                keepCacheTmp = _isValue;
              }
            }
          }
        }
      }
    }
    boolean _equals = Objects.equal(featureTmp, "");
    if (_equals) {
      String _name_5 = pattern.getName();
      featureTmp = _name_5;
    }
    boolean _equals_1 = Objects.equal(sourceTmp, "");
    if (_equals_1) {
      EList<Variable> _parameters_2 = pattern.getParameters();
      Variable _get = _parameters_2.get(0);
      String _name_6 = _get.getName();
      sourceTmp = _name_6;
    }
    Map<String, Integer> _parameterPositionsByName = CorePatternLanguageHelper.getParameterPositionsByName(pattern);
    Set<String> _keySet = _parameterPositionsByName.keySet();
    boolean _contains = _keySet.contains(sourceTmp);
    boolean _not = (!_contains);
    if (_not) {
      if (feedback) {
        this.errorFeedback.reportError(annotation, (("No parameter for source " + sourceTmp) + " !"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_1 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_2 = ("Query-based feature pattern " + _fullyQualifiedName_1);
      String _plus_3 = (_plus_2 + ": No parameter for source ");
      String _plus_4 = (_plus_3 + sourceTmp);
      String _plus_5 = (_plus_4 + " !");
      throw new IllegalArgumentException(_plus_5);
    }
    EList<Variable> _parameters_3 = pattern.getParameters();
    Map<String, Integer> _parameterPositionsByName_1 = CorePatternLanguageHelper.getParameterPositionsByName(pattern);
    Integer _get_1 = _parameterPositionsByName_1.get(sourceTmp);
    final Variable sourcevar = _parameters_3.get((_get_1).intValue());
    final Type sourceType = sourcevar.getType();
    if (((!(sourceType instanceof ClassType)) || (!(((ClassType) sourceType).getClassname() instanceof EClass)))) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, (("Source " + sourceTmp) + " is not EClass!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_2 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_6 = ("Query-based feature pattern " + _fullyQualifiedName_2);
      String _plus_7 = (_plus_6 + ": Source ");
      String _plus_8 = (_plus_7 + sourceTmp);
      String _plus_9 = (_plus_8 + " is not EClass!");
      throw new IllegalArgumentException(_plus_9);
    }
    EClassifier _classname = ((ClassType) sourceType).getClassname();
    EClass source = ((EClass) _classname);
    parameters.sourceVar = sourceTmp;
    parameters.source = source;
    boolean _or = false;
    boolean _equals_2 = Objects.equal(source, null);
    if (_equals_2) {
      _or = true;
    } else {
      EPackage _ePackage = source.getEPackage();
      boolean _equals_3 = Objects.equal(_ePackage, null);
      _or = _equals_3;
    }
    if (_or) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "Source EClass or EPackage not found!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_3 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_10 = ("Query-based feature pattern " + _fullyQualifiedName_3);
      String _plus_11 = (_plus_10 + ": Source EClass or EPackage not found!");
      throw new IllegalArgumentException(_plus_11);
    }
    final EPackage pckg = source.getEPackage();
    boolean _equals_4 = Objects.equal(pckg, null);
    if (_equals_4) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "EPackage not found!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_4 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_12 = ("Query-based feature pattern " + _fullyQualifiedName_4);
      String _plus_13 = (_plus_12 + ": EPackage not found!");
      throw new IllegalArgumentException(_plus_13);
    }
    parameters.ePackage = pckg;
    final String featureString = featureTmp;
    EList<EStructuralFeature> _eAllStructuralFeatures = source.getEAllStructuralFeatures();
    final Function1<EStructuralFeature, Boolean> _function = new Function1<EStructuralFeature, Boolean>() {
      public Boolean apply(final EStructuralFeature it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, featureString));
      }
    };
    final Iterable<EStructuralFeature> features = IterableExtensions.<EStructuralFeature>filter(_eAllStructuralFeatures, _function);
    int _size_1 = IterableExtensions.size(features);
    boolean _notEquals = (_size_1 != 1);
    if (_notEquals) {
      if (feedback) {
        String _name_7 = source.getName();
        String _plus_14 = ((("Feature " + featureTmp) + " not found in class ") + _name_7);
        String _plus_15 = (_plus_14 + "!");
        this.errorFeedback.reportError(annotation, _plus_15, QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_5 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_16 = ("Query-based feature pattern " + _fullyQualifiedName_5);
      String _plus_17 = (_plus_16 + ": Feature ");
      String _plus_18 = (_plus_17 + featureTmp);
      String _plus_19 = (_plus_18 + " not found in class ");
      String _name_8 = source.getName();
      String _plus_20 = (_plus_19 + _name_8);
      String _plus_21 = (_plus_20 + "!");
      throw new IllegalArgumentException(_plus_21);
    }
    Iterator<EStructuralFeature> _iterator = features.iterator();
    final EStructuralFeature feature = _iterator.next();
    boolean _and = false;
    boolean _and_1 = false;
    boolean _isDerived = feature.isDerived();
    if (!_isDerived) {
      _and_1 = false;
    } else {
      boolean _isTransient = feature.isTransient();
      _and_1 = _isTransient;
    }
    if (!_and_1) {
      _and = false;
    } else {
      boolean _isVolatile = feature.isVolatile();
      _and = _isVolatile;
    }
    boolean _not_1 = (!_and);
    if (_not_1) {
      if (feedback) {
        this.errorFeedback.reportError(annotation, (("Feature " + featureTmp) + " must be set derived, transient, volatile, non-changeable!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_6 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_22 = ("Query-based feature pattern " + _fullyQualifiedName_6);
      String _plus_23 = (_plus_22 + ": Feature ");
      String _plus_24 = (_plus_23 + featureTmp);
      String _plus_25 = (_plus_24 + " must be set derived, transient, volatile!");
      throw new IllegalArgumentException(_plus_25);
    }
    parameters.feature = feature;
    boolean _equals_5 = Objects.equal(kindTmp, "");
    if (_equals_5) {
      boolean _isMany = feature.isMany();
      if (_isMany) {
        kindTmp = "many";
      } else {
        kindTmp = "single";
      }
    }
    Set<String> _keySet_1 = QueryBasedFeatureGenerator.kinds.keySet();
    boolean _contains_1 = _keySet_1.contains(kindTmp);
    boolean _not_2 = (!_contains_1);
    if (_not_2) {
      if (feedback) {
        Set<String> _keySet_2 = QueryBasedFeatureGenerator.kinds.keySet();
        String _plus_26 = ("Kind not set, or not in " + _keySet_2);
        String _plus_27 = (_plus_26 + "!");
        this.errorFeedback.reportError(annotation, _plus_27, QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_7 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_28 = ("Query-based feature pattern " + _fullyQualifiedName_7);
      String _plus_29 = (_plus_28 + ": Kind not set, or not in ");
      Set<String> _keySet_3 = QueryBasedFeatureGenerator.kinds.keySet();
      String _plus_30 = (_plus_29 + _keySet_3);
      String _plus_31 = (_plus_30 + "!");
      throw new IllegalArgumentException(_plus_31);
    }
    final QueryBasedFeatureKind kind = QueryBasedFeatureGenerator.kinds.get(kindTmp);
    parameters.kind = kind;
    boolean _equals_6 = Objects.equal(targetTmp, "");
    if (_equals_6) {
      EList<Variable> _parameters_4 = pattern.getParameters();
      Variable _get_2 = _parameters_4.get(1);
      String _name_9 = _get_2.getName();
      targetTmp = _name_9;
    } else {
      Map<String, Integer> _parameterPositionsByName_2 = CorePatternLanguageHelper.getParameterPositionsByName(pattern);
      Set<String> _keySet_4 = _parameterPositionsByName_2.keySet();
      boolean _contains_2 = _keySet_4.contains(targetTmp);
      boolean _not_3 = (!_contains_2);
      if (_not_3) {
        if (feedback) {
          this.errorFeedback.reportError(annotation, (("Target " + targetTmp) + " not set or no such parameter!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
        }
        String _fullyQualifiedName_8 = CorePatternLanguageHelper.getFullyQualifiedName(pattern);
        String _plus_32 = ("Derived feature pattern " + _fullyQualifiedName_8);
        String _plus_33 = (_plus_32 + ": Target ");
        String _plus_34 = (_plus_33 + targetTmp);
        String _plus_35 = (_plus_34 + " not set or no such parameter!");
        throw new IllegalArgumentException(_plus_35);
      }
    }
    parameters.targetVar = targetTmp;
    parameters.keepCache = keepCacheTmp;
    return parameters;
  }
}
