/**
 * Copyright (c) 2010-2016, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.emf.types;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.viatra.query.patternlanguage.emf.eMFPatternLanguage.ClassType;
import org.eclipse.viatra.query.patternlanguage.emf.eMFPatternLanguage.ReferenceType;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.scoping.IMetamodelProvider;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.JavaType;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternBody;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.RelationType;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Type;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Variable;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableReference;
import org.eclipse.viatra.query.patternlanguage.typing.AbstractTypeSystem;
import org.eclipse.viatra.query.runtime.emf.EMFQueryMetaContext;
import org.eclipse.viatra.query.runtime.emf.types.BaseEMFTypeKey;
import org.eclipse.viatra.query.runtime.emf.types.EClassTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EDataTypeInSlotsKey;
import org.eclipse.viatra.query.runtime.emf.types.EStructuralFeatureInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.Primitives;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * @author Zoltan Ujhelyi
 */
@SuppressWarnings("all")
public class EMFTypeSystem extends AbstractTypeSystem {
  private final static String NON_EMF_TYPE_ENCOUNTERED = "EMF Type System only supports EMF Types but %s found.";
  
  /**
   * This function can be used to extract EClassifier instances from IInputKey instances.
   * If the IInputKey instance does not represent an EClassifier, null is returned
   * @since 1.3
   */
  public final static Function<IInputKey, EClassifier> EXTRACT_CLASSIFIER = new Function<IInputKey, EClassifier>() {
    @Override
    public EClassifier apply(final IInputKey key) {
      EClassifier _switchResult = null;
      boolean _matched = false;
      if (key instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        _switchResult = ((EClassTransitiveInstancesKey)key).getEmfKey();
      }
      if (!_matched) {
        if (key instanceof EDataTypeInSlotsKey) {
          _matched=true;
          _switchResult = ((EDataTypeInSlotsKey)key).getEmfKey();
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      return _switchResult;
    }
  };
  
  @Inject
  private IMetamodelProvider metamodelProvider;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  @Inject
  private Primitives primitives;
  
  @Inject
  private TypeReferences typeReferences;
  
  @Inject
  public EMFTypeSystem(final Logger logger) {
    super(EMFQueryMetaContext.INSTANCE);
  }
  
  @Override
  public IInputKey extractTypeDescriptor(final Type type) {
    Class<? extends Type> _class = type.getClass();
    Preconditions.checkArgument((((type instanceof ClassType) || (type instanceof ReferenceType)) || (type instanceof JavaType)), 
      EMFTypeSystem.NON_EMF_TYPE_ENCOUNTERED, _class);
    if ((type instanceof ClassType)) {
      final EClassifier classifier = ((ClassType) type).getClassname();
      return this.classifierToInputKey(classifier);
    } else {
      if ((type instanceof ReferenceType)) {
        EStructuralFeature _refname = ((ReferenceType)type).getRefname();
        EClassifier _eType = null;
        if (_refname!=null) {
          _eType=_refname.getEType();
        }
        return this.classifierToInputKey(_eType);
      } else {
        if ((type instanceof JavaType)) {
          JvmDeclaredType _classRef = ((JavaType)type).getClassRef();
          String _identifier = _classRef.getIdentifier();
          return new JavaTransitiveInstancesKey(_identifier);
        }
      }
    }
    throw new UnsupportedOperationException();
  }
  
  public IInputKey classifierToInputKey(final EClassifier classifier) {
    BaseEMFTypeKey<? extends EClassifier> _switchResult = null;
    boolean _matched = false;
    if (classifier instanceof EClass) {
      _matched=true;
      _switchResult = new EClassTransitiveInstancesKey(((EClass) classifier));
    }
    if (!_matched) {
      if (classifier instanceof EDataType) {
        _matched=true;
        _switchResult = new EDataTypeInSlotsKey(((EDataType) classifier));
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  @Override
  public IInputKey extractColumnDescriptor(final RelationType type, final int columnIndex) {
    Class<? extends RelationType> _class = type.getClass();
    Preconditions.checkArgument((type instanceof ReferenceType), EMFTypeSystem.NON_EMF_TYPE_ENCOUNTERED, _class);
    if ((type instanceof ReferenceType)) {
      final EStructuralFeature feature = ((ReferenceType) type).getRefname();
      return this.extractColumnDescriptor(feature, columnIndex);
    }
    throw new UnsupportedOperationException();
  }
  
  private IInputKey extractColumnDescriptor(final EStructuralFeature feature, final int columnIndex) {
    if ((0 == columnIndex)) {
      EClass _eContainingClass = feature.getEContainingClass();
      return new EClassTransitiveInstancesKey(_eContainingClass);
    } else {
      if ((feature instanceof EReference)) {
        EClass _eReferenceType = ((EReference) feature).getEReferenceType();
        return new EClassTransitiveInstancesKey(_eReferenceType);
      } else {
        EDataType _eAttributeType = ((EAttribute) feature).getEAttributeType();
        return new EDataTypeInSlotsKey(_eAttributeType);
      }
    }
  }
  
  @Override
  public boolean isConformant(final IInputKey expectedType, final IInputKey actualType) {
    if ((expectedType instanceof EClassTransitiveInstancesKey)) {
      if ((actualType instanceof EClassTransitiveInstancesKey)) {
        EClass _emfKey = ((EClassTransitiveInstancesKey) expectedType).getEmfKey();
        EClass _emfKey_1 = ((EClassTransitiveInstancesKey) actualType).getEmfKey();
        return this.isConform(_emfKey, _emfKey_1);
      }
    } else {
      if ((expectedType instanceof EDataTypeInSlotsKey)) {
        if ((actualType instanceof EDataTypeInSlotsKey)) {
          EDataType _emfKey_2 = ((EDataTypeInSlotsKey) expectedType).getEmfKey();
          EDataType _emfKey_3 = ((EDataTypeInSlotsKey) actualType).getEmfKey();
          return this.isConform(_emfKey_2, _emfKey_3);
        } else {
          if ((actualType instanceof JavaTransitiveInstancesKey)) {
            final Class<?> expectedJavaClass = this.getJavaClass(((EDataTypeInSlotsKey)expectedType));
            final Class<?> actualJavaClass = this.getJavaClass(((JavaTransitiveInstancesKey)actualType));
            return expectedJavaClass.isAssignableFrom(actualJavaClass);
          }
        }
      } else {
        if ((expectedType instanceof JavaTransitiveInstancesKey)) {
          if ((actualType instanceof JavaTransitiveInstancesKey)) {
            final Class<?> expectedClass = ((JavaTransitiveInstancesKey)expectedType).getInstanceClass();
            final Class<?> actualClass = ((JavaTransitiveInstancesKey)actualType).getInstanceClass();
            return expectedClass.isAssignableFrom(actualClass);
          } else {
            if ((actualType instanceof EDataTypeInSlotsKey)) {
              final Class<?> expectedJavaClass_1 = this.getJavaClass(((JavaTransitiveInstancesKey)expectedType));
              final Class<?> actualJavaClass_1 = this.getJavaClass(((EDataTypeInSlotsKey)actualType));
              return expectedJavaClass_1.isAssignableFrom(actualJavaClass_1);
            }
          }
        }
      }
    }
    return false;
  }
  
  private Class<?> getJavaClass(final EDataTypeInSlotsKey key) {
    EDataType _emfKey = key.getEmfKey();
    Class<?> dataTypeClass = _emfKey.getInstanceClass();
    boolean _isPrimitive = dataTypeClass.isPrimitive();
    if (_isPrimitive) {
      Class<?> _wrapperClassForType = AbstractTypeSystem.getWrapperClassForType(dataTypeClass);
      dataTypeClass = _wrapperClassForType;
    }
    return dataTypeClass;
  }
  
  private Class<?> getJavaClass(final JavaTransitiveInstancesKey javaKey) {
    Class<?> javaTypeClass = javaKey.getInstanceClass();
    boolean _isPrimitive = javaTypeClass.isPrimitive();
    if (_isPrimitive) {
      Class<?> _wrapperClassForType = AbstractTypeSystem.getWrapperClassForType(javaTypeClass);
      javaTypeClass = _wrapperClassForType;
    }
    return javaTypeClass;
  }
  
  public boolean isConformant(final ClassType expectedType, final ClassType actualType) {
    final IInputKey expectedClassifier = this.extractTypeDescriptor(expectedType);
    final IInputKey actualClassifier = this.extractTypeDescriptor(actualType);
    return this.isConformant(expectedClassifier, actualClassifier);
  }
  
  private boolean isConform(final EClassifier expectedClassifier, final EClassifier actualClassifier) {
    if ((actualClassifier instanceof EClass)) {
      List<EClass> _compatibleTypesOf = EcoreUtil2.getCompatibleTypesOf(((EClass) actualClassifier));
      return _compatibleTypesOf.contains(expectedClassifier);
    } else {
      return expectedClassifier.equals(actualClassifier);
    }
  }
  
  @Override
  public boolean isConformToRelationColumn(final IInputKey relationType, final int columnIndex, final IInputKey columnType) {
    if ((relationType instanceof EStructuralFeatureInstancesKey)) {
      final EStructuralFeature feature = ((EStructuralFeatureInstancesKey) relationType).getEmfKey();
      IInputKey _extractColumnDescriptor = this.extractColumnDescriptor(feature, columnIndex);
      return this.isConformant(_extractColumnDescriptor, columnType);
    } else {
      return false;
    }
  }
  
  public boolean isConformToRelationSource(final ReferenceType relationType, final ClassType sourceType) {
    final EStructuralFeature featureType = relationType.getRefname();
    final EClassifier classifier = sourceType.getClassname();
    final EClass sourceClass = featureType.getEContainingClass();
    return this.isConform(sourceClass, classifier);
  }
  
  public boolean isConformToRelationTarget(final ReferenceType relationType, final ClassType targetType) {
    final EStructuralFeature featureType = relationType.getRefname();
    final EClassifier classifier = targetType.getClassname();
    final EClassifier targetClassifier = featureType.getEType();
    return this.isConform(targetClassifier, classifier);
  }
  
  @Override
  public JvmTypeReference toJvmTypeReference(final IInputKey type, final EObject context) {
    if ((type instanceof EClassTransitiveInstancesKey)) {
      EClass _emfKey = ((EClassTransitiveInstancesKey) type).getEmfKey();
      return this.getJvmType(_emfKey, context);
    } else {
      if ((type instanceof EDataTypeInSlotsKey)) {
        EDataType _emfKey_1 = ((EDataTypeInSlotsKey) type).getEmfKey();
        return this.getJvmType(_emfKey_1, context);
      } else {
        if ((type instanceof JavaTransitiveInstancesKey)) {
          String _wrappedKey = ((JavaTransitiveInstancesKey) type).getWrappedKey();
          return this.typeReferences.getTypeForName(_wrappedKey, context);
        }
      }
    }
    return this.typeReferences.getTypeForName(Object.class, context);
  }
  
  private JvmTypeReference getJvmType(final EClassifier classifier, final EObject context) {
    JvmTypeReference _xblockexpression = null;
    {
      boolean _notEquals = (!Objects.equal(classifier, null));
      if (_notEquals) {
        final String className = this.metamodelProvider.getQualifiedClassName(classifier, context);
        boolean _isNullOrEmpty = Strings.isNullOrEmpty(className);
        boolean _not = (!_isNullOrEmpty);
        if (_not) {
          return this.getTypeReferenceForTypeName(className, context);
        }
      }
      Class<?> _xifexpression = null;
      if ((classifier instanceof EClass)) {
        _xifexpression = EObject.class;
      } else {
        _xifexpression = Object.class;
      }
      final Class<?> clazz = _xifexpression;
      _xblockexpression = this.typeReferences.getTypeForName(clazz, context);
    }
    return _xblockexpression;
  }
  
  private JvmTypeReference getTypeReferenceForTypeName(final String typeName, final EObject context) {
    final JvmTypeReference typeRef = this.typeReferences.getTypeForName(typeName, context);
    final JvmTypeReference typeReference = this.primitives.asWrapperTypeIfPrimitive(typeRef);
    boolean _equals = Objects.equal(typeReference, null);
    if (_equals) {
      EObject errorContext = context;
      String contextName = context.toString();
      if ((((context instanceof Variable) && (((Variable) context).eContainer() instanceof PatternBody)) && 
        (((Variable) context).getReferences().size() > 0))) {
        String _name = ((Variable) context).getName();
        contextName = _name;
        EList<VariableReference> _references = ((Variable) context).getReferences();
        VariableReference _get = _references.get(0);
        errorContext = _get;
      }
      String _format = String.format(
        "Cannot resolve corresponding Java type for variable %s. Are the required bundle dependencies set?", contextName);
      this.errorFeedback.reportError(errorContext, _format, EMFPatternLanguageJvmModelInferrer.INVALID_TYPEREF_CODE, Severity.WARNING, 
        IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
    }
    return typeReference;
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> minimizeTypeInformation(final Set<IInputKey> types, final boolean mergeWithSupertypes) {
    int _size = types.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      return types;
    }
    Iterable<JavaTransitiveInstancesKey> _filter = Iterables.<JavaTransitiveInstancesKey>filter(types, JavaTransitiveInstancesKey.class);
    Iterable<JavaTransitiveInstancesKey> _filterNull = IterableExtensions.<JavaTransitiveInstancesKey>filterNull(_filter);
    final Iterable<JavaTransitiveInstancesKey> eJavaTypes = this.minimizeJavaTypeList(_filterNull);
    Iterable<EDataTypeInSlotsKey> _filter_1 = Iterables.<EDataTypeInSlotsKey>filter(types, EDataTypeInSlotsKey.class);
    Iterable<EDataTypeInSlotsKey> _filterNull_1 = IterableExtensions.<EDataTypeInSlotsKey>filterNull(_filter_1);
    Iterable<EDataTypeInSlotsKey> _minimizeEDataTypeList = this.minimizeEDataTypeList(_filterNull_1);
    final Function1<EDataTypeInSlotsKey, Boolean> _function = new Function1<EDataTypeInSlotsKey, Boolean>() {
      @Override
      public Boolean apply(final EDataTypeInSlotsKey it) {
        boolean _xblockexpression = false;
        {
          EDataType _emfKey = it.getEmfKey();
          Class<?> _instanceClass = _emfKey.getInstanceClass();
          Class<?> _wrapperClassForType = AbstractTypeSystem.getWrapperClassForType(_instanceClass);
          final JavaTransitiveInstancesKey javaType = new JavaTransitiveInstancesKey(_wrapperClassForType);
          final Function1<JavaTransitiveInstancesKey, Boolean> _function = new Function1<JavaTransitiveInstancesKey, Boolean>() {
            @Override
            public Boolean apply(final JavaTransitiveInstancesKey it) {
              return Boolean.valueOf(Objects.equal(it, javaType));
            }
          };
          boolean _exists = IterableExtensions.<JavaTransitiveInstancesKey>exists(eJavaTypes, _function);
          _xblockexpression = (!_exists);
        }
        return Boolean.valueOf(_xblockexpression);
      }
    };
    final Iterable<EDataTypeInSlotsKey> eDataTypes = IterableExtensions.<EDataTypeInSlotsKey>filter(_minimizeEDataTypeList, _function);
    Iterable<EClassTransitiveInstancesKey> _filter_2 = Iterables.<EClassTransitiveInstancesKey>filter(types, EClassTransitiveInstancesKey.class);
    Iterable<EClassTransitiveInstancesKey> _filterNull_2 = IterableExtensions.<EClassTransitiveInstancesKey>filterNull(_filter_2);
    final Iterable<EClassTransitiveInstancesKey> eClassTypes = this.minimizeEClassKeyList(_filterNull_2, mergeWithSupertypes);
    Iterable<IInputKey> _concat = Iterables.<IInputKey>concat(eDataTypes, eJavaTypes);
    Iterable<IInputKey> _concat_1 = Iterables.<IInputKey>concat(eClassTypes, _concat);
    return IterableExtensions.<IInputKey>toSet(_concat_1);
  }
  
  private Iterable<EClassTransitiveInstancesKey> minimizeEClassKeyList(final Iterable<EClassTransitiveInstancesKey> types, final boolean mergeWithSupertypes) {
    final Function1<EClassTransitiveInstancesKey, EClass> _function = new Function1<EClassTransitiveInstancesKey, EClass>() {
      @Override
      public EClass apply(final EClassTransitiveInstancesKey it) {
        return it.getEmfKey();
      }
    };
    Iterable<EClass> _map = IterableExtensions.<EClassTransitiveInstancesKey, EClass>map(types, _function);
    final HashSet<EClass> emfTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(_map, EClass.class)));
    Iterable<EClass> _minimizeEClassList = this.minimizeEClassList(emfTypes, mergeWithSupertypes);
    final Function1<EClass, EClassTransitiveInstancesKey> _function_1 = new Function1<EClass, EClassTransitiveInstancesKey>() {
      @Override
      public EClassTransitiveInstancesKey apply(final EClass it) {
        return new EClassTransitiveInstancesKey(it);
      }
    };
    return IterableExtensions.<EClass, EClassTransitiveInstancesKey>map(_minimizeEClassList, _function_1);
  }
  
  private Iterable<EClass> minimizeEClassList(final Iterable<EClass> types, final boolean mergeWithSupertypes) {
    final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
      @Override
      public Boolean apply(final EClass it) {
        return Boolean.valueOf(((!Objects.equal("EObject", it.getName())) || (!Objects.equal(it.getEPackage().getNsURI(), EcorePackage.eNS_URI))));
      }
    };
    final Iterable<EClass> nonTopTypes = IterableExtensions.<EClass>filter(types, _function);
    final HashSet<EClass> emfTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(nonTopTypes, EClass.class)));
    final Procedure1<EClass> _function_1 = new Procedure1<EClass>() {
      @Override
      public void apply(final EClass key) {
        EList<EClass> _eAllSuperTypes = key.getEAllSuperTypes();
        emfTypes.removeAll(_eAllSuperTypes);
      }
    };
    IterableExtensions.<EClass>forEach(nonTopTypes, _function_1);
    if ((mergeWithSupertypes && (emfTypes.size() > 1))) {
      final Function1<EClass, Iterable<EClass>> _function_2 = new Function1<EClass, Iterable<EClass>>() {
        @Override
        public Iterable<EClass> apply(final EClass key) {
          final Function1<EClass, EClass> _function = new Function1<EClass, EClass>() {
            @Override
            public EClass apply(final EClass current) {
              EClass _xblockexpression = null;
              {
                final EClassifier type = EcoreUtil2.getCompatibleType(key, current, null);
                EClass _xifexpression = null;
                if ((type instanceof EClass)) {
                  _xifexpression = ((EClass) type);
                } else {
                  _xifexpression = current;
                }
                _xblockexpression = _xifexpression;
              }
              return _xblockexpression;
            }
          };
          return IterableExtensions.<EClass, EClass>map(emfTypes, _function);
        }
      };
      Iterable<Iterable<EClass>> _map = IterableExtensions.<EClass, Iterable<EClass>>map(emfTypes, _function_2);
      final Iterable<EClass> compatibleTypes = Iterables.<EClass>concat(_map);
      final Function1<EClass, Boolean> _function_3 = new Function1<EClass, Boolean>() {
        @Override
        public Boolean apply(final EClass it) {
          Collection<EClass> _allSuperTypes = EcoreUtil2.getAllSuperTypes(it);
          final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
            @Override
            public Boolean apply(final EClass supertype) {
              final Function1<EClass, Boolean> _function = new Function1<EClass, Boolean>() {
                @Override
                public Boolean apply(final EClass it) {
                  return Boolean.valueOf(Objects.equal(it, supertype));
                }
              };
              return Boolean.valueOf(IterableExtensions.<EClass>exists(compatibleTypes, _function));
            }
          };
          boolean _exists = IterableExtensions.<EClass>exists(_allSuperTypes, _function);
          return Boolean.valueOf((!_exists));
        }
      };
      Iterable<EClass> _filter = IterableExtensions.<EClass>filter(compatibleTypes, _function_3);
      final Set<EClass> filteredTypes = IterableExtensions.<EClass>toSet(_filter);
      int _size = filteredTypes.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        return CollectionLiterals.<EClass>newHashSet(EcorePackage.Literals.EOBJECT);
      } else {
        return filteredTypes;
      }
    }
    return emfTypes;
  }
  
  private Iterable<EDataTypeInSlotsKey> minimizeEDataTypeList(final Iterable<EDataTypeInSlotsKey> types) {
    final Function1<EDataTypeInSlotsKey, EDataType> _function = new Function1<EDataTypeInSlotsKey, EDataType>() {
      @Override
      public EDataType apply(final EDataTypeInSlotsKey it) {
        return it.getEmfKey();
      }
    };
    Iterable<EDataType> _map = IterableExtensions.<EDataTypeInSlotsKey, EDataType>map(types, _function);
    final Function1<EDataType, Boolean> _function_1 = new Function1<EDataType, Boolean>() {
      @Override
      public Boolean apply(final EDataType it) {
        String _instanceClassName = it.getInstanceClassName();
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(_instanceClassName);
        return Boolean.valueOf((!_isNullOrEmpty));
      }
    };
    Iterable<EDataType> _filter = IterableExtensions.<EDataType>filter(_map, _function_1);
    final HashSet<EDataType> emfTypes = CollectionLiterals.<EDataType>newHashSet(((EDataType[])Conversions.unwrapArray(_filter, EDataType.class)));
    Object _clone = emfTypes.clone();
    final HashSet<EDataType> result = ((HashSet<EDataType>) _clone);
    final Iterator<EDataType> it = emfTypes.iterator();
    while (it.hasNext()) {
      {
        final EDataType dataType = it.next();
        final Function1<EDataType, Boolean> _function_2 = new Function1<EDataType, Boolean>() {
          @Override
          public Boolean apply(final EDataType it) {
            String _instanceClassName = it.getInstanceClassName();
            String _instanceClassName_1 = dataType.getInstanceClassName();
            return Boolean.valueOf(Objects.equal(_instanceClassName, _instanceClassName_1));
          }
        };
        Iterable<EDataType> _filter_1 = IterableExtensions.<EDataType>filter(emfTypes, _function_2);
        Iterable<EDataType> _drop = IterableExtensions.<EDataType>drop(_filter_1, 1);
        CollectionExtensions.<EDataType>removeAll(result, _drop);
      }
    }
    final Function1<EDataType, EDataTypeInSlotsKey> _function_2 = new Function1<EDataType, EDataTypeInSlotsKey>() {
      @Override
      public EDataTypeInSlotsKey apply(final EDataType it) {
        return new EDataTypeInSlotsKey(it);
      }
    };
    return IterableExtensions.<EDataType, EDataTypeInSlotsKey>map(result, _function_2);
  }
  
  private Iterable<JavaTransitiveInstancesKey> minimizeJavaTypeList(final Iterable<JavaTransitiveInstancesKey> types) {
    final Function1<JavaTransitiveInstancesKey, Boolean> _function = new Function1<JavaTransitiveInstancesKey, Boolean>() {
      @Override
      public Boolean apply(final JavaTransitiveInstancesKey it) {
        return Boolean.valueOf(((!Objects.equal(it.getInstanceClass(), null)) && (!Objects.equal(it.getInstanceClass(), Object.class))));
      }
    };
    Iterable<JavaTransitiveInstancesKey> _filter = IterableExtensions.<JavaTransitiveInstancesKey>filter(types, _function);
    final Function1<JavaTransitiveInstancesKey, Class<?>> _function_1 = new Function1<JavaTransitiveInstancesKey, Class<?>>() {
      @Override
      public Class<?> apply(final JavaTransitiveInstancesKey it) {
        return it.getInstanceClass();
      }
    };
    Iterable<Class<?>> _map = IterableExtensions.<JavaTransitiveInstancesKey, Class<?>>map(_filter, _function_1);
    final Iterable<Class<?>> nonTopTypes = IterableExtensions.<Class<?>>filterNull(_map);
    final HashSet<Class<?>> javaTypes = CollectionLiterals.<Class<?>>newHashSet(((Class<?>[])Conversions.unwrapArray(nonTopTypes, Class.class)));
    final Procedure1<Class<?>> _function_2 = new Procedure1<Class<?>>() {
      @Override
      public void apply(final Class<?> key) {
        final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
          @Override
          public Boolean apply(final Class<?> it) {
            return Boolean.valueOf(((!Objects.equal(it, key)) && key.isAssignableFrom(it)));
          }
        };
        Iterable<Class<?>> _filter = IterableExtensions.<Class<?>>filter(javaTypes, _function);
        CollectionExtensions.<Class<?>>removeAll(javaTypes, _filter);
      }
    };
    IterableExtensions.<Class<?>>forEach(nonTopTypes, _function_2);
    final Function1<Class<?>, JavaTransitiveInstancesKey> _function_3 = new Function1<Class<?>, JavaTransitiveInstancesKey>() {
      @Override
      public JavaTransitiveInstancesKey apply(final Class<?> it) {
        return new JavaTransitiveInstancesKey(it);
      }
    };
    return IterableExtensions.<Class<?>, JavaTransitiveInstancesKey>map(javaTypes, _function_3);
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> addTypeInformation(final Set<IInputKey> types, final IInputKey newType) {
    ImmutableSet.Builder<IInputKey> _builder = ImmutableSet.<IInputKey>builder();
    ImmutableSet.Builder<IInputKey> _addAll = _builder.addAll(types);
    ImmutableSet.Builder<IInputKey> _add = _addAll.add(newType);
    ImmutableSet<IInputKey> _build = _add.build();
    return this.minimizeTypeInformation(_build, false);
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> addTypeInformation(final Set<IInputKey> types, final Set<IInputKey> newTypes) {
    ImmutableSet.Builder<IInputKey> _builder = ImmutableSet.<IInputKey>builder();
    ImmutableSet.Builder<IInputKey> _addAll = _builder.addAll(types);
    ImmutableSet.Builder<IInputKey> _addAll_1 = _addAll.addAll(newTypes);
    ImmutableSet<IInputKey> _build = _addAll_1.build();
    return this.minimizeTypeInformation(_build, 
      false);
  }
  
  /**
   * @since 1.3
   * @return True if the given classifiers has a common subtype in the selected EPackages.
   */
  public boolean hasCommonSubtype(final Set<IInputKey> typeKeys, final Iterable<EPackage> ePackages) {
    boolean _xblockexpression = false;
    {
      final Function1<EPackage, Set<EClass>> _function = new Function1<EPackage, Set<EClass>>() {
        @Override
        public Set<EClass> apply(final EPackage it) {
          return EMFTypeSystem.getAllEClassifiers(it);
        }
      };
      Iterable<Set<EClass>> _map = IterableExtensions.<EPackage, Set<EClass>>map(ePackages, _function);
      final Iterable<EClass> knownTypes = Iterables.<EClass>concat(_map);
      boolean _xifexpression = false;
      final Function1<IInputKey, Boolean> _function_1 = new Function1<IInputKey, Boolean>() {
        @Override
        public Boolean apply(final IInputKey it) {
          return Boolean.valueOf((it instanceof EClassTransitiveInstancesKey));
        }
      };
      boolean _forall = IterableExtensions.<IInputKey>forall(typeKeys, _function_1);
      if (_forall) {
        boolean _xblockexpression_1 = false;
        {
          final Function1<IInputKey, EClass> _function_2 = new Function1<IInputKey, EClass>() {
            @Override
            public EClass apply(final IInputKey it) {
              return ((EClassTransitiveInstancesKey) it).getEmfKey();
            }
          };
          Iterable<EClass> _map_1 = IterableExtensions.<IInputKey, EClass>map(typeKeys, _function_2);
          final Set<EClass> classifiers = IterableExtensions.<EClass>toSet(_map_1);
          final Function1<EClass, Boolean> _function_3 = new Function1<EClass, Boolean>() {
            @Override
            public Boolean apply(final EClass it) {
              EList<EClass> _eAllSuperTypes = it.getEAllSuperTypes();
              return Boolean.valueOf(_eAllSuperTypes.containsAll(classifiers));
            }
          };
          _xblockexpression_1 = IterableExtensions.<EClass>exists(knownTypes, _function_3);
        }
        _xifexpression = _xblockexpression_1;
      } else {
        return false;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private static Set<EClass> getAllEClassifiers(final EPackage ePackage) {
    EList<EClassifier> _eClassifiers = ePackage.getEClassifiers();
    Iterable<EClass> _filter = Iterables.<EClass>filter(_eClassifiers, EClass.class);
    return CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(_filter, EClass.class)));
  }
  
  @Override
  public String typeString(final IInputKey type) {
    String _switchResult = null;
    boolean _matched = false;
    boolean _equals = Objects.equal(type, null);
    if (_equals) {
      _matched=true;
      _switchResult = "«null»";
    }
    if (!_matched) {
      if (((type instanceof EClassTransitiveInstancesKey) && (!((EClassTransitiveInstancesKey) type).getEmfKey().eIsProxy()))) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        EClass _emfKey = ((EClassTransitiveInstancesKey) type).getEmfKey();
        EPackage _ePackage = _emfKey.getEPackage();
        String _nsURI = _ePackage.getNsURI();
        _builder.append(_nsURI, "");
        _builder.append("::");
        EClass _emfKey_1 = ((EClassTransitiveInstancesKey) type).getEmfKey();
        String _name = _emfKey_1.getName();
        _builder.append(_name, "");
        _switchResult = _builder.toString();
      }
    }
    if (!_matched) {
      if (type instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        StringConcatenation _builder_1 = new StringConcatenation();
        EClass _emfKey_2 = ((EClassTransitiveInstancesKey)type).getEmfKey();
        String _string = _emfKey_2.toString();
        _builder_1.append(_string, "");
        _switchResult = _builder_1.toString();
      }
    }
    if (!_matched) {
      if (type instanceof EDataTypeInSlotsKey) {
        _matched=true;
        StringConcatenation _builder_1 = new StringConcatenation();
        EDataType _emfKey_2 = ((EDataTypeInSlotsKey)type).getEmfKey();
        EPackage _ePackage_1 = _emfKey_2.getEPackage();
        String _nsURI_1 = _ePackage_1.getNsURI();
        _builder_1.append(_nsURI_1, "");
        _builder_1.append("::");
        EDataType _emfKey_3 = ((EDataTypeInSlotsKey)type).getEmfKey();
        String _name_1 = _emfKey_3.getName();
        _builder_1.append(_name_1, "");
        _switchResult = _builder_1.toString();
      }
    }
    if (!_matched) {
      _switchResult = super.typeString(type);
    }
    return _switchResult;
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> getCompatibleSupertypes(final Set<IInputKey> types) {
    Set<IInputKey> _xifexpression = null;
    final Function1<IInputKey, Boolean> _function = new Function1<IInputKey, Boolean>() {
      @Override
      public Boolean apply(final IInputKey it) {
        return Boolean.valueOf((it instanceof EClassTransitiveInstancesKey));
      }
    };
    boolean _forall = IterableExtensions.<IInputKey>forall(types, _function);
    if (_forall) {
      Iterable<EClassTransitiveInstancesKey> _filter = Iterables.<EClassTransitiveInstancesKey>filter(types, EClassTransitiveInstancesKey.class);
      _xifexpression = this.getCompatibleEClasses(_filter);
    } else {
      final Function1<IInputKey, Boolean> _function_1 = new Function1<IInputKey, Boolean>() {
        @Override
        public Boolean apply(final IInputKey it) {
          return Boolean.valueOf((it instanceof JavaTransitiveInstancesKey));
        }
      };
      boolean _forall_1 = IterableExtensions.<IInputKey>forall(types, _function_1);
      if (_forall_1) {
        return types;
      } else {
        return types;
      }
    }
    return _xifexpression;
  }
  
  private Set<IInputKey> getCompatibleEClasses(final Iterable<EClassTransitiveInstancesKey> types) {
    final Function1<EClassTransitiveInstancesKey, List<EClass>> _function = new Function1<EClassTransitiveInstancesKey, List<EClass>>() {
      @Override
      public List<EClass> apply(final EClassTransitiveInstancesKey it) {
        EClass _emfKey = it.getEmfKey();
        return EcoreUtil2.getCompatibleTypesOf(_emfKey);
      }
    };
    final Iterable<List<EClass>> candidates = IterableExtensions.<EClassTransitiveInstancesKey, List<EClass>>map(types, _function);
    final Iterator<List<EClass>> iterator = candidates.iterator();
    boolean _hasNext = iterator.hasNext();
    if (_hasNext) {
      List<EClass> _next = iterator.next();
      final Set<EClass> compatibleTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(_next, EClass.class)));
      while (iterator.hasNext()) {
        List<EClass> _next_1 = iterator.next();
        compatibleTypes.retainAll(_next_1);
      }
      final Function1<EClass, IInputKey> _function_1 = new Function1<EClass, IInputKey>() {
        @Override
        public IInputKey apply(final EClass it) {
          EClassTransitiveInstancesKey _eClassTransitiveInstancesKey = new EClassTransitiveInstancesKey(it);
          return ((IInputKey) _eClassTransitiveInstancesKey);
        }
      };
      Iterable<IInputKey> _map = IterableExtensions.<EClass, IInputKey>map(compatibleTypes, _function_1);
      return IterableExtensions.<IInputKey>toSet(_map);
    }
    return Collections.<IInputKey>unmodifiableSet(CollectionLiterals.<IInputKey>newHashSet());
  }
  
  @Override
  public boolean isValidType(final Type type) {
    boolean _xblockexpression = false;
    {
      if ((type instanceof ClassType)) {
        final EClassifier classifier = ((ClassType)type).getClassname();
        return ((!Objects.equal(classifier, null)) && (!classifier.eIsProxy()));
      } else {
        if ((type instanceof ReferenceType)) {
          final EStructuralFeature feature = ((ReferenceType)type).getRefname();
          return ((!Objects.equal(feature, null)) && (!feature.eIsProxy()));
        }
      }
      _xblockexpression = super.isValidType(type);
    }
    return _xblockexpression;
  }
}
