/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Mark Czotter - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.emf.util;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.viatra.query.patternlanguage.emf.services.EMFPatternLanguageGrammarAccess;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.validation.EMFIssueCodes;
import org.eclipse.viatra.query.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Pattern;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternBody;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternModel;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Variable;
import org.eclipse.viatra.query.patternlanguage.typing.ITypeInferrer;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFQuerySpecification;
import org.eclipse.viatra.query.runtime.emf.types.EClassTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EDataTypeInSlotsKey;
import org.eclipse.viatra.query.runtime.emf.types.EStructuralFeatureInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.TerminalRule;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Utility class for the EMFPatternLanguageJvmModelInferrer.
 * 
 * @author Mark Czotter
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrerUtil {
  @Inject
  @Extension
  private TypeReferences _typeReferences;
  
  @Inject
  private Logger logger;
  
  @Inject
  private ITypeInferrer typeInferrer;
  
  @Inject
  private IJvmModelAssociations associations;
  
  @Inject
  private EMFPatternLanguageGrammarAccess grammar;
  
  @Inject
  private IErrorFeedback feedback;
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final Pattern pattern) {
    String name = pattern.getName();
    boolean _contains = name.contains(".");
    if (_contains) {
      int _lastIndexOf = name.lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      return name.substring(_plus);
    }
    return name;
  }
  
  public boolean validClassName(final String simpleName) {
    return (Character.isJavaIdentifierStart(simpleName.charAt(0)) && 
      IterableExtensions.<Character>forall(((Iterable<Character>)Conversions.doWrapArray(simpleName.toCharArray())), new Function1<Character, Boolean>() {
        @Override
        public Boolean apply(final Character it) {
          return Boolean.valueOf(Character.isJavaIdentifierPart((it).charValue()));
        }
      }));
  }
  
  public String modelFileName(final EObject object) {
    final Resource eResource = object.eResource();
    boolean _notEquals = (!Objects.equal(eResource, null));
    if (_notEquals) {
      URI _uRI = eResource.getURI();
      URI _trimFileExtension = _uRI.trimFileExtension();
      final String name = _trimFileExtension.lastSegment();
      boolean _validClassName = this.validClassName(name);
      boolean _not = (!_validClassName);
      if (_not) {
        String _format = String.format("The file name %s is not a valid Java type name. Please, rename the file!", name);
        this.feedback.reportErrorNoLocation(object, _format, 
          EMFIssueCodes.OTHER_ISSUE, Severity.ERROR, IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
      }
      return name;
    } else {
      return "";
    }
  }
  
  /**
   * Returns the QuerySpecificationClass name based on the Pattern's name
   */
  public String querySpecificationClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "QuerySpecification");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the IQuerySpecificationProvider class name based on the Pattern's name
   */
  public String querySpecificationProviderClassName(final Pattern pattern) {
    return "Provider";
  }
  
  /**
   * Returns the holder class name based on the Pattern's name
   */
  public String querySpecificationHolderClassName(final Pattern pattern) {
    return "LazyHolder";
  }
  
  /**
   * Returns the PQuery class name based on the Pattern's name
   */
  public String querySpecificationPQueryClassName(final Pattern pattern) {
    return "GeneratedPQuery";
  }
  
  /**
   * Returns the MatcherClass name based on the Pattern's name
   */
  public String matcherClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Matcher");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the MatchClass name based on the Pattern's name
   */
  public String matchClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Match");
    }
    return _xblockexpression;
  }
  
  public String matchImmutableInnerClassName(final Pattern pattern) {
    return "Immutable";
  }
  
  public String matchMutableInnerClassName(final Pattern pattern) {
    return "Mutable";
  }
  
  /**
   * Returns the ProcessorClass name based on the Pattern's name
   */
  public String processorClassName(final Pattern pattern) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        String _realPatternName = this.realPatternName(pattern);
        name = _realPatternName;
      }
      String _firstUpper = StringExtensions.toFirstUpper(name);
      _xblockexpression = (_firstUpper + "Processor");
    }
    return _xblockexpression;
  }
  
  /**
   * Returns field name for Variable
   */
  public String fieldName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = StringExtensions.toFirstUpper(_name);
    return ("f" + _firstUpper);
  }
  
  /**
   * Returns parameter name for Variable
   */
  public String parameterName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("p" + _firstUpper);
  }
  
  public String positionConstant(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _upperCase = null;
    if (_name!=null) {
      _upperCase=_name.toUpperCase();
    }
    return ("POSITION_" + _upperCase);
  }
  
  /**
   * Returns correct getter method name for variable.
   * For variable with name 'class' returns getValueOfClass, otherwise returns <code>get#variable.name.toFirstUpper#</code>.
   */
  public String getterMethodName(final Variable variable) {
    String _name = variable.getName();
    boolean _equals = Objects.equal(_name, "class");
    if (_equals) {
      return "getValueOfClass";
    } else {
      String _name_1 = null;
      if (variable!=null) {
        _name_1=variable.getName();
      }
      String _firstUpper = null;
      if (_name_1!=null) {
        _firstUpper=StringExtensions.toFirstUpper(_name_1);
      }
      return ("get" + _firstUpper);
    }
  }
  
  /**
   * Returns correct setter method name for variable.
   * Currently returns <code>set#variable.name.toFirstUpper#</code>.
   */
  public String setterMethodName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("set" + _firstUpper);
  }
  
  /**
   * Calls the typeProvider.
   * @return JvmTypeReference pointing the EClass that defines the Variable's type.
   * @see ITypeInferrer
   */
  public JvmTypeReference calculateType(final Variable variable) {
    return this.typeInferrer.getJvmType(variable, variable);
  }
  
  /**
   * Serializes the EObject into Java String variable.
   */
  public CharSequence serializeToJava(final EObject eObject) {
    final String parseString = this.serialize(eObject);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(parseString);
    if (_isNullOrEmpty) {
      return "";
    }
    final String[] splits = parseString.split("[\r\n]+");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("String patternString = \"\"");
    final StringConcatenation stringRep = ((StringConcatenation) _builder);
    stringRep.newLine();
    for (final String s : splits) {
      {
        stringRep.append((("+\" " + s) + " \""));
        stringRep.newLine();
      }
    }
    stringRep.append(";");
    return stringRep;
  }
  
  /**
   * Serializes the input for Javadoc
   */
  public String serializeToJavadoc(final Pattern pattern) {
    String javadocString = this.serialize(pattern);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(javadocString);
    if (_isNullOrEmpty) {
      return "Serialization error, check Log";
    }
    String _quote = java.util.regex.Pattern.quote("\\\"");
    String _quoteReplacement = Matcher.quoteReplacement("\"");
    String _replaceAll = javadocString.replaceAll(_quote, _quoteReplacement);
    javadocString = _replaceAll;
    String _replaceAll_1 = javadocString.replaceAll("@", "{@literal @}");
    javadocString = _replaceAll_1;
    String _replaceAll_2 = javadocString.replaceAll("<", "{@literal <}");
    javadocString = _replaceAll_2;
    String _replaceAll_3 = javadocString.replaceAll(">", "{@literal >}");
    javadocString = _replaceAll_3;
    return javadocString.trim();
  }
  
  /**
   * Returns the file header comment at the beginning of the text corresponding
   * to the pattern model.
   * The comment text is escaped, so it does not include stars in multi-line comments.
   * 
   * @since 1.3
   */
  public String getFileComment(final PatternModel patternModel) {
    final ICompositeNode patternNode = NodeModelUtils.getNode(patternModel);
    INode _firstChild = null;
    if (patternNode!=null) {
      _firstChild=patternNode.getFirstChild();
    }
    INode _nextSibling = null;
    if (_firstChild!=null) {
      _nextSibling=_firstChild.getNextSibling();
    }
    final INode possibleFileComment = _nextSibling;
    boolean _notEquals = (!Objects.equal(possibleFileComment, null));
    if (_notEquals) {
      final EObject grammarElement = possibleFileComment.getGrammarElement();
      TerminalRule _mL_COMMENTRule = this.grammar.getML_COMMENTRule();
      boolean _equals = Objects.equal(grammarElement, _mL_COMMENTRule);
      if (_equals) {
        String _text = possibleFileComment.getText();
        final String multiLineCommentText = this.escape(_text);
        return multiLineCommentText;
      } else {
        TerminalRule _sL_COMMENTRule = this.grammar.getSL_COMMENTRule();
        boolean _equals_1 = Objects.equal(grammarElement, _sL_COMMENTRule);
        if (_equals_1) {
          String _text_1 = possibleFileComment.getText();
          final String singleLineCommentText = this.escape(_text_1);
          return singleLineCommentText;
        }
      }
    }
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Generated from ");
    Resource _eResource = patternModel.eResource();
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    _builder.append(_uRI, "");
    return _builder.toString();
  }
  
  /**
   * Returns the file header comment at the beginning of the text corresponding
   * to the pattern model containing the given pattern.
   * The comment text is escaped, so it does not include stars in multi-line comments.
   * 
   * @since 1.3
   */
  public String getFileComment(final Pattern pattern) {
    final PatternModel patternModel = EcoreUtil2.<PatternModel>getContainerOfType(pattern, PatternModel.class);
    return this.getFileComment(patternModel);
  }
  
  /**
   * Escapes the input to be usable in literal strings
   */
  public String escapeToQuotedString(final String inputString) {
    String string = inputString;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(string);
    if (_isNullOrEmpty) {
      return "";
    }
    String _replace = string.replace("\\", "\\\\");
    string = _replace;
    String _replace_1 = string.replace("\n", "\\n");
    string = _replace_1;
    String _replace_2 = string.replace("\t", "\\t");
    string = _replace_2;
    String _replace_3 = string.replace("\"", "\\\"");
    string = _replace_3;
    return string.trim();
  }
  
  /**
   * Serializes EObject to a String representation. Escapes only the double qoutes.
   */
  private String serialize(final EObject eObject) {
    try {
      final ICompositeNode eObjectNode = NodeModelUtils.getNode(eObject);
      boolean _notEquals = (!Objects.equal(eObjectNode, null));
      if (_notEquals) {
        String _text = eObjectNode.getText();
        return this.escape(_text);
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        boolean _notEquals_1 = (!Objects.equal(this.logger, null));
        if (_notEquals_1) {
          EClass _eClass = eObject.eClass();
          String _name = _eClass.getName();
          String _plus = ("Error when serializing " + _name);
          this.logger.error(_plus, e);
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  private String escape(final String escapable) {
    boolean _equals = Objects.equal(escapable, null);
    if (_equals) {
      return null;
    }
    String escapedString = escapable.replaceAll("\"", "\\\\\"");
    String _replaceAll = escapedString.replaceAll("\\*+/", "");
    String _replaceAll_1 = _replaceAll.replaceAll("/*\\*", "");
    escapedString = _replaceAll_1;
    return escapedString;
  }
  
  /**
   * Returns the packageName: PatternModel.packageName or "" when nullOrEmpty.
   */
  public String getPackageName(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    }
    return packageName.toLowerCase();
  }
  
  public String getUtilPackageName(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return (_packageName + ".util");
  }
  
  /**
   * Returns the packageName: PatternModel.packageName + Pattern.name, packageName is ignored, when nullOrEmpty.
   */
  public String getPackageNameOld(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    } else {
      packageName = (packageName + ".");
    }
    String _name = pattern.getName();
    String _plus = (packageName + _name);
    return _plus.toLowerCase();
  }
  
  public String getPackagePath(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return _packageName.replace(".", "/");
  }
  
  /**
   * Calculates the correct package path for a selected fqn
   */
  public String getPackagePath(final String fqn) {
    String _xblockexpression = null;
    {
      Splitter _on = Splitter.on(".");
      final Iterable<String> split = _on.split(fqn);
      int _size = IterableExtensions.size(split);
      int _minus = (_size - 1);
      Iterable<String> _take = IterableExtensions.<String>take(split, _minus);
      _xblockexpression = IterableExtensions.join(_take, "/");
    }
    return _xblockexpression;
  }
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final String fqn) {
    Splitter _on = Splitter.on(".");
    Iterable<String> _split = _on.split(fqn);
    return IterableExtensions.<String>last(_split);
  }
  
  public JvmType findInferredSpecification(final Pattern pattern) {
    return this.findInferredClass(pattern, BaseGeneratedEMFQuerySpecification.class);
  }
  
  public JvmType findInferredClass(final EObject pattern, final Class<?> clazz) {
    Set<EObject> _jvmElements = this.associations.getJvmElements(pattern);
    Iterable<JvmType> _filter = Iterables.<JvmType>filter(_jvmElements, JvmType.class);
    final Function1<JvmType, Boolean> _function = new Function1<JvmType, Boolean>() {
      @Override
      public Boolean apply(final JvmType it) {
        return Boolean.valueOf(EMFPatternLanguageJvmModelInferrerUtil.this.isCompatibleWith(it, clazz));
      }
    };
    return IterableExtensions.<JvmType>findFirst(_filter, _function);
  }
  
  public boolean isCompatibleWith(final JvmType type, final Class<?> clazz) {
    return (this._typeReferences.is(type, clazz) || 
      ((type instanceof JvmDeclaredType) && IterableExtensions.<JvmTypeReference>exists(((JvmDeclaredType) type).getSuperTypes(), new Function1<JvmTypeReference, Boolean>() {
        @Override
        public Boolean apply(final JvmTypeReference it) {
          return Boolean.valueOf(EMFPatternLanguageJvmModelInferrerUtil.this._typeReferences.is(it, clazz));
        }
      })));
  }
  
  public boolean isPublic(final Pattern pattern) {
    boolean _isPrivate = CorePatternLanguageHelper.isPrivate(pattern);
    return (!_isPrivate);
  }
  
  public List<Variable> variables(final XExpression ex) {
    List<Variable> _xblockexpression = null;
    {
      final PatternBody body = EcoreUtil2.<PatternBody>getContainerOfType(ex, PatternBody.class);
      TreeIterator<EObject> _eAllContents = ex.eAllContents();
      List<XExpression> _newImmutableList = CollectionLiterals.<XExpression>newImmutableList(ex);
      Iterator<XExpression> _iterator = _newImmutableList.iterator();
      Iterator<EObject> _plus = Iterators.<EObject>concat(_eAllContents, _iterator);
      Iterator<XFeatureCall> _filter = Iterators.<XFeatureCall>filter(_plus, XFeatureCall.class);
      final Function1<XFeatureCall, String> _function = new Function1<XFeatureCall, String>() {
        @Override
        public String apply(final XFeatureCall it) {
          return it.getConcreteSyntaxFeatureName();
        }
      };
      Iterator<String> _map = IteratorExtensions.<XFeatureCall, String>map(_filter, _function);
      final List<String> valNames = IteratorExtensions.<String>toList(_map);
      EList<Variable> _variables = body.getVariables();
      final Function1<Variable, Boolean> _function_1 = new Function1<Variable, Boolean>() {
        @Override
        public Boolean apply(final Variable it) {
          String _name = it.getName();
          return Boolean.valueOf(valNames.contains(_name));
        }
      };
      Iterable<Variable> _filter_1 = IterableExtensions.<Variable>filter(_variables, _function_1);
      final Function1<Variable, String> _function_2 = new Function1<Variable, String>() {
        @Override
        public String apply(final Variable it) {
          return it.getName();
        }
      };
      _xblockexpression = IterableExtensions.<Variable, String>sortBy(_filter_1, _function_2);
    }
    return _xblockexpression;
  }
  
  public String expressionMethodName(final XExpression ex) {
    String _expressionPostfix = EMFPatternLanguageJvmModelInferrerUtil.getExpressionPostfix(ex);
    return ("evaluateExpression_" + _expressionPostfix);
  }
  
  private static String getExpressionPostfix(final XExpression xExpression) {
    final Pattern pattern = EcoreUtil2.<Pattern>getContainerOfType(xExpression, Pattern.class);
    boolean _notEquals = (!Objects.equal(pattern, null));
    Preconditions.checkArgument(_notEquals, "Expression is not inside a pattern");
    int bodyNo = 0;
    EList<PatternBody> _bodies = pattern.getBodies();
    for (final PatternBody patternBody : _bodies) {
      {
        bodyNo = (bodyNo + 1);
        int exNo = 0;
        Collection<XExpression> _allTopLevelXBaseExpressions = CorePatternLanguageHelper.getAllTopLevelXBaseExpressions(patternBody);
        for (final XExpression xExpression2 : _allTopLevelXBaseExpressions) {
          {
            exNo = (exNo + 1);
            boolean _equals = xExpression.equals(xExpression2);
            if (_equals) {
              String _plus = (Integer.valueOf(bodyNo) + "_");
              return (_plus + Integer.valueOf(exNo));
            }
          }
        }
      }
    }
    throw new RuntimeException("Expression not found in pattern");
  }
  
  /**
   * Output code is intended for generated query specification classes,
   *  since it depends on 'getFeatureLiteral()' / 'getClassifierLiteral()'
   * 
   * <p> the "safe" classifier lookup is used if the result is used for initializing a PParameter
   */
  public StringConcatenationClient serializeInputKey(final IInputKey key, final boolean forParameter) {
    return new StringConcatenationClient() {
      @Override
      protected void appendTo(final StringConcatenationClient.TargetStringConcatenation target) {
        EMFPatternLanguageJvmModelInferrerUtil.this.appendInputKey(target, key, forParameter);
      }
    };
  }
  
  /**
   * Calculates the name of the variable that stores a PParameter for a pattern
   * @since 1.4
   */
  public String getPParameterName(final Variable parameter) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("parameter_");
    String _parameterName = this.parameterName(parameter);
    _builder.append(_parameterName, "");
    return _builder.toString();
  }
  
  /**
   * Output code is intended for generated query specification classes,
   *  since it depends on 'getFeatureLiteral()' / 'getClassifierLiteral()'
   * 
   * <p> the "safe" classifier lookup is used if the result is used for initializing a PParameter
   */
  public void appendInputKey(final StringConcatenationClient.TargetStringConcatenation target, final IInputKey key, final boolean forParameter) {
    boolean _matched = false;
    if (key instanceof EStructuralFeatureInstancesKey) {
      _matched=true;
      final EStructuralFeature literal = ((EStructuralFeatureInstancesKey)key).getEmfKey();
      final EClass container = literal.getEContainingClass();
      EPackage _ePackage = container.getEPackage();
      final String packageNsUri = _ePackage.getNsURI();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("new ");
      target.append(_builder);
      target.append(EStructuralFeatureInstancesKey.class);
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("(getFeatureLiteral(\"");
      _builder_1.append(packageNsUri, "");
      _builder_1.append("\", \"");
      String _name = container.getName();
      _builder_1.append(_name, "");
      _builder_1.append("\", \"");
      String _name_1 = literal.getName();
      _builder_1.append(_name_1, "");
      _builder_1.append("\"))");
      target.append(_builder_1);
    }
    if (!_matched) {
      if (key instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        final EClass literal = ((EClassTransitiveInstancesKey)key).getEmfKey();
        EPackage _ePackage = literal.getEPackage();
        final String packageNsUri = _ePackage.getNsURI();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(EClassTransitiveInstancesKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("((");
        target.append(_builder_1);
        target.append(EClass.class);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(")");
        CharSequence _classifierGetterName = this.classifierGetterName(forParameter);
        _builder_2.append(_classifierGetterName, "");
        _builder_2.append("(\"");
        _builder_2.append(packageNsUri, "");
        _builder_2.append("\", \"");
        String _name = literal.getName();
        _builder_2.append(_name, "");
        _builder_2.append("\"))");
        target.append(_builder_2);
      }
    }
    if (!_matched) {
      if (key instanceof EDataTypeInSlotsKey) {
        _matched=true;
        final EDataType literal = ((EDataTypeInSlotsKey)key).getEmfKey();
        EPackage _ePackage = literal.getEPackage();
        final String packageNsUri = _ePackage.getNsURI();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(EDataTypeInSlotsKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("((");
        target.append(_builder_1);
        target.append(EDataType.class);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(")");
        CharSequence _classifierGetterName = this.classifierGetterName(forParameter);
        _builder_2.append(_classifierGetterName, "");
        _builder_2.append("(\"");
        _builder_2.append(packageNsUri, "");
        _builder_2.append("\", \"");
        String _name = literal.getName();
        _builder_2.append(_name, "");
        _builder_2.append("\"))");
        target.append(_builder_2);
      }
    }
    if (!_matched) {
      if (key instanceof JavaTransitiveInstancesKey) {
        _matched=true;
        final String clazz = ((JavaTransitiveInstancesKey)key).getPrettyPrintableName();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(JavaTransitiveInstancesKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("(");
        _builder_1.append(clazz, "");
        _builder_1.append(".class)");
        target.append(_builder_1);
      }
    }
    if (!_matched) {
      if (Objects.equal(key, null)) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("null");
        target.append(_builder);
      }
    }
  }
  
  private CharSequence classifierGetterName(final boolean forParameter) {
    CharSequence _xifexpression = null;
    if (forParameter) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("getClassifierLiteralSafe");
      _xifexpression = _builder;
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("getClassifierLiteral");
      _xifexpression = _builder_1;
    }
    return _xifexpression;
  }
}
