/**
 * Copyright (c) 2010-2014, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.typing;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.logging.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.BoolValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CheckConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CompareConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.CompareFeature;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.DoubleValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Expression;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.FunctionEvaluationValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.IntValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.JavaType;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.ListValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionHead;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PathExpressionTail;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Pattern;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternCall;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternCompositionConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.StringValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Type;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.TypeCheckConstraint;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.ValueReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Variable;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableValue;
import org.eclipse.viatra.query.patternlanguage.typing.ITypeSystem;
import org.eclipse.viatra.query.patternlanguage.typing.TypeInformation;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.ConditionalJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.ParameterTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.TypeConformJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.TypeJudgement;
import org.eclipse.viatra.query.patternlanguage.typing.judgements.XbaseExpressionTypeJudgement;
import org.eclipse.viatra.query.patternlanguage.util.AggregatorUtil;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.typesystem.IBatchTypeResolver;

/**
 * @author Zoltan Ujhelyi
 * @since 1.3
 */
@SuppressWarnings("all")
public class PatternLanguageTypeRules {
  @Inject
  private ITypeSystem typeSystem;
  
  @Inject
  private IBatchTypeResolver typeResolver;
  
  @Inject
  private Logger logger;
  
  protected void _inferTypes(final Pattern pattern, final TypeInformation information) {
    EList<Variable> _parameters = pattern.getParameters();
    final Procedure1<Variable> _function = new Procedure1<Variable>() {
      @Override
      public void apply(final Variable parameter) {
        Type _type = parameter.getType();
        boolean _isValidType = PatternLanguageTypeRules.this.typeSystem.isValidType(_type);
        if (_isValidType) {
          Type _type_1 = parameter.getType();
          final IInputKey typeKey = PatternLanguageTypeRules.this.typeSystem.extractTypeDescriptor(_type_1);
          information.declareType(parameter, typeKey);
        }
        Set<Variable> _localReferencesOfParameter = CorePatternLanguageHelper.getLocalReferencesOfParameter(parameter);
        final Procedure1<Variable> _function = new Procedure1<Variable>() {
          @Override
          public void apply(final Variable ref) {
            information.provideType(new TypeConformJudgement(ref, parameter) {
              @Override
              public Set<Expression> getDependingExpressions() {
                return Collections.<Expression>unmodifiableSet(CollectionLiterals.<Expression>newHashSet());
              }
            });
            TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(parameter, ref);
            information.provideType(_typeConformJudgement);
          }
        };
        IterableExtensions.<Variable>forEach(_localReferencesOfParameter, _function);
      }
    };
    IterableExtensions.<Variable>forEach(_parameters, _function);
  }
  
  protected void _inferTypes(final CheckConstraint constraint, final TypeInformation information) {
  }
  
  protected void _inferTypes(final CompareConstraint constraint, final TypeInformation information) {
    if (((Objects.equal(constraint.getFeature(), CompareFeature.EQUALITY) && (!Objects.equal(constraint.getLeftOperand(), null))) && (!Objects.equal(constraint.getRightOperand(), null)))) {
      ValueReference _leftOperand = constraint.getLeftOperand();
      ValueReference _rightOperand = constraint.getRightOperand();
      TypeConformJudgement _typeConformJudgement = new TypeConformJudgement(_leftOperand, _rightOperand);
      information.provideType(_typeConformJudgement);
      ValueReference _rightOperand_1 = constraint.getRightOperand();
      ValueReference _leftOperand_1 = constraint.getLeftOperand();
      TypeConformJudgement _typeConformJudgement_1 = new TypeConformJudgement(_rightOperand_1, _leftOperand_1);
      information.provideType(_typeConformJudgement_1);
    }
  }
  
  protected void _inferTypes(final PatternCompositionConstraint constraint, final TypeInformation information) {
    final PatternCall call = constraint.getCall();
    this.inferCallTypes(call, information);
  }
  
  private void inferCallTypes(final PatternCall call, final TypeInformation information) {
    final Pattern pattern = call.getPatternRef();
    for (int i = 0; (i < Math.min(call.getParameters().size(), pattern.getParameters().size())); i++) {
      EList<ValueReference> _parameters = call.getParameters();
      ValueReference _get = _parameters.get(i);
      EList<Variable> _parameters_1 = pattern.getParameters();
      Variable _get_1 = _parameters_1.get(i);
      ParameterTypeJudgement _parameterTypeJudgement = new ParameterTypeJudgement(_get, _get_1);
      information.provideType(_parameterTypeJudgement);
    }
  }
  
  protected void _inferTypes(final TypeCheckConstraint constraint, final TypeInformation information) {
    final Type constraintType = constraint.getType();
    if (((constraintType instanceof JavaType) && this.typeSystem.isValidType(constraintType))) {
      final IInputKey sourceType = this.typeSystem.extractTypeDescriptor(constraintType);
      boolean _notEquals = (!Objects.equal(sourceType, null));
      if (_notEquals) {
        VariableReference _var = constraint.getVar();
        TypeJudgement _typeJudgement = new TypeJudgement(_var, sourceType);
        information.provideType(_typeJudgement);
      }
    }
  }
  
  protected void _inferTypes(final PathExpressionConstraint constraint, final TypeInformation information) {
    PathExpressionHead _head = constraint.getHead();
    Type _type = _head.getType();
    boolean _isValidType = this.typeSystem.isValidType(_type);
    boolean _not = (!_isValidType);
    if (_not) {
      return;
    }
    PathExpressionHead _head_1 = constraint.getHead();
    Type _type_1 = _head_1.getType();
    final IInputKey sourceType = this.typeSystem.extractTypeDescriptor(_type_1);
    PathExpressionHead _head_2 = constraint.getHead();
    PathExpressionTail tail = _head_2.getTail();
    while ((!Objects.equal(tail.getTail(), null))) {
      PathExpressionTail _tail = tail.getTail();
      tail = _tail;
    }
    Type _type_2 = tail.getType();
    boolean _isValidType_1 = this.typeSystem.isValidType(_type_2);
    boolean _not_1 = (!_isValidType_1);
    if (_not_1) {
      return;
    }
    Type _type_3 = tail.getType();
    final IInputKey targetType = this.typeSystem.extractTypeDescriptor(_type_3);
    if (((!Objects.equal(sourceType, null)) && (!Objects.equal(targetType, null)))) {
      PathExpressionHead _head_3 = constraint.getHead();
      VariableReference _src = _head_3.getSrc();
      TypeJudgement _typeJudgement = new TypeJudgement(_src, sourceType);
      information.provideType(_typeJudgement);
      PathExpressionHead _head_4 = constraint.getHead();
      ValueReference _dst = _head_4.getDst();
      TypeJudgement _typeJudgement_1 = new TypeJudgement(_dst, targetType);
      information.provideType(_typeJudgement_1);
    }
  }
  
  protected void _inferTypes(final AggregatedValue reference, final TypeInformation information) {
    PatternCall _call = reference.getCall();
    this.inferCallTypes(_call, information);
    if ((Objects.equal(reference, null) || Objects.equal(reference.getAggregator(), null))) {
      return;
    }
    final List<VariableValue> values = AggregatorUtil.getAllAggregatorVariables(reference);
    int _size = values.size();
    boolean _equals = (_size == 0);
    if (_equals) {
      boolean _mustHaveAggregatorVariables = AggregatorUtil.mustHaveAggregatorVariables(reference);
      if (_mustHaveAggregatorVariables) {
        return;
      }
      JvmDeclaredType _aggregator = reference.getAggregator();
      final List<JvmType> returnTypes = AggregatorUtil.getReturnTypes(_aggregator);
      if ((Objects.equal(returnTypes, null) || (returnTypes.size() != 1))) {
        JvmDeclaredType _aggregator_1 = reference.getAggregator();
        String _simpleName = _aggregator_1.getSimpleName();
        String _format = String.format("Return type for aggregator %s is non uniquely specified.", _simpleName);
        this.logger.warning(_format);
        return;
      }
      final JvmType returnType = returnTypes.get(0);
      String _identifier = returnType.getIdentifier();
      JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(_identifier);
      TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
      information.provideType(_typeJudgement);
    } else {
      if (((values.size() != 1) || (!AggregatorUtil.mustHaveAggregatorVariables(reference)))) {
        return;
      }
      JvmDeclaredType _aggregator_2 = reference.getAggregator();
      final List<JvmType> parameterTypes = AggregatorUtil.getParameterTypes(_aggregator_2);
      JvmDeclaredType _aggregator_3 = reference.getAggregator();
      final List<JvmType> returnTypes_1 = AggregatorUtil.getReturnTypes(_aggregator_3);
      if ((Objects.equal(returnTypes_1, null) || (returnTypes_1.size() != parameterTypes.size()))) {
        JvmDeclaredType _aggregator_4 = reference.getAggregator();
        String _identifier_1 = _aggregator_4.getIdentifier();
        String _format_1 = String.format(
          "Incorrect aggregator type annotation for aggregator %s: Different number of parameters and return types", _identifier_1);
        this.logger.warning(_format_1);
        return;
      }
      for (int i = 0; (i < returnTypes_1.size()); i++) {
        JvmType _get = returnTypes_1.get(i);
        String _identifier_2 = _get.getIdentifier();
        JavaTransitiveInstancesKey _javaTransitiveInstancesKey_1 = new JavaTransitiveInstancesKey(_identifier_2);
        PatternCall _call_1 = reference.getCall();
        EList<ValueReference> _parameters = _call_1.getParameters();
        int _aggregateVariableIndex = AggregatorUtil.getAggregateVariableIndex(reference);
        ValueReference _get_1 = _parameters.get(_aggregateVariableIndex);
        JvmType _get_2 = parameterTypes.get(i);
        String _identifier_3 = _get_2.getIdentifier();
        JavaTransitiveInstancesKey _javaTransitiveInstancesKey_2 = new JavaTransitiveInstancesKey(_identifier_3);
        ConditionalJudgement _conditionalJudgement = new ConditionalJudgement(reference, _javaTransitiveInstancesKey_1, _get_1, _javaTransitiveInstancesKey_2);
        information.provideType(_conditionalJudgement);
      }
    }
  }
  
  protected void _inferTypes(final Expression reference, final TypeInformation information) {
  }
  
  protected void _inferTypes(final FunctionEvaluationValue reference, final TypeInformation information) {
    XExpression _expression = reference.getExpression();
    XbaseExpressionTypeJudgement _xbaseExpressionTypeJudgement = new XbaseExpressionTypeJudgement(reference, _expression, this.typeResolver);
    information.provideType(_xbaseExpressionTypeJudgement);
  }
  
  protected void _inferTypes(final BoolValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(Boolean.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final DoubleValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(Double.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final IntValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(Integer.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final ListValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(List.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final StringValue reference, final TypeInformation information) {
    JavaTransitiveInstancesKey _javaTransitiveInstancesKey = new JavaTransitiveInstancesKey(String.class);
    TypeJudgement _typeJudgement = new TypeJudgement(reference, _javaTransitiveInstancesKey);
    information.provideType(_typeJudgement);
  }
  
  protected void _inferTypes(final VariableValue reference, final TypeInformation information) {
  }
  
  public void inferTypes(final EObject reference, final TypeInformation information) {
    if (reference instanceof AggregatedValue) {
      _inferTypes((AggregatedValue)reference, information);
      return;
    } else if (reference instanceof BoolValue) {
      _inferTypes((BoolValue)reference, information);
      return;
    } else if (reference instanceof DoubleValue) {
      _inferTypes((DoubleValue)reference, information);
      return;
    } else if (reference instanceof FunctionEvaluationValue) {
      _inferTypes((FunctionEvaluationValue)reference, information);
      return;
    } else if (reference instanceof IntValue) {
      _inferTypes((IntValue)reference, information);
      return;
    } else if (reference instanceof ListValue) {
      _inferTypes((ListValue)reference, information);
      return;
    } else if (reference instanceof StringValue) {
      _inferTypes((StringValue)reference, information);
      return;
    } else if (reference instanceof VariableValue) {
      _inferTypes((VariableValue)reference, information);
      return;
    } else if (reference instanceof CheckConstraint) {
      _inferTypes((CheckConstraint)reference, information);
      return;
    } else if (reference instanceof CompareConstraint) {
      _inferTypes((CompareConstraint)reference, information);
      return;
    } else if (reference instanceof PathExpressionConstraint) {
      _inferTypes((PathExpressionConstraint)reference, information);
      return;
    } else if (reference instanceof PatternCompositionConstraint) {
      _inferTypes((PatternCompositionConstraint)reference, information);
      return;
    } else if (reference instanceof TypeCheckConstraint) {
      _inferTypes((TypeCheckConstraint)reference, information);
      return;
    } else if (reference instanceof Expression) {
      _inferTypes((Expression)reference, information);
      return;
    } else if (reference instanceof Pattern) {
      _inferTypes((Pattern)reference, information);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(reference, information).toString());
    }
  }
}
