/**
 * Copyright (c) 2010-2016, Tamas Szabo, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 *   Tamas Szabo, Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.util;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.query.patternlanguage.helper.JavaTypesHelper;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternCall;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.ValueReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableReference;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.VariableValue;
import org.eclipse.viatra.query.runtime.matchers.psystem.aggregations.AggregatorType;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationType;
import org.eclipse.xtext.common.types.JvmAnnotationValue;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeAnnotationValue;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * @author Tamas Szabo, Zoltan Ujhelyi
 * @since 1.4
 */
@SuppressWarnings("all")
public class AggregatorUtil {
  private final static String PARAMETER_TYPES_NAME = "parameterTypes";
  
  private final static String RETURN_TYPES_NAME = "returnTypes";
  
  private static List<JvmType> getAggregatorType(final JvmDeclaredType aggregatorType, final String typeString) {
    List<JvmType> _xblockexpression = null;
    {
      EList<JvmAnnotationReference> _annotations = aggregatorType.getAnnotations();
      final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationReference it) {
          JvmAnnotationType _annotation = it.getAnnotation();
          String _qualifiedName = _annotation.getQualifiedName();
          String _name = AggregatorType.class.getName();
          return Boolean.valueOf(Objects.equal(_qualifiedName, _name));
        }
      };
      final JvmAnnotationReference annotationType = IterableExtensions.<JvmAnnotationReference>findFirst(_annotations, _function);
      EList<JvmAnnotationValue> _explicitValues = null;
      if (annotationType!=null) {
        _explicitValues=annotationType.getExplicitValues();
      }
      JvmAnnotationValue _findFirst = null;
      if (_explicitValues!=null) {
        final Function1<JvmAnnotationValue, Boolean> _function_1 = new Function1<JvmAnnotationValue, Boolean>() {
          @Override
          public Boolean apply(final JvmAnnotationValue it) {
            String _valueName = it.getValueName();
            return Boolean.valueOf(Objects.equal(_valueName, typeString));
          }
        };
        _findFirst=IterableExtensions.<JvmAnnotationValue>findFirst(_explicitValues, _function_1);
      }
      final JvmAnnotationValue annotationValue = _findFirst;
      List<JvmType> _xifexpression = null;
      if ((annotationValue instanceof JvmTypeAnnotationValue)) {
        EList<JvmTypeReference> _values = ((JvmTypeAnnotationValue)annotationValue).getValues();
        final Function1<JvmTypeReference, JvmType> _function_2 = new Function1<JvmTypeReference, JvmType>() {
          @Override
          public JvmType apply(final JvmTypeReference it) {
            return it.getType();
          }
        };
        _xifexpression = ListExtensions.<JvmTypeReference, JvmType>map(_values, _function_2);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static List<JvmType> getReturnTypes(final JvmDeclaredType aggregatorType) {
    return AggregatorUtil.getAggregatorType(aggregatorType, AggregatorUtil.RETURN_TYPES_NAME);
  }
  
  public static List<JvmType> getParameterTypes(final JvmDeclaredType aggregatorType) {
    return AggregatorUtil.getAggregatorType(aggregatorType, AggregatorUtil.PARAMETER_TYPES_NAME);
  }
  
  /**
   * An aggregator expression may only have aggregated value as parameters if the corresponding {@link AggregatorType} annotation
   * does not define a single Void parameter. However, in that case, it _must_ have an aggregate parameter.
   */
  public static boolean mustHaveAggregatorVariables(final AggregatedValue value) {
    boolean _xblockexpression = false;
    {
      JvmDeclaredType _aggregator = value.getAggregator();
      final List<JvmType> types = AggregatorUtil.getParameterTypes(_aggregator);
      _xblockexpression = ((!Objects.equal(types, null)) && ((types.size() > 1) || ((types.size() == 1) && (!JavaTypesHelper.is(types.get(0), Void.class)))));
    }
    return _xblockexpression;
  }
  
  public static int getAggregateVariableIndex(final AggregatedValue value) {
    int index = 0;
    PatternCall _call = value.getCall();
    EList<ValueReference> _parameters = _call.getParameters();
    for (final ValueReference param : _parameters) {
      {
        if (((param instanceof VariableValue) && ((VariableValue) param).getValue().isAggregator())) {
          return index;
        }
        index++;
      }
    }
    return (-1);
  }
  
  private final static Function1<VariableValue, Boolean> aggregator = new Function1<VariableValue, Boolean>() {
    @Override
    public Boolean apply(final VariableValue v) {
      VariableReference _value = v.getValue();
      return Boolean.valueOf(_value.isAggregator());
    }
  };
  
  /**
   * Returns the aggregate variable the aggregator should work with. Given in a well-formed AggregatedValue only a
   * single aggregate variable should be present, this should be unique.
   */
  public static VariableValue getAggregatorVariable(final AggregatedValue value) {
    PatternCall _call = value.getCall();
    EList<ValueReference> _parameters = _call.getParameters();
    Iterable<VariableValue> _filter = Iterables.<VariableValue>filter(_parameters, VariableValue.class);
    return IterableExtensions.<VariableValue>findFirst(_filter, AggregatorUtil.aggregator);
  }
  
  /**
   * Returns all aggregate variables of the AggregatedValue. If the AggregatedValue has more aggregate variables,
   * it represents an error in the specification.
   */
  public static List<VariableValue> getAllAggregatorVariables(final AggregatedValue value) {
    PatternCall _call = value.getCall();
    EList<ValueReference> _parameters = _call.getParameters();
    Iterable<VariableValue> _filter = Iterables.<VariableValue>filter(_parameters, VariableValue.class);
    Iterable<VariableValue> _filter_1 = IterableExtensions.<VariableValue>filter(_filter, AggregatorUtil.aggregator);
    return ImmutableList.<VariableValue>copyOf(_filter_1);
  }
}
