/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Maps;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.viatra.query.runtime.api.AdvancedViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.exception.ViatraQueryException;
import org.eclipse.viatra.query.runtime.matchers.backend.IQueryBackendFactory;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.rete.matcher.ReteBackendFactory;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;
import org.junit.Test;

/**
 * This abstract test class can be used to measure the steady-state memory requirements of the base index and
 * Rete networks of individual queries on a given {@link QueryScope} and with a given query group.
 * 
 * <p/>
 * This test case prepares a ViatraQueryEngine on the given scope and with the provided query group.
 * After the initial preparation is done, the engine is wiped (deletes the Rete network but keeps the base index).
 * Next, the following is performed for each query in the group:
 * <p/>
 * <ol>
 *   <li> Wipe the engine </li>
 *   <li> Create the matcher and count matches </li>
 *   <li> Wipe the engine </li>
 * </ol>
 * 
 * After each step, the used, total and free heap space is logged in MBytes after 5 GC calls and 1 second of waiting.
 * Note that even this does not always provide an absolute steady state or a precise result, but can be useful for
 * finding problematic queries.
 */
@SuppressWarnings("all")
public abstract class QueryPerformanceTest {
  /**
   * @since 1.3
   */
  @Data
  protected static class QueryPerformanceData {
    private final int sequence;
    
    private final int countMatches;
    
    private final long usedHeapBefore;
    
    private final long usedHeapAfter;
    
    private final long usedHeap;
    
    private final long elapsed;
    
    public QueryPerformanceData(final int sequence, final int countMatches, final long usedHeapBefore, final long usedHeapAfter, final long usedHeap, final long elapsed) {
      super();
      this.sequence = sequence;
      this.countMatches = countMatches;
      this.usedHeapBefore = usedHeapBefore;
      this.usedHeapAfter = usedHeapAfter;
      this.usedHeap = usedHeap;
      this.elapsed = elapsed;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + this.sequence;
      result = prime * result + this.countMatches;
      result = prime * result + (int) (this.usedHeapBefore ^ (this.usedHeapBefore >>> 32));
      result = prime * result + (int) (this.usedHeapAfter ^ (this.usedHeapAfter >>> 32));
      result = prime * result + (int) (this.usedHeap ^ (this.usedHeap >>> 32));
      result = prime * result + (int) (this.elapsed ^ (this.elapsed >>> 32));
      return result;
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      QueryPerformanceTest.QueryPerformanceData other = (QueryPerformanceTest.QueryPerformanceData) obj;
      if (other.sequence != this.sequence)
        return false;
      if (other.countMatches != this.countMatches)
        return false;
      if (other.usedHeapBefore != this.usedHeapBefore)
        return false;
      if (other.usedHeapAfter != this.usedHeapAfter)
        return false;
      if (other.usedHeap != this.usedHeap)
        return false;
      if (other.elapsed != this.elapsed)
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("sequence", this.sequence);
      b.add("countMatches", this.countMatches);
      b.add("usedHeapBefore", this.usedHeapBefore);
      b.add("usedHeapAfter", this.usedHeapAfter);
      b.add("usedHeap", this.usedHeap);
      b.add("elapsed", this.elapsed);
      return b.toString();
    }
    
    @Pure
    public int getSequence() {
      return this.sequence;
    }
    
    @Pure
    public int getCountMatches() {
      return this.countMatches;
    }
    
    @Pure
    public long getUsedHeapBefore() {
      return this.usedHeapBefore;
    }
    
    @Pure
    public long getUsedHeapAfter() {
      return this.usedHeapAfter;
    }
    
    @Pure
    public long getUsedHeap() {
      return this.usedHeap;
    }
    
    @Pure
    public long getElapsed() {
      return this.elapsed;
    }
  }
  
  @Extension
  protected static Logger logger = ViatraQueryLoggingUtil.getLogger(QueryPerformanceTest.class);
  
  /**
   * @since 1.3
   */
  protected AdvancedViatraQueryEngine queryEngine;
  
  private Map<String, QueryPerformanceTest.QueryPerformanceData> results = Maps.<String, QueryPerformanceTest.QueryPerformanceData>newTreeMap();
  
  /**
   * This method shall return a scope that identifies the input artifact used for performance testing the queries.
   */
  public abstract QueryScope getScope() throws ViatraQueryException;
  
  /**
   * This method shall return the query group that contains the set of queries to evaluate.
   */
  public abstract IQueryGroup getQueryGroup() throws ViatraQueryException;
  
  /**
   * This method shall return the query backend class that will be used for evaluation.
   * The backend must be already registered in the {@link QueryBackendRegistry}.
   * 
   * Default implementation returns the registered default backend class.
   */
  public IQueryBackendFactory getQueryBackendFactory() {
    return new ReteBackendFactory();
  }
  
  protected void prepare() {
    try {
      QueryPerformanceTest.logger.info("Preparing query performance test");
      final QueryScope preparedScope = this.getScope();
      QueryPerformanceTest.logMemoryProperties("Scope prepared");
      AdvancedViatraQueryEngine _createUnmanagedEngine = AdvancedViatraQueryEngine.createUnmanagedEngine(preparedScope);
      this.queryEngine = _createUnmanagedEngine;
      IQueryGroup _queryGroup = this.getQueryGroup();
      _queryGroup.prepare(this.queryEngine);
      QueryPerformanceTest.logMemoryProperties("Base index created");
      this.queryEngine.wipe();
      QueryPerformanceTest.logMemoryProperties("VIATRA Query engine wiped");
      QueryPerformanceTest.logger.info("Prepared query performance test");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * This test case executes the performance evaluation on the given scope and with the provided query group.
   */
  @Test
  public void queryPerformance() {
    try {
      QueryPerformanceTest.logger.setLevel(Level.DEBUG);
      this.prepare();
      QueryPerformanceTest.logger.info("Starting query performance test");
      IQueryGroup _queryGroup = this.getQueryGroup();
      final Set<IQuerySpecification<?>> specifications = _queryGroup.getSpecifications();
      final int numOfSpecifications = ((Object[])Conversions.unwrapArray(specifications, Object.class)).length;
      int current = 0;
      for (final IQuerySpecification<?> _specification : specifications) {
        {
          current++;
          String _fullyQualifiedName = _specification.getFullyQualifiedName();
          String _plus = ("Measuring query " + _fullyQualifiedName);
          String _plus_1 = (_plus + "(");
          String _plus_2 = (_plus_1 + Integer.valueOf(current));
          String _plus_3 = (_plus_2 + "/");
          String _plus_4 = (_plus_3 + Integer.valueOf(numOfSpecifications));
          String _plus_5 = (_plus_4 + ")");
          QueryPerformanceTest.logger.debug(_plus_5);
          final long usedHeapBefore = this.wipe(_specification);
          this.performMeasurements(_specification, current, usedHeapBefore);
        }
      }
      QueryPerformanceTest.logger.info("Finished query performance test");
      this.printResults();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * @since 1.3
   */
  public long wipe(final IQuerySpecification<?> _specification) {
    this.queryEngine.wipe();
    final long usedHeapBefore = QueryPerformanceTest.logMemoryProperties("Wiped engine before building");
    return usedHeapBefore;
  }
  
  /**
   * @since 1.3
   */
  public QueryPerformanceTest.QueryPerformanceData performMeasurements(final IQuerySpecification<?> _specification, final int current, final long usedHeapBefore) {
    try {
      final IQuerySpecification<? extends ViatraQueryMatcher> specification = ((IQuerySpecification<? extends ViatraQueryMatcher>) _specification);
      QueryPerformanceTest.logger.debug("Building Rete");
      final Stopwatch watch = Stopwatch.createStarted();
      IQueryBackendFactory _queryBackendFactory = this.getQueryBackendFactory();
      HashMap<String, Object> _newHashMap = CollectionLiterals.<String, Object>newHashMap();
      QueryEvaluationHint _queryEvaluationHint = new QueryEvaluationHint(_queryBackendFactory, _newHashMap);
      final ViatraQueryMatcher matcher = this.queryEngine.getMatcher(specification, _queryEvaluationHint);
      watch.stop();
      final int countMatches = matcher.countMatches();
      final long usedHeapAfter = QueryPerformanceTest.logMemoryProperties("Matcher created");
      final long usedHeap = (usedHeapAfter - usedHeapBefore);
      final long elapsed = watch.elapsed(TimeUnit.MILLISECONDS);
      final QueryPerformanceTest.QueryPerformanceData result = new QueryPerformanceTest.QueryPerformanceData(current, countMatches, usedHeapBefore, usedHeapAfter, usedHeap, elapsed);
      String _fullyQualifiedName = specification.getFullyQualifiedName();
      this.results.put(_fullyQualifiedName, result);
      String _fullyQualifiedName_1 = specification.getFullyQualifiedName();
      String _plus = ("Query " + _fullyQualifiedName_1);
      String _plus_1 = (_plus + "( ");
      String _plus_2 = (_plus_1 + Integer.valueOf(countMatches));
      String _plus_3 = (_plus_2 + " matches, used ");
      String _plus_4 = (_plus_3 + Long.valueOf(usedHeap));
      String _plus_5 = (_plus_4 + 
        " kByte heap, took ");
      String _plus_6 = (_plus_5 + Long.valueOf(elapsed));
      String _plus_7 = (_plus_6 + " ms)");
      QueryPerformanceTest.logger.info(_plus_7);
      this.queryEngine.wipe();
      QueryPerformanceTest.logMemoryProperties("Wiped engine after building");
      QueryPerformanceTest.logger.debug("\n-------------------------------------------\n");
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected void printResults() {
    final StringBuilder resultSB = new StringBuilder("\n\nPerformance test results:\n");
    resultSB.append(
      "pattern, sequence, matches count, heap before (kb), heap after (kb), used heap (kb), elapsed (ms)\n");
    Set<Map.Entry<String, QueryPerformanceTest.QueryPerformanceData>> _entrySet = this.results.entrySet();
    final Procedure1<Map.Entry<String, QueryPerformanceTest.QueryPerformanceData>> _function = new Procedure1<Map.Entry<String, QueryPerformanceTest.QueryPerformanceData>>() {
      @Override
      public void apply(final Map.Entry<String, QueryPerformanceTest.QueryPerformanceData> entry) {
        String _key = entry.getKey();
        resultSB.append(_key);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value = entry.getValue();
        resultSB.append(_value.sequence);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value_1 = entry.getValue();
        resultSB.append(_value_1.countMatches);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value_2 = entry.getValue();
        resultSB.append(_value_2.usedHeapBefore);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value_3 = entry.getValue();
        resultSB.append(_value_3.usedHeapAfter);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value_4 = entry.getValue();
        resultSB.append(_value_4.usedHeap);
        resultSB.append(", ");
        QueryPerformanceTest.QueryPerformanceData _value_5 = entry.getValue();
        resultSB.append(_value_5.elapsed);
        resultSB.append("\n");
      }
    };
    IterableExtensions.<Map.Entry<String, QueryPerformanceTest.QueryPerformanceData>>forEach(_entrySet, _function);
    QueryPerformanceTest.logger.info(resultSB);
  }
  
  /**
   * Calls garbage collector 5 times, sleeps 1 second and logs the used, total and free heap sizes in MByte.
   * 
   * @param logger
   * @return The amount of used heap memory in kBytes
   */
  protected static long logMemoryProperties(final String status) {
    long _xblockexpression = (long) 0;
    {
      IntegerRange _upTo = new IntegerRange(0, 4);
      final Procedure1<Integer> _function = new Procedure1<Integer>() {
        @Override
        public void apply(final Integer it) {
          Runtime _runtime = Runtime.getRuntime();
          _runtime.gc();
        }
      };
      IterableExtensions.<Integer>forEach(_upTo, _function);
      try {
        Thread.sleep(1000);
      } catch (final Throwable _t) {
        if (_t instanceof InterruptedException) {
          final InterruptedException e = (InterruptedException)_t;
          QueryPerformanceTest.logger.trace("Sleep after GC interrupted");
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      Runtime _runtime = Runtime.getRuntime();
      long _talMemory = _runtime.totalMemory();
      final long totalHeapKB = (_talMemory / 1024);
      Runtime _runtime_1 = Runtime.getRuntime();
      long _freeMemory = _runtime_1.freeMemory();
      final long freeHeapKB = (_freeMemory / 1024);
      final long usedHeapKB = (totalHeapKB - freeHeapKB);
      QueryPerformanceTest.logger.debug(
        (((((((status + ": Used Heap size: ") + Long.valueOf((usedHeapKB / 1024))) + " MByte (Total: ") + Long.valueOf((totalHeapKB / 1024))) + 
          " MByte, Free: ") + Long.valueOf((freeHeapKB / 1024))) + " MByte)"));
      _xblockexpression = usedHeapKB;
    }
    return _xblockexpression;
  }
}
