/**
 * Copyright (c) 2010-2016, Gabor Bergmann, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Gabor Bergmann - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.eclipse.viatra.query.runtime.api.GenericQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.matchers.backend.IQueryBackendFactory;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PDisjunction;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.testing.core.QueryPerformanceTest;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * This abstract test class can be used to measure the steady-state memory requirements
 * of the Rete networks of individual queries on a given QueryScope,
 * relative to the network built to evaluate its dependencies.
 * In other words, the "local cost" of a query is measured; this is the memory footprint of the query
 * imposed on top of the memory footprint of all other queries invoked by it.
 * 
 * <p>
 * This test case prepares an ViatraEngine on the given scope and with the provided query group.
 * After the initial preparation is done, the engine is wiped (deletes the Rete network but keeps the base index).
 * Next, the following is performed for each query in the group:
 * <p/>
 * <ol>
 *   <li> Wipe the engine </li>
 *   <li> Prepare all queries referenced by the query under consideration, then measure memory </li>
 *   <li> Create the matcher for the query and count matches, then measure memory again to see the difference </li>
 *   <li> Wipe the engine </li>
 * </ol>
 * 
 * After each step, the used, total and free heap space is logged in MBytes after 5 GC calls and 1 second of waiting.
 * Note that even this does not always provide an absolute steady state or a precise result, but can be useful for
 * finding problematic queries.
 * 
 * @since 1.3
 */
@SuppressWarnings("all")
public abstract class RelativeQueryPerformanceTest extends QueryPerformanceTest {
  private Map<String, Long> absoluteHeapResults = Maps.<String, Long>newTreeMap();
  
  private Map<String, Long> relativeHeapResults = Maps.<String, Long>newTreeMap();
  
  @Override
  public QueryPerformanceTest.QueryPerformanceData performMeasurements(final IQuerySpecification<?> specification, final int current, final long usedHeapBefore) {
    try {
      final IQueryGroup prerequisites = RelativeQueryPerformanceTest.getDirectPrerequisites(specification);
      long prerequisitesHeap = 0L;
      {
        QueryPerformanceTest.logger.debug("Building Prerequisites");
        final Stopwatch watch = Stopwatch.createStarted();
        IQueryBackendFactory _queryBackendFactory = this.getQueryBackendFactory();
        HashMap<String, Object> _newHashMap = CollectionLiterals.<String, Object>newHashMap();
        QueryEvaluationHint _queryEvaluationHint = new QueryEvaluationHint(_queryBackendFactory, _newHashMap);
        this.queryEngine.prepareGroup(prerequisites, _queryEvaluationHint);
        watch.stop();
        final long usedHeapAfter = QueryPerformanceTest.logMemoryProperties("Prerequisites built");
        prerequisitesHeap = (usedHeapAfter - usedHeapBefore);
        String _fullyQualifiedName = specification.getFullyQualifiedName();
        String _plus = ("Prerequisites of query " + _fullyQualifiedName);
        String _plus_1 = (_plus + "(used ");
        String _plus_2 = (_plus_1 + Long.valueOf(prerequisitesHeap));
        String _plus_3 = (_plus_2 + 
          " kByte heap, took ");
        long _elapsed = watch.elapsed(TimeUnit.MILLISECONDS);
        String _plus_4 = (_plus_3 + Long.valueOf(_elapsed));
        String _plus_5 = (_plus_4 + " ms)");
        QueryPerformanceTest.logger.info(_plus_5);
      }
      final QueryPerformanceTest.QueryPerformanceData result = super.performMeasurements(specification, current, usedHeapBefore);
      String _fullyQualifiedName = specification.getFullyQualifiedName();
      long _usedHeap = result.getUsedHeap();
      this.absoluteHeapResults.put(_fullyQualifiedName, Long.valueOf(_usedHeap));
      String _fullyQualifiedName_1 = specification.getFullyQualifiedName();
      long _usedHeap_1 = result.getUsedHeap();
      long _minus = (_usedHeap_1 - prerequisitesHeap);
      this.relativeHeapResults.put(_fullyQualifiedName_1, Long.valueOf(_minus));
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  protected void printResults() {
    super.printResults();
    final StringBuilder resultSB = new StringBuilder("\nAbsoluteHeap[kB]\trelativeHeap[kB]\tquery\n");
    Set<Map.Entry<String, Long>> _entrySet = this.absoluteHeapResults.entrySet();
    final Procedure1<Map.Entry<String, Long>> _function = new Procedure1<Map.Entry<String, Long>>() {
      @Override
      public void apply(final Map.Entry<String, Long> entry) {
        final String query = entry.getKey();
        Long _value = entry.getValue();
        Long _get = RelativeQueryPerformanceTest.this.relativeHeapResults.get(query);
        String _format = String.format("%12d\t%12d\t%s\n", _value, _get, query);
        resultSB.append(_format);
      }
    };
    IterableExtensions.<Map.Entry<String, Long>>forEach(_entrySet, _function);
    QueryPerformanceTest.logger.info(resultSB);
  }
  
  protected static IQueryGroup getDirectPrerequisites(final IQuerySpecification<?> query) {
    IQueryGroup _xblockexpression = null;
    {
      PQuery _internalQueryRepresentation = query.getInternalQueryRepresentation();
      PDisjunction _disjunctBodies = _internalQueryRepresentation.getDisjunctBodies();
      final Set<PQuery> referredQueries = _disjunctBodies.getDirectReferredQueries();
      final Function1<PQuery, Iterable<IQuerySpecification>> _function = new Function1<PQuery, Iterable<IQuerySpecification>>() {
        @Override
        public Iterable<IQuerySpecification> apply(final PQuery it) {
          List<Object> _publishedAs = it.publishedAs();
          return Iterables.<IQuerySpecification>filter(_publishedAs, IQuerySpecification.class);
        }
      };
      Iterable<Iterable<IQuerySpecification>> _map = IterableExtensions.<PQuery, Iterable<IQuerySpecification>>map(referredQueries, _function);
      final Iterable<IQuerySpecification> referredSpecifications = Iterables.<IQuerySpecification>concat(_map);
      _xblockexpression = GenericQueryGroup.of(((IQuerySpecification<?>[])Conversions.unwrapArray(referredSpecifications, IQuerySpecification.class)));
    }
    return _xblockexpression;
  }
}
