/*******************************************************************************
 * Copyright (c) 2010-2013, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Istvan Rath - initial API and implementation
 *   Abel Hegedus - migrate to EMF scope
 *   
 *******************************************************************************/
package org.eclipse.viatra.addon.viewers.runtime.extensions;

import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.viatra.addon.viewers.runtime.ViewersRuntimePlugin;
import org.eclipse.viatra.addon.viewers.runtime.model.ViewerState;
import org.eclipse.viatra.query.runtime.api.IModelConnectorTypeEnum;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.base.api.BaseIndexOptions;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.exception.ViatraQueryException;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

/**
 * Utility class to serve as an extension for {@link ViewPart}s wishing to use VIATRA Viewers.
 * 
 * @author Istvan Rath
 *
 */
public abstract class ViatraViewersViewSupport extends ViatraViewersPartSupport {

    /**
     * Defines the matching scope for the underlying {@link ViatraQueryEngine}. TODO is {@link IModelConnectorTypeEnum}
     * the proper choice for this?
     */
    protected IModelConnectorTypeEnum connectorType = IModelConnectorTypeEnum.RESOURCESET;

    /**
     * The {@link ViewerState} that represents the stateful model behind the contents shown by the owner.
     */
    protected ViewerState state;

    /**
     * Constructs a new View support instance.
     */
    public ViatraViewersViewSupport(IViewPart _owner, ViewersComponentConfiguration _config,
            IModelConnectorTypeEnum _scope) {
        super(_owner, _config);
        this.connectorType = _scope;
    }

    protected IViewPart getOwner() {
        return (IViewPart) owner;
    }

    @Override
    public void dispose() {
        unbindModel();
        super.dispose();
    }

    @Override
    protected void onSelectionChanged(List<Object> objects) {
        // extract model source
        try {
            EMFScope target = extractModelSource(objects);
            if (target != null && !target.equals(this.modelSource)) {
                // we have found a new target
                unsetModelSource();
                setModelSource(target);
            }
        } catch (ViatraQueryException e) {
            throw new RuntimeException("Failed to extract model source", e);
        }
    }

    protected EMFScope extractModelSource(List<Object> objects) throws ViatraQueryException {
        Set<Notifier> notifiers = ImmutableSet.copyOf(Iterables.filter(objects, Notifier.class));
        // extract logic
        switch (connectorType) {
        default:
        case RESOURCESET:
            for (Notifier n : notifiers) {
                if (n instanceof ResourceSet) {
                    return new EMFScope(n);
                } else if (n instanceof Resource) {
                    return new EMFScope(((Resource) n).getResourceSet());
                } else {
                    EObject eO = (EObject) n;
                    return new EMFScope(eO.eResource().getResourceSet());
                }
            }
        case RESOURCE:
            for (Notifier n : notifiers) {
                if (n instanceof ResourceSet) {
                    continue; // we cannot extract a resource from a resourceset reliably
                } else if (n instanceof Resource) {
                    return new EMFScope(((Resource) n));
                } else {
                    EObject eO = (EObject) n;
                    return new EMFScope(eO.eResource());
                }
            }
        }

        return new EMFScope(notifiers, new BaseIndexOptions());
    }

    protected EMFScope modelSource;

    private void setModelSource(EMFScope p) {
        this.modelSource = p;
        // TODO propertySheetPage setCurrent
        bindModel();
        showView();
    }

    private void unsetModelSource() {
        hideView();
        unbindModel();
        // proppage setCurrentEditor null
        this.modelSource = null;
    }

    protected ViatraQueryEngine getEngine() {
        Assert.isNotNull(this.modelSource);
        try {
            return ViatraQueryEngine.on(this.modelSource);
        } catch (ViatraQueryException e) {
            ViewersRuntimePlugin.getDefault().getLog()
                    .log(new Status(Status.ERROR, ViewersRuntimePlugin.PLUGIN_ID, e.getLocalizedMessage(), e));
        }
        return null;
    }

    // ***************** layout stuff ***************** //

    private Composite parent, cover;
    private Control contents;

    private StackLayout layout;

    /**
     * Create the SWT UI for the owner.
     * 
     * @param _parent
     *            the SWT part received by the owner in its createPartControl method
     * @param _contents
     *            the SWT UI to be displayed for actual contents
     */
    public void createPartControl(Composite _parent, Control _contents) {
        parent = _parent;
        layout = new StackLayout();
        parent.setLayout(layout);
        contents = _contents;
        cover = new Composite(parent, SWT.NO_SCROLL);
        layout.topControl = cover;
        // init
        init();
    }

    private void showView() {
        layout.topControl = contents;
        parent.layout();
    }

    private void hideView() {
        layout.topControl = cover;
        parent.layout();
    }

    /**
     * Subclasses bind their viewer models here.
     */
    protected abstract void bindModel();

    /**
     * Subclasses unbind their viewer models here.
     */
    protected abstract void unbindModel();

    // ******************** TODO propertysheetpage support
}
