/**
 * Copyright (c) 2004-2015, Istvan David, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Istvan David - initial API and implementation
 */
package org.eclipse.viatra.cep.core.engine.compiler.builders;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.cep.core.engine.compiler.TransformationBasedCompiler;
import org.eclipse.viatra.cep.core.metamodels.automaton.Automaton;
import org.eclipse.viatra.cep.core.metamodels.automaton.AutomatonFactory;
import org.eclipse.viatra.cep.core.metamodels.automaton.EpsilonTransition;
import org.eclipse.viatra.cep.core.metamodels.automaton.Guard;
import org.eclipse.viatra.cep.core.metamodels.automaton.NegativeTransition;
import org.eclipse.viatra.cep.core.metamodels.automaton.Parameter;
import org.eclipse.viatra.cep.core.metamodels.automaton.State;
import org.eclipse.viatra.cep.core.metamodels.automaton.TimedZone;
import org.eclipse.viatra.cep.core.metamodels.automaton.Transition;
import org.eclipse.viatra.cep.core.metamodels.automaton.TypedTransition;
import org.eclipse.viatra.cep.core.metamodels.automaton.Within;
import org.eclipse.viatra.cep.core.metamodels.events.AbstractMultiplicity;
import org.eclipse.viatra.cep.core.metamodels.events.AtLeastOne;
import org.eclipse.viatra.cep.core.metamodels.events.ComplexEventOperator;
import org.eclipse.viatra.cep.core.metamodels.events.ComplexEventPattern;
import org.eclipse.viatra.cep.core.metamodels.events.EventPattern;
import org.eclipse.viatra.cep.core.metamodels.events.EventPatternReference;
import org.eclipse.viatra.cep.core.metamodels.events.FOLLOWS;
import org.eclipse.viatra.cep.core.metamodels.events.Infinite;
import org.eclipse.viatra.cep.core.metamodels.events.Multiplicity;
import org.eclipse.viatra.cep.core.metamodels.events.Timewindow;
import org.eclipse.viatra.cep.core.metamodels.trace.TimedZoneTrace;
import org.eclipse.viatra.cep.core.metamodels.trace.TraceFactory;
import org.eclipse.viatra.cep.core.metamodels.trace.TraceModel;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class BuilderPrimitives {
  @Extension
  protected final AutomatonFactory automatonFactory = AutomatonFactory.eINSTANCE;
  
  @Extension
  protected final TraceFactory traceFactory = TraceFactory.eINSTANCE;
  
  private TraceModel traceModel;
  
  public BuilderPrimitives(final TraceModel traceModel) {
    this.traceModel = traceModel;
  }
  
  /**
   * Creates a new {@link State}, connects it with the preState and applies a {@link Guard} with the type of the
   * {@link EventPatternReference}.
   */
  public State transitionToNewState(final Automaton automaton, final EventPatternReference eventPatternReference, final State preState) {
    State _xblockexpression = null;
    {
      State nextState = this.automatonFactory.createState();
      EList<State> _states = automaton.getStates();
      _states.add(nextState);
      this.transitionBetween(eventPatternReference, preState, nextState);
      _xblockexpression = nextState;
    }
    return _xblockexpression;
  }
  
  /**
   * Creates a new {@link State}, connects it with the preState via a {@link NegativeTransition} and applies a
   * {@link Guard} with the type of the {@link EventPatternReference}.
   */
  public State negTransitionToNewState(final Automaton automaton, final EventPatternReference eventPatternReference, final State preState) {
    State _xblockexpression = null;
    {
      State nextState = this.automatonFactory.createState();
      EList<State> _states = automaton.getStates();
      _states.add(nextState);
      this.negTransitionBetween(eventPatternReference, preState, nextState);
      _xblockexpression = nextState;
    }
    return _xblockexpression;
  }
  
  /**
   * Connects the preState and the postState and applies a {@link Guard} with the type of the
   * {@link EventPatternReference}.
   */
  public void transitionBetween(final EventPatternReference eventPatternReference, final State preState, final State postState) {
    final TypedTransition transition = this.newTransition(preState, postState);
    this.addGuard(transition, eventPatternReference);
    this.handleTransitionParameters(eventPatternReference, transition);
  }
  
  /**
   * Connects the preState and the postState and applies a {@link Guard} with the type of the
   * {@link EventPatternReference}.
   */
  public boolean transitionBetween(final EventPattern eventPattern, final State preState, final State postState) {
    boolean _xblockexpression = false;
    {
      final TypedTransition transition = this.newTransition(preState, postState);
      _xblockexpression = this.addGuard(transition, eventPattern);
    }
    return _xblockexpression;
  }
  
  /**
   * Connects the preState and the postState with a {@link NegativeTransition} and applies a {@link Guard}
   * with the type of the {@link EventPatternReference}.
   */
  public void negTransitionBetween(final EventPatternReference eventPatternReference, final State preState, final State postState) {
    final NegativeTransition transition = this.newNegTransition(preState, postState);
    this.addGuard(transition, eventPatternReference);
    this.handleTransitionParameters(eventPatternReference, transition);
  }
  
  /**
   * Primitive for creating a new {@link Transition}.
   */
  public TypedTransition newTransition(final State preState, final State postState) {
    TypedTransition _xblockexpression = null;
    {
      final TypedTransition transition = this.automatonFactory.createTypedTransition();
      transition.setPreState(preState);
      transition.setPostState(postState);
      _xblockexpression = transition;
    }
    return _xblockexpression;
  }
  
  /**
   * Primitive for creating a new {@link NegativeTransition}.
   */
  public NegativeTransition newNegTransition(final State preState, final State postState) {
    NegativeTransition _xblockexpression = null;
    {
      final NegativeTransition transition = this.automatonFactory.createNegativeTransition();
      transition.setPreState(preState);
      transition.setPostState(postState);
      _xblockexpression = transition;
    }
    return _xblockexpression;
  }
  
  /**
   * Applies a {@link Guard} with the type of the {@link EventPatternReference} to the {@link Transition}.
   */
  public boolean addGuard(final TypedTransition transition, final EventPatternReference eventPatternReference) {
    EventPattern _eventPattern = eventPatternReference.getEventPattern();
    return this.addGuard(transition, _eventPattern);
  }
  
  /**
   * Applies a {@link Guard} with the type of the {@link EventPattern} to the {@link Transition}.
   */
  public boolean addGuard(final TypedTransition transition, final EventPattern eventPattern) {
    boolean _xblockexpression = false;
    {
      Guard guard = this.automatonFactory.createGuard();
      guard.setEventType(eventPattern);
      EList<Guard> _guards = transition.getGuards();
      _xblockexpression = _guards.add(guard);
    }
    return _xblockexpression;
  }
  
  /**
   * Maps the an {@link EventPatternReference} wrt its multiplicity between the preState and a fixed postState.
   * Creates an {@link EpsilonTransition} between the last created intermediate state and the postState.
   */
  public void mapWithMultiplicity(final EventPatternReference eventPatternReference, final Automaton automaton, final State preState, final State postState) {
    final State lastState = this.mapWithMultiplicity(eventPatternReference, automaton, preState);
    boolean _equals = Objects.equal(lastState, null);
    if (_equals) {
      this.createEpsilon(preState, postState);
    } else {
      this.createEpsilon(lastState, postState);
    }
  }
  
  /**
   * Maps the eventPatternReference wrt its multiplicity without a fixed postState.
   */
  public State mapWithMultiplicity(final EventPatternReference eventPatternReference, final Automaton automaton, final State preState) {
    State _xblockexpression = null;
    {
      State nextState = null;
      AbstractMultiplicity _multiplicity = eventPatternReference.getMultiplicity();
      boolean _equals = Objects.equal(_multiplicity, null);
      if (_equals) {
        boolean _equals_1 = Objects.equal(nextState, null);
        if (_equals_1) {
          State _transitionToNewState = this.transitionToNewState(automaton, eventPatternReference, preState);
          nextState = _transitionToNewState;
        } else {
          State _transitionToNewState_1 = this.transitionToNewState(automaton, eventPatternReference, nextState);
          nextState = _transitionToNewState_1;
        }
      } else {
        AbstractMultiplicity _multiplicity_1 = eventPatternReference.getMultiplicity();
        if ((_multiplicity_1 instanceof Multiplicity)) {
          for (int i = 0; (i < ((Multiplicity) eventPatternReference.getMultiplicity()).getValue()); i++) {
            boolean _equals_2 = Objects.equal(nextState, null);
            if (_equals_2) {
              State _transitionToNewState_2 = this.transitionToNewState(automaton, eventPatternReference, preState);
              nextState = _transitionToNewState_2;
            } else {
              State _transitionToNewState_3 = this.transitionToNewState(automaton, eventPatternReference, nextState);
              nextState = _transitionToNewState_3;
            }
          }
        } else {
          boolean _or = false;
          AbstractMultiplicity _multiplicity_2 = eventPatternReference.getMultiplicity();
          if ((_multiplicity_2 instanceof AtLeastOne)) {
            _or = true;
          } else {
            AbstractMultiplicity _multiplicity_3 = eventPatternReference.getMultiplicity();
            _or = (_multiplicity_3 instanceof Infinite);
          }
          if (_or) {
            AbstractMultiplicity _multiplicity_4 = eventPatternReference.getMultiplicity();
            if ((_multiplicity_4 instanceof AtLeastOne)) {
              boolean _equals_2 = Objects.equal(nextState, null);
              if (_equals_2) {
                State _transitionToNewState_2 = this.transitionToNewState(automaton, eventPatternReference, preState);
                nextState = _transitionToNewState_2;
              } else {
                State _transitionToNewState_3 = this.transitionToNewState(automaton, eventPatternReference, nextState);
                nextState = _transitionToNewState_3;
              }
            }
            boolean _equals_3 = Objects.equal(nextState, null);
            if (_equals_3) {
              State _selfTransition = this.selfTransition(eventPatternReference, preState);
              nextState = _selfTransition;
            } else {
              State _selfTransition_1 = this.selfTransition(eventPatternReference, nextState);
              nextState = _selfTransition_1;
            }
          } else {
            throw new IllegalArgumentException();
          }
        }
      }
      _xblockexpression = nextState;
    }
    return _xblockexpression;
  }
  
  /**
   * Creates a self-transition on the state.
   */
  public State selfTransition(final EventPatternReference eventPatternReference, final State state) {
    State _xblockexpression = null;
    {
      TypedTransition selfTransition = this.automatonFactory.createTypedTransition();
      Guard selfGuard = this.automatonFactory.createGuard();
      EventPattern _eventPattern = eventPatternReference.getEventPattern();
      selfGuard.setEventType(_eventPattern);
      EList<Guard> _guards = selfTransition.getGuards();
      _guards.add(selfGuard);
      selfTransition.setPreState(state);
      selfTransition.setPostState(state);
      _xblockexpression = state;
    }
    return _xblockexpression;
  }
  
  /**
   * Creates an Epsilon-transition between two states.
   */
  public void createEpsilon(final State preState, final State postState) {
    boolean _and = false;
    boolean _notEquals = (!Objects.equal(preState, null));
    if (!_notEquals) {
      _and = false;
    } else {
      boolean _notEquals_1 = (!Objects.equal(postState, null));
      _and = _notEquals_1;
    }
    Preconditions.checkArgument(_and);
    EpsilonTransition epsilon = this.automatonFactory.createEpsilonTransition();
    epsilon.setPreState(preState);
    epsilon.setPostState(postState);
  }
  
  /**
   * Removes a transition from the model.
   */
  public boolean removeTransition(final Transition transition) {
    boolean _xblockexpression = false;
    {
      transition.setPostState(null);
      State _preState = transition.getPreState();
      EList<Transition> _outTransitions = _preState.getOutTransitions();
      _xblockexpression = _outTransitions.remove(transition);
    }
    return _xblockexpression;
  }
  
  /**
   * Removes a State from the model.
   */
  public boolean removeState(final State state) {
    EObject _eContainer = state.eContainer();
    EList<State> _states = ((Automaton) _eContainer).getStates();
    return _states.remove(state);
  }
  
  /**
   * Maps parameters onto a transition.
   */
  public void handleTransitionParameters(final EventPatternReference eventPatternReference, final TypedTransition transition) {
    final EList<String> parameterSymbolicNames = eventPatternReference.getParameterSymbolicNames();
    for (int i = 0; (i < parameterSymbolicNames.size()); i++) {
      {
        final String symbolicName = parameterSymbolicNames.get(i);
        boolean _equalsIgnoreCase = symbolicName.equalsIgnoreCase(TransformationBasedCompiler.OMITTED_PARAMETER_SYMBOLIC_NAME);
        boolean _not = (!_equalsIgnoreCase);
        if (_not) {
          Parameter transitionParameter = this.automatonFactory.createParameter();
          transitionParameter.setSymbolicName(symbolicName);
          transitionParameter.setPosition(i);
          EList<Parameter> _parameters = transition.getParameters();
          _parameters.add(transitionParameter);
        }
      }
    }
  }
  
  public void alignTimewindow(final Automaton automaton, final ComplexEventPattern eventPattern, final TypedTransition transition) {
    EList<TimedZoneTrace> _timedZoneTraces = this.traceModel.getTimedZoneTraces();
    final Function1<TimedZoneTrace, Boolean> _function = new Function1<TimedZoneTrace, Boolean>() {
      @Override
      public Boolean apply(final TimedZoneTrace tzTrace) {
        Transition _transition = tzTrace.getTransition();
        return Boolean.valueOf(_transition.equals(transition));
      }
    };
    final TimedZoneTrace timedZoneTrace = IterableExtensions.<TimedZoneTrace>findFirst(_timedZoneTraces, _function);
    boolean _equals = Objects.equal(timedZoneTrace, null);
    if (_equals) {
      return;
    }
    State _preState = transition.getPreState();
    EList<Transition> _outTransitions = _preState.getOutTransitions();
    final Function1<Transition, Boolean> _function_1 = new Function1<Transition, Boolean>() {
      @Override
      public Boolean apply(final Transition tr) {
        boolean _equals = tr.equals(transition);
        return Boolean.valueOf((!_equals));
      }
    };
    Iterable<Transition> _filter = IterableExtensions.<Transition>filter(_outTransitions, _function_1);
    final Procedure1<Transition> _function_2 = new Procedure1<Transition>() {
      @Override
      public void apply(final Transition tr) {
        State _postState = tr.getPostState();
        EList<Transition> _outTransitions = _postState.getOutTransitions();
        int _size = _outTransitions.size();
        boolean _equals = (_size == 1);
        Preconditions.checkArgument(_equals);
        Within timedZone = BuilderPrimitives.this.automatonFactory.createWithin();
        TimedZone _timedZone = timedZoneTrace.getTimedZone();
        long _time = _timedZone.getTime();
        timedZone.setTime(_time);
        State _preState = transition.getPreState();
        timedZone.setInState(_preState);
        State _postState_1 = transition.getPostState();
        timedZone.setOutState(_postState_1);
        EList<TimedZone> _timedZones = automaton.getTimedZones();
        _timedZones.add(timedZone);
        final TimedZoneTrace tzTrace = BuilderPrimitives.this.traceFactory.createTimedZoneTrace();
        tzTrace.setTimedZone(timedZone);
        tzTrace.setTransition(tr);
        EList<TimedZoneTrace> _timedZoneTraces = BuilderPrimitives.this.traceModel.getTimedZoneTraces();
        _timedZoneTraces.add(tzTrace);
        State _postState_2 = tr.getPostState();
        EList<Transition> _outTransitions_1 = _postState_2.getOutTransitions();
        Transition _head = IterableExtensions.<Transition>head(_outTransitions_1);
        boolean _not = (!(_head instanceof EpsilonTransition));
        if (_not) {
          State _postState_3 = tr.getPostState();
          EList<TimedZone> _inStateOf = _postState_3.getInStateOf();
          _inStateOf.add(timedZone);
        }
      }
    };
    IterableExtensions.<Transition>forEach(_filter, _function_2);
    EList<TimedZone> _timedZones = automaton.getTimedZones();
    TimedZone _timedZone = timedZoneTrace.getTimedZone();
    _timedZones.remove(_timedZone);
    EList<TimedZoneTrace> _timedZoneTraces_1 = this.traceModel.getTimedZoneTraces();
    _timedZoneTraces_1.remove(timedZoneTrace);
  }
  
  public boolean alignTimewindow(final Automaton automaton, final ComplexEventPattern eventPattern, final TypedTransition transition, final State newlyCreatedState) {
    boolean _xblockexpression = false;
    {
      EList<TimedZoneTrace> _timedZoneTraces = this.traceModel.getTimedZoneTraces();
      final Function1<TimedZoneTrace, Boolean> _function = new Function1<TimedZoneTrace, Boolean>() {
        @Override
        public Boolean apply(final TimedZoneTrace tzTrace) {
          Transition _transition = tzTrace.getTransition();
          return Boolean.valueOf(_transition.equals(transition));
        }
      };
      final TimedZoneTrace timedZoneTrace = IterableExtensions.<TimedZoneTrace>findFirst(_timedZoneTraces, _function);
      boolean _xifexpression = false;
      boolean _notEquals = (!Objects.equal(timedZoneTrace, null));
      if (_notEquals) {
        boolean _xblockexpression_1 = false;
        {
          EList<TimedZone> _inStateOf = newlyCreatedState.getInStateOf();
          TimedZone _timedZone = timedZoneTrace.getTimedZone();
          _inStateOf.add(_timedZone);
          EList<TimedZoneTrace> _timedZoneTraces_1 = this.traceModel.getTimedZoneTraces();
          _xblockexpression_1 = _timedZoneTraces_1.remove(timedZoneTrace);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public void initializeTimewindow(final Automaton automaton, final EventPattern eventPattern, final State inState, final State outState) {
    if ((!(eventPattern instanceof ComplexEventPattern))) {
      return;
    }
    Timewindow _timewindow = ((ComplexEventPattern) eventPattern).getTimewindow();
    boolean _equals = Objects.equal(_timewindow, null);
    if (_equals) {
      return;
    }
    ComplexEventOperator _operator = ((ComplexEventPattern) eventPattern).getOperator();
    Preconditions.checkArgument((_operator instanceof FOLLOWS));
    Within timedZone = this.automatonFactory.createWithin();
    Timewindow _timewindow_1 = ((ComplexEventPattern) eventPattern).getTimewindow();
    long _time = _timewindow_1.getTime();
    timedZone.setTime(_time);
    timedZone.setInState(inState);
    timedZone.setOutState(outState);
    EList<TimedZone> _timedZones = automaton.getTimedZones();
    _timedZones.add(timedZone);
    final TimedZoneTrace tzTrace = this.traceFactory.createTimedZoneTrace();
    tzTrace.setTimedZone(timedZone);
    EList<Transition> _outTransitions = inState.getOutTransitions();
    Transition _head = IterableExtensions.<Transition>head(_outTransitions);
    tzTrace.setTransition(_head);
    EList<TimedZoneTrace> _timedZoneTraces = this.traceModel.getTimedZoneTraces();
    _timedZoneTraces.add(tzTrace);
  }
}
