/**
 * Copyright (c) 2004-2014, Istvan David, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Istvan David - initial API and implementation
 */
package org.eclipse.viatra.cep.core.eventprocessingstrategy;

import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.cep.core.engine.IEventModelManager;
import org.eclipse.viatra.cep.core.eventprocessingstrategy.AbstractStrategy;
import org.eclipse.viatra.cep.core.logging.LoggerUtils;
import org.eclipse.viatra.cep.core.metamodels.automaton.Automaton;
import org.eclipse.viatra.cep.core.metamodels.automaton.AutomatonFactory;
import org.eclipse.viatra.cep.core.metamodels.automaton.EventToken;
import org.eclipse.viatra.cep.core.metamodels.automaton.InitState;
import org.eclipse.viatra.cep.core.metamodels.automaton.InternalModel;
import org.eclipse.viatra.cep.core.metamodels.automaton.State;
import org.eclipse.viatra.cep.core.utils.AutomatonUtils;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Common ancestor of the <i>immediate</i> type of strategies. Defines how the {@link Automaton} resets should be
 * handled in general.
 * 
 * @author Istvan David
 */
@SuppressWarnings("all")
public abstract class AbstractImmediateStrategy extends AbstractStrategy {
  @Extension
  private final Logger logger = LoggerUtils.getInstance().getLogger();
  
  public AbstractImmediateStrategy(final IEventModelManager eventModelManager) {
    super(eventModelManager);
  }
  
  @Override
  public void handleAutomatonResets(final InternalModel model, final AutomatonFactory factory) {
    EList<Automaton> _automata = model.getAutomata();
    final Function1<Automaton, Boolean> _function = new Function1<Automaton, Boolean>() {
      @Override
      public Boolean apply(final Automaton automaton) {
        return Boolean.valueOf(AbstractImmediateStrategy.this.needsReset(automaton));
      }
    };
    Iterable<Automaton> _filter = IterableExtensions.<Automaton>filter(_automata, _function);
    final Procedure1<Automaton> _function_1 = new Procedure1<Automaton>() {
      @Override
      public void apply(final Automaton automaton) {
        String _id = AbstractImmediateStrategy.this.id(automaton);
        String _format = String.format("ImmediateStrategy: No update in automaton: %s. Resetting automaton.", _id);
        AbstractImmediateStrategy.this.logger.debug(_format);
        Iterable<State> _normalStates = AutomatonUtils.getNormalStates(automaton);
        final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
          @Override
          public Boolean apply(final State state) {
            return Boolean.valueOf(AutomatonUtils.notEmpty(state));
          }
        };
        Iterable<State> _filter = IterableExtensions.<State>filter(_normalStates, _function);
        final Procedure1<State> _function_1 = new Procedure1<State>() {
          @Override
          public void apply(final State state) {
            String _prettyLabel = AbstractImmediateStrategy.this.prettyLabel(state);
            String _format = String.format("ImmediateStrategy: Deleting tokens from state: %s.", _prettyLabel);
            AbstractImmediateStrategy.this.logger.debug(_format);
            AbstractImmediateStrategy.this.clear(state);
          }
        };
        IterableExtensions.<State>forEach(_filter, _function_1);
        InitState initState = automaton.getInitialState();
        boolean _isEmpty = AutomatonUtils.isEmpty(initState);
        if (_isEmpty) {
          AutomatonUtils.newEventToken(automaton, initState);
        }
      }
    };
    IterableExtensions.<Automaton>forEach(_filter, _function_1);
  }
  
  private String prettyLabel(final State state) {
    String _xblockexpression = null;
    {
      String _label = state.getLabel();
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(_label);
      if (_isNullOrEmpty) {
        return state.toString();
      }
      _xblockexpression = state.getLabel();
    }
    return _xblockexpression;
  }
  
  private void clear(final State state) {
    EList<EventToken> _eventTokens = state.getEventTokens();
    _eventTokens.clear();
  }
  
  private String id(final Automaton automaton) {
    return automaton.getEventPatternId();
  }
  
  private boolean needsReset(final Automaton automaton) {
    IEventModelManager _eventModelManager = this.getEventModelManager();
    List<Automaton> _enabledAutomataForTheLatestEvent = _eventModelManager.getEnabledAutomataForTheLatestEvent();
    boolean _contains = _enabledAutomataForTheLatestEvent.contains(automaton);
    return (!_contains);
  }
}
