/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.viatra.query.runtime.api.IMatchProcessor;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.testing.snapshot.BooleanSubstitution;
import org.eclipse.viatra.query.testing.snapshot.DateSubstitution;
import org.eclipse.viatra.query.testing.snapshot.DoubleSubstitution;
import org.eclipse.viatra.query.testing.snapshot.EMFSubstitution;
import org.eclipse.viatra.query.testing.snapshot.EnumSubstitution;
import org.eclipse.viatra.query.testing.snapshot.FloatSubstitution;
import org.eclipse.viatra.query.testing.snapshot.InputSpecification;
import org.eclipse.viatra.query.testing.snapshot.IntSubstitution;
import org.eclipse.viatra.query.testing.snapshot.LongSubstitution;
import org.eclipse.viatra.query.testing.snapshot.MatchRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSubstitutionRecord;
import org.eclipse.viatra.query.testing.snapshot.MiscellaneousSubstitution;
import org.eclipse.viatra.query.testing.snapshot.QuerySnapshot;
import org.eclipse.viatra.query.testing.snapshot.SnapshotFactory;
import org.eclipse.viatra.query.testing.snapshot.StringSubstitution;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Helper methods for dealing with snapshots and match set records.
 */
@SuppressWarnings("all")
public class SnapshotHelper {
  /**
   * Returns the actual value of the substitution based on its type
   */
  public Object derivedValue(final MatchSubstitutionRecord substitution) {
    Object _switchResult = null;
    boolean _matched = false;
    if (!_matched) {
      if (substitution instanceof BooleanSubstitution) {
        _matched=true;
        _switchResult = Boolean.valueOf(((BooleanSubstitution)substitution).isValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof DateSubstitution) {
        _matched=true;
        _switchResult = ((DateSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof DoubleSubstitution) {
        _matched=true;
        _switchResult = Double.valueOf(((DoubleSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof EMFSubstitution) {
        _matched=true;
        _switchResult = ((EMFSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof EnumSubstitution) {
        _matched=true;
        _switchResult = ((EnumSubstitution)substitution).getValueLiteral();
      }
    }
    if (!_matched) {
      if (substitution instanceof FloatSubstitution) {
        _matched=true;
        _switchResult = Float.valueOf(((FloatSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof IntSubstitution) {
        _matched=true;
        _switchResult = Integer.valueOf(((IntSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof LongSubstitution) {
        _matched=true;
        _switchResult = Long.valueOf(((LongSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof MiscellaneousSubstitution) {
        _matched=true;
        _switchResult = ((MiscellaneousSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof StringSubstitution) {
        _matched=true;
        _switchResult = ((StringSubstitution)substitution).getValue();
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns the EMF root that was used by the matchers recorded into the given snapshot,
   *  based on the input specification and the model roots.
   */
  public Notifier getEMFRootForSnapshot(final QuerySnapshot snapshot) {
    Notifier _xifexpression = null;
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.EOBJECT);
    if (_equals) {
      EObject _xifexpression_1 = null;
      EList<EObject> _modelRoots = snapshot.getModelRoots();
      int _size = _modelRoots.size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        EList<EObject> _modelRoots_1 = snapshot.getModelRoots();
        _xifexpression_1 = _modelRoots_1.get(0);
      }
      _xifexpression = _xifexpression_1;
    } else {
      Notifier _xifexpression_2 = null;
      InputSpecification _inputSpecification_1 = snapshot.getInputSpecification();
      boolean _equals_1 = Objects.equal(_inputSpecification_1, InputSpecification.RESOURCE);
      if (_equals_1) {
        Resource _xifexpression_3 = null;
        EList<EObject> _modelRoots_2 = snapshot.getModelRoots();
        int _size_1 = _modelRoots_2.size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          EList<EObject> _modelRoots_3 = snapshot.getModelRoots();
          EObject _get = _modelRoots_3.get(0);
          _xifexpression_3 = _get.eResource();
        }
        _xifexpression_2 = _xifexpression_3;
      } else {
        ResourceSet _xifexpression_4 = null;
        InputSpecification _inputSpecification_2 = snapshot.getInputSpecification();
        boolean _equals_2 = Objects.equal(_inputSpecification_2, InputSpecification.RESOURCE_SET);
        if (_equals_2) {
          Resource _eResource = snapshot.eResource();
          _xifexpression_4 = _eResource.getResourceSet();
        }
        _xifexpression_2 = _xifexpression_4;
      }
      _xifexpression = _xifexpression_2;
    }
    return _xifexpression;
  }
  
  /**
   * Returns the model roots that were used by the given ViatraQueryEngine.
   */
  public List<EObject> getModelRootsForEngine(final ViatraQueryEngine engine) {
    List<EObject> _switchResult = null;
    QueryScope _scope = engine.getScope();
    final QueryScope scope = _scope;
    boolean _matched = false;
    if (!_matched) {
      if (scope instanceof EMFScope) {
        _matched=true;
        Set<? extends Notifier> _scopeRoots = ((EMFScope)scope).getScopeRoots();
        final Function1<Notifier, List<EObject>> _function = new Function1<Notifier, List<EObject>>() {
          @Override
          public List<EObject> apply(final Notifier it) {
            List<EObject> _switchResult = null;
            boolean _matched = false;
            if (!_matched) {
              if (it instanceof EObject) {
                _matched=true;
                _switchResult = Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList(((EObject)it)));
              }
            }
            if (!_matched) {
              if (it instanceof Resource) {
                _matched=true;
                _switchResult = ((Resource)it).getContents();
              }
            }
            if (!_matched) {
              if (it instanceof ResourceSet) {
                _matched=true;
                EList<Resource> _resources = ((ResourceSet)it).getResources();
                final Function1<Resource, EList<EObject>> _function = new Function1<Resource, EList<EObject>>() {
                  @Override
                  public EList<EObject> apply(final Resource it) {
                    return it.getContents();
                  }
                };
                List<EList<EObject>> _map = ListExtensions.<Resource, EList<EObject>>map(_resources, _function);
                Iterable<EObject> _flatten = Iterables.<EObject>concat(_map);
                _switchResult = IterableExtensions.<EObject>toList(_flatten);
              }
            }
            return _switchResult;
          }
        };
        Iterable<List<EObject>> _map = IterableExtensions.map(_scopeRoots, _function);
        Iterable<EObject> _flatten = Iterables.<EObject>concat(_map);
        _switchResult = IterableExtensions.<EObject>toList(_flatten);
      }
    }
    if (!_matched) {
      _switchResult = Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList());
    }
    return _switchResult;
  }
  
  /**
   * Returns the input specification for the given matcher.
   */
  public InputSpecification getInputSpecificationForMatcher(final ViatraQueryMatcher matcher) {
    InputSpecification _switchResult = null;
    ViatraQueryEngine _engine = matcher.getEngine();
    QueryScope _scope = _engine.getScope();
    final QueryScope scope = _scope;
    boolean _matched = false;
    if (!_matched) {
      if (scope instanceof EMFScope) {
        _matched=true;
        InputSpecification _switchResult_1 = null;
        Set<? extends Notifier> _scopeRoots = ((EMFScope)scope).getScopeRoots();
        Notifier _head = IterableExtensions.head(_scopeRoots);
        boolean _matched_1 = false;
        if (!_matched_1) {
          if (_head instanceof EObject) {
            _matched_1=true;
            _switchResult_1 = InputSpecification.EOBJECT;
          }
        }
        if (!_matched_1) {
          if (_head instanceof Resource) {
            _matched_1=true;
            _switchResult_1 = InputSpecification.RESOURCE;
          }
        }
        if (!_matched_1) {
          if (_head instanceof ResourceSet) {
            _matched_1=true;
            _switchResult_1 = InputSpecification.RESOURCE_SET;
          }
        }
        _switchResult = _switchResult_1;
      }
    }
    if (!_matched) {
      _switchResult = InputSpecification.UNSET;
    }
    return _switchResult;
  }
  
  /**
   * Saves the matches of the given matcher (using the partial match) into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public MatchSetRecord saveMatchesToSnapshot(final ViatraQueryMatcher matcher, final IPatternMatch partialMatch, final QuerySnapshot snapshot) {
    final String patternFQN = matcher.getPatternName();
    final MatchSetRecord actualRecord = SnapshotFactory.eINSTANCE.createMatchSetRecord();
    actualRecord.setPatternQualifiedName(patternFQN);
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    _matchSetRecords.add(actualRecord);
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.UNSET);
    if (_equals) {
      EList<EObject> _modelRoots = snapshot.getModelRoots();
      ViatraQueryEngine _engine = matcher.getEngine();
      List<EObject> _modelRootsForEngine = this.getModelRootsForEngine(_engine);
      _modelRoots.addAll(_modelRootsForEngine);
      EList<EObject> _modelRoots_1 = snapshot.getModelRoots();
      _modelRoots_1.remove(snapshot);
      InputSpecification _inputSpecificationForMatcher = this.getInputSpecificationForMatcher(matcher);
      snapshot.setInputSpecification(_inputSpecificationForMatcher);
    }
    MatchRecord _createMatchRecordForMatch = this.createMatchRecordForMatch(partialMatch);
    actualRecord.setFilter(_createMatchRecordForMatch);
    final IMatchProcessor<IPatternMatch> _function = new IMatchProcessor<IPatternMatch>() {
      @Override
      public void process(final IPatternMatch match) {
        EList<MatchRecord> _matches = actualRecord.getMatches();
        MatchRecord _createMatchRecordForMatch = SnapshotHelper.this.createMatchRecordForMatch(match);
        _matches.add(_createMatchRecordForMatch);
      }
    };
    matcher.forEachMatch(partialMatch, _function);
    return actualRecord;
  }
  
  /**
   * Creates a match record that corresponds to the given match.
   *  Each parameter with a value is saved as a substitution.
   */
  public MatchRecord createMatchRecordForMatch(final IPatternMatch match) {
    final MatchRecord matchRecord = SnapshotFactory.eINSTANCE.createMatchRecord();
    List<String> _parameterNames = match.parameterNames();
    final Procedure1<String> _function = new Procedure1<String>() {
      @Override
      public void apply(final String param) {
        Object _get = match.get(param);
        boolean _notEquals = (!Objects.equal(_get, null));
        if (_notEquals) {
          EList<MatchSubstitutionRecord> _substitutions = matchRecord.getSubstitutions();
          Object _get_1 = match.get(param);
          MatchSubstitutionRecord _createSubstitution = SnapshotHelper.this.createSubstitution(param, _get_1);
          _substitutions.add(_createSubstitution);
        }
      }
    };
    IterableExtensions.<String>forEach(_parameterNames, _function);
    return matchRecord;
  }
  
  /**
   * Creates a match set record which holds the snapshot of a single matcher instance. It is also possible to enter
   * a filter for the matcher.
   */
  public <Match extends IPatternMatch> MatchSetRecord createMatchSetRecordForMatcher(final ViatraQueryMatcher<Match> matcher, final Match filter) {
    final MatchSetRecord matchSetRecord = SnapshotFactory.eINSTANCE.createMatchSetRecord();
    final IMatchProcessor<Match> _function = new IMatchProcessor<Match>() {
      @Override
      public void process(final Match match) {
        EList<MatchRecord> _matches = matchSetRecord.getMatches();
        MatchRecord _createMatchRecordForMatch = SnapshotHelper.this.createMatchRecordForMatch(match);
        _matches.add(_createMatchRecordForMatch);
      }
    };
    matcher.forEachMatch(filter, _function);
    return matchSetRecord;
  }
  
  /**
   * Creates a partial match that corresponds to the given match record.
   *  Each substitution is used as a value for the parameter with the same name.
   * @deprecated use createMatchForMatchRecord(IQuerySpecification, MatchRecord) instead
   */
  @Deprecated
  public IPatternMatch createMatchForMachRecord(final ViatraQueryMatcher matcher, final MatchRecord matchRecord) {
    final IPatternMatch match = matcher.newEmptyMatch();
    EList<MatchSubstitutionRecord> _substitutions = matchRecord.getSubstitutions();
    final Procedure1<MatchSubstitutionRecord> _function = new Procedure1<MatchSubstitutionRecord>() {
      @Override
      public void apply(final MatchSubstitutionRecord it) {
        Object target = SnapshotHelper.this.derivedValue(it);
        String _parameterName = it.getParameterName();
        match.set(_parameterName, target);
      }
    };
    IterableExtensions.<MatchSubstitutionRecord>forEach(_substitutions, _function);
    return match;
  }
  
  /**
   * Creates a partial match that corresponds to the given match record.
   *  Each substitution is used as a value for the parameter with the same name.
   */
  public <Match extends IPatternMatch> Match createMatchForMatchRecord(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final MatchRecord matchRecord) {
    IPatternMatch _newEmptyMatch = querySpecification.newEmptyMatch();
    final Match match = ((Match) _newEmptyMatch);
    EList<MatchSubstitutionRecord> _substitutions = matchRecord.getSubstitutions();
    final Procedure1<MatchSubstitutionRecord> _function = new Procedure1<MatchSubstitutionRecord>() {
      @Override
      public void apply(final MatchSubstitutionRecord it) {
        Object target = SnapshotHelper.this.derivedValue(it);
        String _parameterName = it.getParameterName();
        match.set(_parameterName, target);
      }
    };
    IterableExtensions.<MatchSubstitutionRecord>forEach(_substitutions, _function);
    return match;
  }
  
  /**
   * Saves all matches of the given matcher into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public MatchSetRecord saveMatchesToSnapshot(final ViatraQueryMatcher matcher, final QuerySnapshot snapshot) {
    IPatternMatch _newEmptyMatch = matcher.newEmptyMatch();
    return this.saveMatchesToSnapshot(matcher, _newEmptyMatch, snapshot);
  }
  
  /**
   * Returns the match set record for the given pattern FQN from the snapshot,
   *  if there is only one such record.
   */
  public MatchSetRecord getMatchSetRecordForPattern(final QuerySnapshot snapshot, final String patternFQN) {
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    final Function1<MatchSetRecord, Boolean> _function = new Function1<MatchSetRecord, Boolean>() {
      @Override
      public Boolean apply(final MatchSetRecord it) {
        String _patternQualifiedName = it.getPatternQualifiedName();
        return Boolean.valueOf(_patternQualifiedName.equals(patternFQN));
      }
    };
    final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
    int _size = IterableExtensions.size(matchsetrecord);
    boolean _equals = (_size == 1);
    if (_equals) {
      Iterator<MatchSetRecord> _iterator = matchsetrecord.iterator();
      return _iterator.next();
    }
    return null;
  }
  
  /**
   * Returns the match set records for the given pattern FQN from the snapshot.
   */
  public ArrayList<MatchSetRecord> getMatchSetRecordsForPattern(final QuerySnapshot snapshot, final String patternFQN) {
    final ArrayList<MatchSetRecord> matchSetRecords = new ArrayList<MatchSetRecord>();
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    final Function1<MatchSetRecord, Boolean> _function = new Function1<MatchSetRecord, Boolean>() {
      @Override
      public Boolean apply(final MatchSetRecord it) {
        String _patternQualifiedName = it.getPatternQualifiedName();
        return Boolean.valueOf(_patternQualifiedName.equals(patternFQN));
      }
    };
    Iterable<MatchSetRecord> _filter = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
    Iterables.<MatchSetRecord>addAll(matchSetRecords, _filter);
    return matchSetRecords;
  }
  
  /**
   * Creates a substitution for the given parameter name using the given value.
   *  The type of the substitution is decided based on the type of the value.
   */
  public MatchSubstitutionRecord createSubstitution(final String parameterName, final Object value) {
    MatchSubstitutionRecord _switchResult = null;
    boolean _matched = false;
    if (!_matched) {
      if (value instanceof EEnumLiteral) {
        _matched=true;
        EnumSubstitution _xblockexpression = null;
        {
          final EnumSubstitution sub = SnapshotFactory.eINSTANCE.createEnumSubstitution();
          String _literal = ((EEnumLiteral) value).getLiteral();
          sub.setValueLiteral(_literal);
          EEnum _eEnum = ((EEnumLiteral) value).getEEnum();
          sub.setEnumType(_eEnum);
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof EObject) {
        _matched=true;
        EMFSubstitution _xblockexpression = null;
        {
          final EMFSubstitution sub = SnapshotFactory.eINSTANCE.createEMFSubstitution();
          sub.setValue(((EObject) value));
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Integer) {
        _matched=true;
        IntSubstitution _xblockexpression = null;
        {
          final IntSubstitution sub = SnapshotFactory.eINSTANCE.createIntSubstitution();
          sub.setValue((((Integer) value)).intValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Long) {
        _matched=true;
        LongSubstitution _xblockexpression = null;
        {
          final LongSubstitution sub = SnapshotFactory.eINSTANCE.createLongSubstitution();
          sub.setValue((((Long) value)).longValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Double) {
        _matched=true;
        DoubleSubstitution _xblockexpression = null;
        {
          final DoubleSubstitution sub = SnapshotFactory.eINSTANCE.createDoubleSubstitution();
          sub.setValue((((Double) value)).doubleValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Float) {
        _matched=true;
        FloatSubstitution _xblockexpression = null;
        {
          final FloatSubstitution sub = SnapshotFactory.eINSTANCE.createFloatSubstitution();
          sub.setValue((((Float) value)).floatValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Boolean) {
        _matched=true;
        BooleanSubstitution _xblockexpression = null;
        {
          final BooleanSubstitution sub = SnapshotFactory.eINSTANCE.createBooleanSubstitution();
          sub.setValue((((Boolean) value)).booleanValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof String) {
        _matched=true;
        StringSubstitution _xblockexpression = null;
        {
          final StringSubstitution sub = SnapshotFactory.eINSTANCE.createStringSubstitution();
          sub.setValue(((String) value));
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Date) {
        _matched=true;
        DateSubstitution _xblockexpression = null;
        {
          final DateSubstitution sub = SnapshotFactory.eINSTANCE.createDateSubstitution();
          sub.setValue(((Date) value));
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      MiscellaneousSubstitution _xblockexpression = null;
      {
        final MiscellaneousSubstitution sub = SnapshotFactory.eINSTANCE.createMiscellaneousSubstitution();
        sub.setValue(value);
        sub.setParameterName(parameterName);
        _xblockexpression = sub;
      }
      _switchResult = _xblockexpression;
    }
    return _switchResult;
  }
  
  /**
   * Retrieve a human-readable string denoting the given record
   */
  protected String _prettyPrint(final MatchRecord record) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<MatchSubstitutionRecord> _substitutions = record.getSubstitutions();
      boolean _hasElements = false;
      for(final MatchSubstitutionRecord substitution : _substitutions) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _parameterName = substitution.getParameterName();
        _builder.append(_parameterName, "");
        _builder.append(" = ");
        Object _derivedValue = this.derivedValue(substitution);
        String _prettyPrint = null;
        if (_derivedValue!=null) {
          _prettyPrint=this.prettyPrint(_derivedValue);
        }
        _builder.append(_prettyPrint, "");
      }
    }
    return _builder.toString();
  }
  
  protected String _prettyPrint(final EObject obj) {
    String _xblockexpression = null;
    {
      final EClass eClass = obj.eClass();
      StringConcatenation _builder = new StringConcatenation();
      String _name = eClass.getName();
      _builder.append(_name, "");
      _builder.append(" (");
      {
        EList<EAttribute> _eAllAttributes = eClass.getEAllAttributes();
        for(final EAttribute attr : _eAllAttributes) {
          _builder.append(" ");
          String _name_1 = attr.getName();
          _builder.append(_name_1, "");
          _builder.append(" = ");
          Object _eGet = obj.eGet(attr);
          String _prettyPrint = null;
          if (_eGet!=null) {
            _prettyPrint=this.prettyPrint(_eGet);
          }
          _builder.append(_prettyPrint, "");
          _builder.append(" ");
        }
      }
      _builder.append(")");
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  protected String _prettyPrint(final Object obj) {
    return obj.toString();
  }
  
  public String prettyPrint(final Object record) {
    if (record instanceof MatchRecord) {
      return _prettyPrint((MatchRecord)record);
    } else if (record instanceof EObject) {
      return _prettyPrint((EObject)record);
    } else if (record != null) {
      return _prettyPrint(record);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(record).toString());
    }
  }
}
