/**
 * Copyright (c) 2010-2015, Balázs Grill, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Balázs Grill - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import org.apache.log4j.Appender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.viatra.query.testing.core.IMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.MatchSetRecordDiff;
import org.eclipse.viatra.query.testing.core.SnapshotHelper;
import org.eclipse.viatra.query.testing.core.TestingSeverityAggregatorLogAppender;
import org.eclipse.viatra.query.testing.snapshot.MatchRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.ComparisonFailure;

/**
 * @author Grill Balázs
 */
@SuppressWarnings("all")
public class ViatraQueryTestCase {
  private final static String SEVERITY_AGGREGATOR_LOGAPPENDER_NAME = (ViatraQueryTestCase.class.getName() + ".severityAggregatorLogAppender");
  
  public final static String UNEXPECTED_MATCH = "Unexpected match";
  
  public final static String EXPECTED_NOT_FOUND = "Expected match not found";
  
  private final ResourceSet resourceSet;
  
  private final List<IMatchSetModelProvider> modelProviders;
  
  @Extension
  private SnapshotHelper _snapshotHelper = new SnapshotHelper();
  
  private final TestingSeverityAggregatorLogAppender appender;
  
  public ViatraQueryTestCase() {
    ResourceSetImpl _resourceSetImpl = new ResourceSetImpl();
    this.resourceSet = _resourceSetImpl;
    LinkedList<IMatchSetModelProvider> _linkedList = new LinkedList<IMatchSetModelProvider>();
    this.modelProviders = _linkedList;
    Logger _logger = ViatraQueryLoggingUtil.getLogger(ViatraQueryEngine.class);
    final Appender a = _logger.getAppender(ViatraQueryTestCase.SEVERITY_AGGREGATOR_LOGAPPENDER_NAME);
    TestingSeverityAggregatorLogAppender _xifexpression = null;
    if ((a instanceof TestingSeverityAggregatorLogAppender)) {
      TestingSeverityAggregatorLogAppender _xblockexpression = null;
      {
        ((TestingSeverityAggregatorLogAppender)a).clear();
        _xblockexpression = ((TestingSeverityAggregatorLogAppender)a);
      }
      _xifexpression = _xblockexpression;
    } else {
      TestingSeverityAggregatorLogAppender _xblockexpression_1 = null;
      {
        final TestingSeverityAggregatorLogAppender na = new TestingSeverityAggregatorLogAppender();
        na.setName(ViatraQueryTestCase.SEVERITY_AGGREGATOR_LOGAPPENDER_NAME);
        Logger _logger_1 = ViatraQueryLoggingUtil.getLogger(ViatraQueryEngine.class);
        _logger_1.addAppender(na);
        _xblockexpression_1 = na;
      }
      _xifexpression = _xblockexpression_1;
    }
    this.appender = _xifexpression;
  }
  
  public void assertLogSeverityThreshold(final Level severity) {
    Level _severity = this.appender.getSeverity();
    int _int = _severity.toInt();
    int _int_1 = severity.toInt();
    boolean _greaterThan = (_int > _int_1);
    if (_greaterThan) {
      Level _severity_1 = this.appender.getSeverity();
      String _string = _severity_1.toString();
      String _plus = (_string + " message on log: ");
      LoggingEvent _event = this.appender.getEvent();
      String _renderedMessage = _event.getRenderedMessage();
      String _plus_1 = (_plus + _renderedMessage);
      Assert.fail(_plus_1);
    }
  }
  
  public void assertLogSeverity(final Level severity) {
    Level _severity = this.appender.getSeverity();
    Assert.assertEquals(severity, _severity);
  }
  
  public Resource loadModel(final URI uri) {
    return this.resourceSet.getResource(uri, true);
  }
  
  public void dispose() {
    final Procedure1<IMatchSetModelProvider> _function = new Procedure1<IMatchSetModelProvider>() {
      @Override
      public void apply(final IMatchSetModelProvider it) {
        it.dispose();
      }
    };
    IterableExtensions.<IMatchSetModelProvider>forEach(this.modelProviders, _function);
    this.modelProviders.clear();
    EList<Resource> _resources = this.resourceSet.getResources();
    final Procedure1<Resource> _function_1 = new Procedure1<Resource>() {
      @Override
      public void apply(final Resource it) {
        it.unload();
      }
    };
    IterableExtensions.<Resource>forEach(_resources, _function_1);
  }
  
  public <T extends EObject> void modifyModel(final Class<T> clazz, final Function1<? super T, ? extends Boolean> condition, final Procedure1<? super T> operation) {
    final Function1<IMatchSetModelProvider, Boolean> _function = new Function1<IMatchSetModelProvider, Boolean>() {
      @Override
      public Boolean apply(final IMatchSetModelProvider it) {
        boolean _updatedByModify = it.updatedByModify();
        return Boolean.valueOf((!_updatedByModify));
      }
    };
    final Iterable<IMatchSetModelProvider> nonIncrementals = IterableExtensions.<IMatchSetModelProvider>filter(this.modelProviders, _function);
    CollectionExtensions.<IMatchSetModelProvider>removeAll(this.modelProviders, nonIncrementals);
    final Procedure1<IMatchSetModelProvider> _function_1 = new Procedure1<IMatchSetModelProvider>() {
      @Override
      public void apply(final IMatchSetModelProvider it) {
        it.dispose();
      }
    };
    IterableExtensions.<IMatchSetModelProvider>forEach(nonIncrementals, _function_1);
    final LinkedList<T> elementsToModify = CollectionLiterals.<T>newLinkedList();
    final TreeIterator<Notifier> iterator = this.resourceSet.getAllContents();
    while (iterator.hasNext()) {
      {
        final Notifier element = iterator.next();
        boolean _isInstance = clazz.isInstance(element);
        if (_isInstance) {
          final T cast = clazz.cast(element);
          Boolean _apply = condition.apply(cast);
          if ((_apply).booleanValue()) {
            T _cast = clazz.cast(element);
            elementsToModify.add(_cast);
          }
        }
      }
    }
    for (final T element : elementsToModify) {
      operation.apply(element);
    }
    this.appender.clear();
  }
  
  public boolean addMatchSetModelProvider(final IMatchSetModelProvider matchSetModelProvider) {
    return this.modelProviders.add(matchSetModelProvider);
  }
  
  public <Match extends IPatternMatch> void assumeMatchSetsAreAvailable(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification) {
    final Procedure1<IMatchSetModelProvider> _function = new Procedure1<IMatchSetModelProvider>() {
      @Override
      public void apply(final IMatchSetModelProvider it) {
        try {
          try {
            it.<Match>getMatchSetRecord(ViatraQueryTestCase.this.resourceSet, querySpecification, null);
          } catch (final Throwable _t) {
            if (_t instanceof IllegalArgumentException) {
              final IllegalArgumentException e = (IllegalArgumentException)_t;
              Assume.assumeNoException(e);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      }
    };
    IterableExtensions.<IMatchSetModelProvider>forEach(this.modelProviders, _function);
  }
  
  public <Match extends IPatternMatch> void assertMatchSetsEqual(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification) {
    int _size = this.modelProviders.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      throw new IllegalArgumentException("At least two model providers shall be set");
    }
    final IMatchSetModelProvider reference = IterableExtensions.<IMatchSetModelProvider>head(this.modelProviders);
    Iterable<IMatchSetModelProvider> _tail = IterableExtensions.<IMatchSetModelProvider>tail(this.modelProviders);
    final Procedure1<IMatchSetModelProvider> _function = new Procedure1<IMatchSetModelProvider>() {
      @Override
      public void apply(final IMatchSetModelProvider it) {
        final MatchSetRecordDiff matchDiff = ViatraQueryTestCase.this.<Match>getMatchSetDiff(querySpecification, reference, it);
        boolean _isEmpty = matchDiff.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final Joiner joiner = Joiner.on("\n");
          String _string = matchDiff.toString();
          Set<MatchRecord> _additions = matchDiff.getAdditions();
          final Function1<MatchRecord, String> _function = new Function1<MatchRecord, String>() {
            @Override
            public String apply(final MatchRecord it) {
              return ViatraQueryTestCase.this._snapshotHelper.prettyPrint(it);
            }
          };
          Iterable<String> _map = IterableExtensions.<MatchRecord, String>map(_additions, _function);
          String _join = joiner.join(_map);
          Set<MatchRecord> _removals = matchDiff.getRemovals();
          final Function1<MatchRecord, String> _function_1 = new Function1<MatchRecord, String>() {
            @Override
            public String apply(final MatchRecord it) {
              return ViatraQueryTestCase.this._snapshotHelper.prettyPrint(it);
            }
          };
          Iterable<String> _map_1 = IterableExtensions.<MatchRecord, String>map(_removals, _function_1);
          String _join_1 = joiner.join(_map_1);
          throw new ComparisonFailure(_string, _join, _join_1);
        }
      }
    };
    IterableExtensions.<IMatchSetModelProvider>forEach(_tail, _function);
  }
  
  public void assertMatchSetsEqual(final IQueryGroup queryGroup) {
    Set<IQuerySpecification<?>> _specifications = queryGroup.getSpecifications();
    final Procedure1<IQuerySpecification<?>> _function = new Procedure1<IQuerySpecification<?>>() {
      @Override
      public void apply(final IQuerySpecification<?> it) {
        ViatraQueryTestCase.this.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<? extends ViatraQueryMatcher<IPatternMatch>>) it));
      }
    };
    IterableExtensions.<IQuerySpecification<?>>forEach(_specifications, _function);
  }
  
  public <Match extends IPatternMatch> MatchSetRecordDiff getMatchSetDiff(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final IMatchSetModelProvider expectedProvider, final IMatchSetModelProvider actualProvider) {
    try {
      MatchSetRecordDiff _xblockexpression = null;
      {
        Match filter = null;
        MatchSetRecord expected = expectedProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
        MatchRecord _filter = expected.getFilter();
        boolean _notEquals = (!Objects.equal(_filter, null));
        if (_notEquals) {
          MatchRecord _filter_1 = expected.getFilter();
          Match _createMatchForMatchRecord = this._snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, _filter_1);
          filter = _createMatchForMatchRecord;
        }
        final MatchSetRecord actual = actualProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
        MatchRecord _filter_2 = actual.getFilter();
        boolean _notEquals_1 = (!Objects.equal(_filter_2, null));
        if (_notEquals_1) {
          boolean _notEquals_2 = (!Objects.equal(filter, null));
          if (_notEquals_2) {
            throw new IllegalArgumentException(
              ((("Filter is provided by more than one sources: " + expectedProvider) + ", ") + actualProvider));
          } else {
            MatchRecord _filter_3 = actual.getFilter();
            Match _createMatchForMatchRecord_1 = this._snapshotHelper.<Match>createMatchForMatchRecord(querySpecification, _filter_3);
            filter = _createMatchForMatchRecord_1;
            MatchSetRecord _matchSetRecord = expectedProvider.<Match>getMatchSetRecord(this.resourceSet, querySpecification, filter);
            expected = _matchSetRecord;
          }
        }
        _xblockexpression = MatchSetRecordDiff.compute(expected, actual);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private <Match extends IPatternMatch> HashSet<String> compareMatchSets(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> querySpecification, final IMatchSetModelProvider expectedProvider, final IMatchSetModelProvider actualProvider) {
    final MatchSetRecordDiff matchdiff = this.<Match>getMatchSetDiff(querySpecification, expectedProvider, actualProvider);
    final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
    Set<MatchRecord> _additions = matchdiff.getAdditions();
    final Function1<MatchRecord, String> _function = new Function1<MatchRecord, String>() {
      @Override
      public String apply(final MatchRecord it) {
        String _prettyPrint = ViatraQueryTestCase.this._snapshotHelper.prettyPrint(it);
        String _plus = ((ViatraQueryTestCase.UNEXPECTED_MATCH + " (") + _prettyPrint);
        return (_plus + ")");
      }
    };
    Iterable<String> _map = IterableExtensions.<MatchRecord, String>map(_additions, _function);
    Iterables.<String>addAll(diff, _map);
    Set<MatchRecord> _removals = matchdiff.getRemovals();
    final Function1<MatchRecord, String> _function_1 = new Function1<MatchRecord, String>() {
      @Override
      public String apply(final MatchRecord it) {
        String _prettyPrint = ViatraQueryTestCase.this._snapshotHelper.prettyPrint(it);
        String _plus = ((ViatraQueryTestCase.EXPECTED_NOT_FOUND + " (") + _prettyPrint);
        return (_plus + ")");
      }
    };
    Iterable<String> _map_1 = IterableExtensions.<MatchRecord, String>map(_removals, _function_1);
    Iterables.<String>addAll(diff, _map_1);
    return diff;
  }
}
