/**
 * Copyright (c) 2010-2015, Balázs Grill, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Balázs Grill - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core.api;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.inject.Injector;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.log4j.Level;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.viatra.query.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Pattern;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.matchers.backend.IQueryBackendFactory;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.registry.IDefaultRegistryView;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryEntry;
import org.eclipse.viatra.query.runtime.registry.QuerySpecificationRegistry;
import org.eclipse.viatra.query.testing.core.PatternBasedMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.SnapshotMatchSetModelProvider;
import org.eclipse.viatra.query.testing.core.ViatraQueryTestCase;
import org.eclipse.viatra.query.testing.core.XmiModelUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * This class defines an API to easily construct test cases. The base conception is to provide
 * a set of match set sources (from a snapshot or from the execution of a pattern) and make assertions
 * on them being equal.
 */
@SuppressWarnings("all")
public class ViatraQueryTest {
  private final ViatraQueryTestCase testCase;
  
  private final List<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> patterns = new LinkedList<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>();
  
  private ViatraQueryTest() {
    ViatraQueryTestCase _viatraQueryTestCase = new ViatraQueryTestCase();
    this.testCase = _viatraQueryTestCase;
  }
  
  /**
   * Test the specified query
   */
  public static <Match extends IPatternMatch> ViatraQueryTest test(final IQuerySpecification<? extends ViatraQueryMatcher<Match>> pattern) {
    ViatraQueryTest _viatraQueryTest = new ViatraQueryTest();
    return _viatraQueryTest.and(pattern);
  }
  
  public static ViatraQueryTest test(final IQueryGroup patterns) {
    ViatraQueryTest _viatraQueryTest = new ViatraQueryTest();
    return _viatraQueryTest.and(patterns);
  }
  
  public static ViatraQueryTest test() {
    return new ViatraQueryTest();
  }
  
  /**
   * Test the specified query
   */
  public static <Match extends IPatternMatch> ViatraQueryTest test(final String pattern) {
    ViatraQueryTest _test = ViatraQueryTest.test();
    return _test.and(pattern);
  }
  
  public ViatraQueryTest and(final IQueryGroup patterns) {
    ViatraQueryTest _xblockexpression = null;
    {
      Set<IQuerySpecification<?>> _specifications = patterns.getSpecifications();
      final Procedure1<IQuerySpecification<?>> _function = new Procedure1<IQuerySpecification<?>>() {
        @Override
        public void apply(final IQuerySpecification<?> it) {
          ViatraQueryTest.this.patterns.add(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it));
        }
      };
      IterableExtensions.<IQuerySpecification<?>>forEach(_specifications, _function);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Test the given pattern parsed from file
   */
  public boolean and(final URI patternModel, final Injector injector, final String patternName) {
    try {
      boolean _xblockexpression = false;
      {
        final ResourceSet resourceSet = XmiModelUtil.prepareXtextResource(injector);
        final Resource resource = resourceSet.getResource(patternModel, true);
        EList<EObject> _contents = resource.getContents();
        boolean _isEmpty = _contents.isEmpty();
        boolean _not = (!_isEmpty);
        Preconditions.checkState(_not);
        EList<EObject> _contents_1 = resource.getContents();
        final EObject patternmodel = _contents_1.get(0);
        Preconditions.checkState((patternmodel instanceof PatternModel));
        EList<Pattern> _patterns = ((PatternModel) patternmodel).getPatterns();
        final Function1<Pattern, Boolean> _function = new Function1<Pattern, Boolean>() {
          @Override
          public Boolean apply(final Pattern it) {
            String _name = it.getName();
            return Boolean.valueOf(Objects.equal(_name, patternName));
          }
        };
        final Iterable<Pattern> patterns = IterableExtensions.<Pattern>filter(_patterns, _function);
        int _size = IterableExtensions.size(patterns);
        boolean _equals = (_size == 1);
        Preconditions.checkState(_equals);
        final SpecificationBuilder builder = new SpecificationBuilder();
        Pattern _get = ((Pattern[])Conversions.unwrapArray(patterns, Pattern.class))[0];
        IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> _orCreateSpecification = builder.getOrCreateSpecification(_get);
        _xblockexpression = this.patterns.add(_orCreateSpecification);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ViatraQueryTest and(final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> pattern) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.patterns.add(pattern);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  public ViatraQueryTest and(final String pattern) {
    ViatraQueryTest _xblockexpression = null;
    {
      IQuerySpecificationRegistry _instance = QuerySpecificationRegistry.getInstance();
      final IDefaultRegistryView view = _instance.getDefaultView();
      IQuerySpecificationRegistryEntry _entry = view.getEntry(pattern);
      IQuerySpecification<?> _get = _entry.get();
      _xblockexpression = this.and(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) _get));
    }
    return _xblockexpression;
  }
  
  private ViatraQueryTest(final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> pattern) {
    this();
    this.patterns.add(pattern);
  }
  
  /**
   * Add match result set with a query initialized using the given hints
   */
  public ViatraQueryTest with(final QueryEvaluationHint hint) {
    ViatraQueryTest _xblockexpression = null;
    {
      final PatternBasedMatchSetModelProvider modelProvider = new PatternBasedMatchSetModelProvider(hint);
      this.testCase.addMatchSetModelProvider(modelProvider);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Add match result set with a query initialized using the given query backend
   */
  public ViatraQueryTest with(final IQueryBackendFactory queryBackendFactory) {
    ViatraQueryTest _xblockexpression = null;
    {
      Map<String, Object> _emptyMap = CollectionLiterals.<String, Object>emptyMap();
      final QueryEvaluationHint hint = new QueryEvaluationHint(queryBackendFactory, _emptyMap);
      _xblockexpression = this.with(hint);
    }
    return _xblockexpression;
  }
  
  /**
   * Add match result set loaded from the given snapshot
   */
  public ViatraQueryTest with(final URI snapshotURI) {
    ViatraQueryTest _xblockexpression = null;
    {
      SnapshotMatchSetModelProvider _snapshotMatchSetModelProvider = new SnapshotMatchSetModelProvider(snapshotURI);
      this.testCase.addMatchSetModelProvider(_snapshotMatchSetModelProvider);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Add match result set loaded from the given snapshot
   */
  public ViatraQueryTest with(final String snapshotURI) {
    URI _resolvePlatformURI = XmiModelUtil.resolvePlatformURI(XmiModelUtil.XmiModelUtilRunningOptionEnum.BOTH, snapshotURI);
    return this.with(_resolvePlatformURI);
  }
  
  /**
   * Load input model
   */
  public ViatraQueryTest on(final URI inputURI) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.testCase.loadModel(inputURI);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Execute the given operation on the model. This call will also remove every non-incremental result set.
   */
  public <T extends EObject> ViatraQueryTest modify(final Class<T> clazz, final Function1<? super T, ? extends Boolean> condition, final Procedure1<? super T> operation) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.testCase.<T>modifyModel(clazz, condition, operation);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Execute all queries and check that the result sets are equal, then return itself to continue the testing. This
   * is separated from assertEquals because JUnit requires test methods to return void, therefore a test shall end with
   * assertEquals and this method shall be used where further actions are intended (e.g. incremental scenarios)
   */
  public ViatraQueryTest assertEqualsThen() {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertEquals();
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Assert that there were no log events with higher level than given severity during the execution
   */
  public void assertLogSeverityThreshold(final Level severity) {
    this.testCase.assertLogSeverityThreshold(severity);
  }
  
  /**
   * Assert that the highest level of log events occurred during execution equals to the given severity
   */
  public void assertLogSeverity(final Level severity) {
    this.testCase.assertLogSeverity(severity);
  }
  
  /**
   * Assert that there were no log events with higher level than given severity during the execution
   */
  public ViatraQueryTest assertLogSeverityThresholdThen(final Level severity) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertLogSeverityThreshold(severity);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Assert that the highest level of log events occurred during execution equals to the given severity
   */
  public ViatraQueryTest assertLogSeverityThen(final Level severity) {
    ViatraQueryTest _xblockexpression = null;
    {
      this.assertLogSeverity(severity);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
  
  /**
   * Execute all queries and check that the result sets are equal and no error log message was thrown
   */
  public void assertEquals() {
    this.assertEquals(Level.INFO);
  }
  
  /**
   * Execute all queries and check that the result sets are equal with a selected log level treshold
   */
  public void assertEquals(final Level treshold) {
    final Procedure1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function = new Procedure1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>() {
      @Override
      public void apply(final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it) {
        ViatraQueryTest.this.testCase.<IPatternMatch>assertMatchSetsEqual(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it));
      }
    };
    IterableExtensions.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>forEach(this.patterns, _function);
    this.testCase.assertLogSeverityThreshold(treshold);
  }
  
  /**
   * Make assumptions that each provided engine and snapshot can provide Match result set for each tested patterns
   */
  public ViatraQueryTest assumeInputs() {
    ViatraQueryTest _xblockexpression = null;
    {
      final Procedure1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _function = new Procedure1<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>() {
        @Override
        public void apply(final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> it) {
          ViatraQueryTest.this.testCase.<IPatternMatch>assumeMatchSetsAreAvailable(((IQuerySpecification<ViatraQueryMatcher<IPatternMatch>>) it));
        }
      };
      IterableExtensions.<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>>forEach(this.patterns, _function);
      _xblockexpression = this;
    }
    return _xblockexpression;
  }
}
