/*--------------------------------------------------------------------------
 * Copyright (c) 2004, 2006-2009 OpenMethods, LLC
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Trip Gilman (OpenMethods), Lonnie G. Pryor (OpenMethods)
 *    - initial API and implementation
 -------------------------------------------------------------------------*/
package org.eclipse.vtp.desktop.views.extended.pallet;

import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.nebula.widgets.pshelf.PShelf;
import org.eclipse.nebula.widgets.pshelf.PShelfItem;
import org.eclipse.nebula.widgets.pshelf.RedmondShelfRenderer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.vtp.desktop.core.Activator;
import org.eclipse.vtp.desktop.model.core.design.IDesign;
import org.eclipse.vtp.desktop.views.pallet.PallateProviderManager;
import org.eclipse.vtp.desktop.views.pallet.Pallet;
import org.eclipse.vtp.desktop.views.pallet.PalletItem;
import org.eclipse.vtp.desktop.views.pallet.PalletItemObserver;
import org.eclipse.vtp.desktop.views.pallet.PalletItemProvider;
import org.eclipse.vtp.desktop.views.pallet.PalletItemTransfer;

/**
 * This implementation of the <code>Pallet</code> interface provides a sectioned
 * view of the pallet categories.  Only one section is expanded at a time.  This
 * is very similar to the behavior of the "Outlook Bar" in the same named
 * email client.
 * 
 * @author trip
 */
public class ShelfPallet implements Pallet
{
	/** The container this pallet is mapped to. */
	private IDesign container = null;
	/** The UI sections indexed by the provider they observe. */
	private Map<PalletItemProvider, ProviderSection> sectionsByProvider = Collections.emptyMap();
	/**	The main UI composite for this pallet*/
	private Composite mainComp = null;
	/**	The instance of PShelf this pallet contains */
	private PShelf expandBar = null;

	/**
	 * Creates a new SimplePallet.
	 */
	public ShelfPallet()
	{
	}

	/**
	 * Sets the container this pallet is mapped to.
	 * 
	 * @param container The container this pallet is mapped to.
	 */
	public void setContainer(IDesign container)
	{
		this.container = container;
	}

	/**
	 * Creates the control used for this pallet.
	 * 
	 * @param parent The parent element that contains this pallet.
	 */
	public void createControl(Composite parent)
	{
		Map<PalletItemProvider, ProviderSection> sectionsByProvider = new TreeMap<PalletItemProvider, ProviderSection>(PALLATE_ITEM_PROVIDER_SORT);
		for(PalletItemProvider provider : PallateProviderManager.getPallateProviders())
		{
			int numItems = 0;
			List<PalletItem> items = provider.getPalletItems();
			for(PalletItem item : items)
			{
				if (item.canBeContainedBy(container))
					numItems++;
			}
			if(numItems > 0)
				sectionsByProvider.put(provider, new ProviderSection(provider));
		}
		this.sectionsByProvider = Collections.unmodifiableMap(sectionsByProvider);
		mainComp = new Composite(parent, SWT.NONE);
		mainComp.setLayout(new FillLayout());//new GridLayout(1, false));
		expandBar = new PShelf(mainComp, SWT.V_SCROLL);
		expandBar.setRenderer(new RedmondShelfRenderer());
//		expandBar.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, false));
		for (ProviderSection ps : this.sectionsByProvider.values())
			ps.initialize();
	}

	/**
	 * Returns the control used for this pallet.
	 * 
	 * @return The control used for this pallet.
	 */
	public Control getControl()
	{
		return mainComp;
	}

	/**
	 * Disposes this pallet.
	 */
	public void destroy()
	{
		for (ProviderSection ps : this.sectionsByProvider.values())
			ps.destroy();
		this.mainComp.dispose();
		this.mainComp = null;
		this.sectionsByProvider = Collections.emptyMap();
	}

	/**
	 * The UI control generated for a single provider.
	 * 
	 * @author Lonnie Pryor
	 */
	private final class ProviderSection extends LabelProvider implements
			IStructuredContentProvider, ITableLabelProvider, IMenuListener,
			DragSourceListener, PalletItemObserver
	{
		/** The provider in question. */
		final PalletItemProvider provider;
		/** The UI section. */
		PShelfItem section = null;
		/** The table viewer. */
		TableViewer viewer = null;

		/**
		 * Creates a new ProviderSection.
		 * 
		 * @param provider The provider in question.
		 */
		ProviderSection(PalletItemProvider provider)
		{
			this.provider = provider;
		}

		/**
		 * Initializes this section.
		 */
		void initialize()
		{
			section = new PShelfItem(expandBar, SWT.NONE);
			section.setText(provider.getName());
			Composite body = section.getBody();
			body.setLayout(new FillLayout());
			viewer = new TableViewer(body, SWT.V_SCROLL);
			viewer.setContentProvider(this);
			viewer.setLabelProvider(this);
			viewer.setSorter(PALLATE_ITEM_SORT);
			viewer.setInput(provider);
			viewer.addDragSupport(DND.DROP_COPY | DND.DROP_MOVE,
					new Transfer[] { PalletItemTransfer.getInstance() }, this);
			MenuManager menuMgr = new MenuManager("#PopupMenu");
			menuMgr.setRemoveAllWhenShown(true);
			menuMgr.addMenuListener(this);
			viewer.getControl().setMenu(
					menuMgr.createContextMenu(viewer.getControl()));
			provider.addPalletItemObserver(this);
		}

		/**
		 * Disposes this section.
		 */
		void destroy()
		{
			provider.removePalletItemObserver(this);
			section = null;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(
		 *      org.eclipse.jface.viewers.Viewer, java.lang.Object,
		 *      java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(
		 *      java.lang.Object)
		 */
		public Object[] getElements(Object inputElement)
		{
			List<PalletItem> palletItems = new LinkedList<PalletItem>();
			PalletItemProvider provider = (PalletItemProvider)inputElement;
			List<PalletItem> items = provider.getPalletItems();
			for(PalletItem item : items)
			{
				if (item.canBeContainedBy(container))
					palletItems.add(item);
			}
			return palletItems.toArray();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.BaseLabelProvider#dispose()
		 */
		public void dispose()
		{
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(
		 *      java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex)
		{
			return ((PalletItem)element).getName();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(
		 *      java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex)
		{
			final Image icon = ((PalletItem)element).getIcon();
			if (icon != null)
				return icon;
			return Activator.getDefault().getImageRegistry().get("ICON_MODULE");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(
		 *      org.eclipse.jface.action.IMenuManager)
		 */
		@SuppressWarnings("unchecked")
		public void menuAboutToShow(IMenuManager manager)
		{
			provider.createMenu(null, manager,
					(PalletItem[])((IStructuredSelection)viewer.getSelection()).toList()
							.toArray(new PalletItem[0]));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.swt.dnd.DragSourceListener#dragStart(
		 *      org.eclipse.swt.dnd.DragSourceEvent)
		 */
		public void dragStart(DragSourceEvent event)
		{
			event.doit = true;
			PalletItemTransfer.getInstance().setPalletItem(
					((PalletItem)((IStructuredSelection)viewer.getSelection())
							.getFirstElement()));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.swt.dnd.DragSourceListener#dragSetData(
		 *      org.eclipse.swt.dnd.DragSourceEvent)
		 */
		public void dragSetData(DragSourceEvent event)
		{
			if (PalletItemTransfer.getInstance().isSupportedType(event.dataType))
				event.data = PalletItemTransfer.getInstance().getPalletItem();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.swt.dnd.DragSourceListener#dragFinished(
		 *      org.eclipse.swt.dnd.DragSourceEvent)
		 */
		public void dragFinished(DragSourceEvent event)
		{
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.vtp.desktop.views.pallet.PalletItemObserver#
		 *      palletItemsChanged()
		 */
		public void palletItemsChanged()
		{
			try
            {
	            System.err.println("Viewer null? " + Boolean.toString(viewer == null));
	            if (viewer != null)
	            {
	            	viewer.getTable().getDisplay().asyncExec(new Runnable()
	            	{
	            		public void run()
	            		{
	            		viewer.refresh();
	            		}
	            	});
	            }
            }
            catch(RuntimeException e)
            {
	            e.printStackTrace();
            }
		}
	}

	/** A comparator that sorts item providers. */
	private static final Comparator<PalletItemProvider> PALLATE_ITEM_PROVIDER_SORT = new Comparator<PalletItemProvider>()
	{
		public int compare(PalletItemProvider left, PalletItemProvider right)
		{
			PalletItemProvider leftProvider = left;
			PalletItemProvider rightProvider = right;
			int diff = leftProvider.getRanking() - rightProvider.getRanking();
			if (diff == 0)
				diff = leftProvider.getName().compareTo(rightProvider.getName());
			return diff;
		}
	};
	/** A comparator that sorts items. */
	private static final ViewerSorter PALLATE_ITEM_SORT = new ViewerSorter()
	{
		public int compare(Viewer viewer, Object left, Object right)
		{
			PalletItem leftItem = (PalletItem)left;
			PalletItem rightItem = (PalletItem)right;
			return leftItem.getName().compareTo(rightItem.getName());
		}
	};

}
