package org.eclipse.vtp.editor.srgxml.ui.internal.wizards;
/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     
 *******************************************************************************/
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.vtp.editor.srgxml.ui.internal.Logger;

/**
 * Additional page for SRGSXML new file wizard asking for a rule name, list of
 * words and/or a file with a word list
 */
public class GrammarRulePage extends WizardPage {
	private String rname = new String();

	private String fname = new String();

	private String wlist = new String();

	private Text ruleName;

	private Text fileName;

	private Text wordList;

	private Label ruleNameLbl;

	private Button fileNameButton;

	private Button fileRadioButton;

	private Button textRadioButton;
	
	String rootRule = null;

	static final int LINE2OFFSET = 15;

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
	 */

	public GrammarRulePage(String pageName, String title,
			org.eclipse.jface.resource.ImageDescriptor titleImage) {
		super(pageName, title, titleImage);

	}

	/**
	 * Create a new page instance
	 * 
	 * @param pageName
	 *            name of page
	 * @param title
	 *            title
	 * @param titleImage
	 *            icon
	 * @param pageStatus
	 *            status
	 */
	public GrammarRulePage(String pageName, String title,
			org.eclipse.jface.resource.ImageDescriptor titleImage,
			boolean pageStatus) {
		super(pageName);
		if (pageStatus) {
			setPageComplete(pageStatus);
		}
	}

	/**
	 * @return Returns the fname.
	 */
	public String getFileName() {
		return fname;
	}

	/**
	 * @return Returns the rname.
	 */
	public String getRuleName() {
		return rname;
	}

	/**
	 * @return Returns the wlist.
	 */
	public String getWordList() {
		return wlist;
	}

	/**
	 * 
	 * @return ok
	 */
	public boolean performOk() {
		String namebuff = new String();

		rname = ruleName.getText();
		rootRule = rname;
		fname = fileName.getText();
		wlist = wordList.getText();

		if (!(ruleNameVerification(rname))) {
			setPageComplete(false);
			setErrorMessage(XMLWizardsMessages.SRGXMLPropertyPage_ruleNameError); //$NON-NLS-1$
			return false;
		} else {
			setErrorMessage(null);
		}

		if (((wlist.length() == 0) && (fname.length() == 0))
				|| ((wlist == null) && (fname == null))) {
			setPageComplete(true);
			return true;
		} else {
			setPageComplete(false);
		}

		if ((rname == null) || (rname.length() == 0)) {
			return false;
		} else {
			namebuff = rname.trim();
			if (namebuff.length() == 0) {
				return false;
			}

			rname = namebuff;
		}

		if (fileName.isEnabled()) {
			fname = fileName.getText();
			wlist = ""; //$NON-NLS-1$
			if ((fname == null) || (fname.length() == 0)) {
				return false;
			} else {
				namebuff = fname.trim();
				if (namebuff.length() == 0) {
					return false;
				}

				fname = namebuff;

				File theFile = new File(fname);
				FileInputStream fin;
				InputStreamReader in;
				BufferedReader buffin;

				String readbuff = new String();
				// Open file
				try {
					fin = new FileInputStream(theFile);
					in = new InputStreamReader(fin);
					buffin = new BufferedReader(in);

					// do {
					while ((readbuff = buffin.readLine()) != null) {
						// use an arraylist
						// readbuff = buffin.readLine();
						if (invalidChar(readbuff)) {
							// setPageComplete(false);
							setErrorMessage(XMLWizardsMessages.SRGXMLPropertyPage_fileError); 
							return false;
						} else {
							setErrorMessage(null);
						}
					}
				} catch (FileNotFoundException e) {
					Logger.logException("error opening SRGXML editor", e); //$NON-NLS-1$
				} catch (IOException e) {
					Logger.logException(e);
					e.printStackTrace();
				}
			}
		} else if (wordList.isEnabled()) {
			wlist = wordList.getText();

			fname = ""; //$NON-NLS-1$
			if ((wlist == null) || (wlist.length() == 0)) {
				return false;
			}
			if (invalidChar(wlist)) {
				// setPageComplete(false);
				setErrorMessage(XMLWizardsMessages.SRGXMLPropertyPage_wordListError); 
				return false;
			} else {
				setErrorMessage(null);
			}
		}

		setPageComplete(true);
		return true;
	}

	/**
	 * @param word word
	 * @return true if the word contains illegal characters ;{}()*?+
	 */
	private boolean invalidChar(String word) {
		String invalidChars = ";{}()*?+"; //$NON-NLS-1$
		if (in(word, invalidChars)) {
			return true;
		}
		return false;
	}

	/**
	 * @param st1	string to check
	 * @param st2	string with characters to check for in st1
	 * @return true if st1 contains any of the characters in st2
	 */
	private boolean in(String st1, String st2) {

		for (int i = 0; i < st2.length(); i++) {
			if (st1.indexOf(st2.charAt(i)) > 0) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Open a file browser
	 */
	private void browseDialog() {
		FileDialog dirDialog = new FileDialog(getShell(), SWT.OPEN);
		String temp = dirDialog.open();
		if ((temp != null) && (temp != "")) { //$NON-NLS-1$

			fileName.setText(temp);

			fileName.setFocus();
			setPageComplete(performOk());
		}
	}

	/**
	 * switch between using a file and a wordlist
	 * @param useWordList true for using a wordlist
	 */
	private void useWordList(boolean useWordList) {
		fileName.setEditable(!useWordList);
		fileName.setEnabled(!useWordList);

		wordList.setEditable(useWordList);
		wordList.setEnabled(useWordList);

		fileNameButton.setEnabled(!useWordList);

		wordList.setFocus();
		setPageComplete(performOk());		
	}
	
	/**
	 * set all widget states to reflect use of a file
	 */
	private void fileRadioDialog() {
		useWordList(false);
	}

	/**
	 * set all widget states to reflect use of a wordlist
	 */
	private void textRadioDialog() {
		useWordList(true);
	}

	/**
	 * create all the widgets
	 * @param group parent
	 */
	public void setFields(Composite group) {
		fileNameButton = new Button(group, SWT.PUSH | SWT.SINGLE);
		fileRadioButton = new Button(group, SWT.RADIO);
		textRadioButton = new Button(group, SWT.RADIO);

		fileName = new Text(group, SWT.SINGLE | SWT.BORDER);
		fileName.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_filename_tooltip); 
		ruleName = new Text(group, SWT.SINGLE | SWT.BORDER);
		ruleNameLbl = new Label(group, SWT.SINGLE);

		ruleNameLbl.setText(
				XMLWizardsMessages.SRGXMLPropertyPage_rulename); 
		ruleNameLbl.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_rulename_tooltip); 
		ruleName.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_rulename_tooltip); 
		fileNameButton.setText(
				XMLWizardsMessages.SRGXMLPropertyPage_browse); 
		fileNameButton.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_browse_tooltip); 
		fileRadioButton.setText(
				XMLWizardsMessages.SRGXMLPropertyPage_textfile); 
		fileRadioButton.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_rb_file_tooltip); 
		textRadioButton.setText(
				XMLWizardsMessages.SRGXMLPropertyPage_wordlist); 
		textRadioButton.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_rb_wordlist_tooltip); 
		wordList = new Text(group, SWT.MULTI | SWT.V_SCROLL | SWT.BORDER);
		wordList.setToolTipText(
				XMLWizardsMessages.GrammarRulePage_wordlist_tooltip); 
		ruleName.setFocus();
		ruleName.setText(
				XMLWizardsMessages.SRGXMLPropertyPage_defaultRule); 
		rname = ruleName.getText();

		wordList.setEditable(false);
		wordList.setEnabled(false);
		fileRadioButton.setSelection(true);
		ruleName.setFocus();

		fileNameButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				browseDialog();
			}
		});

		fileRadioButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				fileRadioDialog();
			}
		});

		textRadioButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				textRadioDialog();
			}
		});

		ruleName.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(performOk());
			}
		});

		fileName.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(performOk());
			}
		});

		wordList.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(performOk());
			}
		});
	}

	/**
	 * Set the layout constraints
	 * @param parent ui
	 */
	public void setFormLayout(Composite parent) {
		FormLayout layout = new FormLayout();
		layout.marginWidth = 3;
		layout.marginHeight = 3;

		parent.setLayout(layout);
		setFields(parent);

		FormData ruleNameLblFD = new FormData();
		ruleNameLblFD.top = new FormAttachment(0, 5);
		ruleNameLblFD.left = new FormAttachment(0, 5);
		ruleNameLbl.setLayoutData(ruleNameLblFD);

		FormData ruleNameFD = new FormData();
		ruleNameFD.top = new FormAttachment(0, 5);
		ruleNameFD.left = new FormAttachment(ruleNameLbl, 5);
		ruleNameFD.right = new FormAttachment(50, 0);
		ruleName.setLayoutData(ruleNameFD);

		FormData fileRadioButtonFD = new FormData();
		fileRadioButtonFD.top = new FormAttachment(ruleName, LINE2OFFSET);
		fileRadioButtonFD.left = new FormAttachment(0, 5);
		fileRadioButton.setLayoutData(fileRadioButtonFD);

		FormData fileNameFD = new FormData();
		fileNameFD.top = new FormAttachment(fileRadioButton, LINE2OFFSET);
		fileNameFD.left = new FormAttachment(0, 5);
		fileNameFD.right = new FormAttachment(40, 0);
		fileName.setLayoutData(fileNameFD);

		FormData fileNameButtonFD = new FormData();
		fileNameButtonFD.top = new FormAttachment(fileRadioButton, LINE2OFFSET);
		fileNameButtonFD.left = new FormAttachment(fileName, 10);
		fileNameButton.setLayoutData(fileNameButtonFD);

		FormData textRadioButtonFD = new FormData();
		textRadioButtonFD.top = new FormAttachment(fileName, LINE2OFFSET);
		textRadioButtonFD.left = new FormAttachment(0, 5);
		textRadioButton.setLayoutData(textRadioButtonFD);

		FormData wordListFD = new FormData();
		wordListFD.top = new FormAttachment(textRadioButton, LINE2OFFSET);
		wordListFD.left = new FormAttachment(0, 5);
		wordListFD.right = new FormAttachment(100, -5);
		wordListFD.bottom = new FormAttachment(100, -5);
		wordList.setLayoutData(wordListFD);

	}

	/**
	 * create the ui
	 * @param parent1 parent composite
	 */
	public void createControl(Composite parent1) {
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent1,
				"org.eclipse.vtp.editor.srgxml.doc" + ".word_list"); //$NON-NLS-1$ //$NON-NLS-2$
		Group grp = new Group(parent1, SWT.NULL);
		grp.setText(XMLWizardsMessages.SRGXMLPropertyPage_tojsgftitle); 
		setFormLayout(grp);

		setControl(grp);
		setPageComplete(true);
	}

	/**
	 * verify rulename
	 * @param rule rulename
	 * @return	true if ok
	 */
	public boolean ruleNameVerification(String rule) {

		StringBuffer letters;

		if (rule.length() == 0) {
			return true;
		}
		letters = new StringBuffer(rule);

		if (!(Character.isLetter(letters.charAt(0)))) {
			return false;
		}

		if (rule.indexOf(":") != -1) { //$NON-NLS-1$
			return false;
		}
		if (rule.indexOf(".") != -1) { //$NON-NLS-1$
			return false;
		}
		if (rule.indexOf("-") != -1) { //$NON-NLS-1$
			return false;
		}
		if (rule.compareTo("NULL") == 0) { //$NON-NLS-1$
			return false;
		}
		if (rule.compareTo("VOID") == 0) { //$NON-NLS-1$
			return false;
		}
		if (rule.compareTo("GARBAGE") == 0) { //$NON-NLS-1$
			return false;
		}
		return true;
	}

}
