/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.dialer.skype;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Bridge layer between the JNI code and the dialer, connection and call
 * classes.
 * 
 * This class abstracts the interaction with the JNI-provided elements of the
 * Skype layer. It manages the low-level initialization and library loading, as
 * well as setting up the messaging layer and providing the hook that the C++
 * code uses to pass events back to Java.
 * 
 * @author mgreenawalt
 */
public class SkDialer {

	private static final String NATIVE_LIB_ERR = Messages.SkDialer_0;

	private ISkDialerListener listener = null;

	private String connectResult = ""; //$NON-NLS-1$

	private String initFailReason = null; // holds reason from dll for init

	// failure

	private static boolean alreadyFinalized = false;

	private static SkDialer _instance = new SkDialer();

	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	private static final boolean DEBUGGING = SkypePlugin.getDefault()
			.isDebugging();

	static {
		if (DEBUGGING) {
			System.out.println("loading library"); //$NON-NLS-1$
		}
		System.loadLibrary("SkDialer"); //$NON-NLS-1$
	}

	public static SkDialer getDialer() {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkDialer.getDialer() called"); //$NON-NLS-1$
			System.out.flush();
		}
		return _instance;
	}

	private SkDialer() {
		super();
		if (DEBUGGING) {
			System.out.println(time.format(new Date()) + ": SkDialer create"); //$NON-NLS-1$
			System.out.flush();
		}
	}

	void setListener(final ISkDialerListener listenerObject) {
		listener = listenerObject;
	}

	void initializeSkDialer(final ISkDialerListener listenerObject)
			throws PhoneException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkDialer.initializeSkDialer() called"); //$NON-NLS-1$
			System.out.flush();
		}
		try {
			initFailReason = null;
			if (!initializeNative()) {
				if (initFailReason == null) {
					throw new PhoneException(
							Messages.SkDialer_2);
				} else {
					throw new PhoneException(initFailReason);
				}
			}
			setListener(listenerObject);
			alreadyFinalized = false;
			connectResult = ""; //$NON-NLS-1$
		} catch (UnsatisfiedLinkError ule) {
			throw new PhoneException(NATIVE_LIB_ERR, ule);
		}
	}

	// private Thread keepAlive = null;
	private static boolean keepAliveRunning = false;

	void connectSkDialer() throws PhoneException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkDialer.connectSkDialer() called"); //$NON-NLS-1$
			System.out.flush();
		}
		try {
			switch (connectNative()) {
			case -2:
				throw new PhoneException(
						Messages.SkDialer_4);
			case -1:
				throw new PhoneException(
						Messages.SkDialer_5);
			case 1:
				connectResult = "ATTACH CONNECTED"; //$NON-NLS-1$
			case 0:
				while ("".equals(connectResult)) { //$NON-NLS-1$
					try {
						Thread.sleep(500);
					} catch (InterruptedException e) {
					}
				}
				if ("ATTACH REFUSED".equals(connectResult)) { //$NON-NLS-1$
					throw new PhoneException(Messages.SkDialer_9);
				}
				break;
			}
			// start the heartbeat
			if (!keepAliveRunning) {
				(new Thread(pinger, "pinger")).start(); //$NON-NLS-1$
				keepAliveRunning = true;
			}

		} catch (UnsatisfiedLinkError ule) {
			throw new PhoneException(NATIVE_LIB_ERR, ule);
		}
	}

	void destroySkDialer() throws PhoneException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkDialer.destroySkDialer() called"); //$NON-NLS-1$
			System.out.flush();
		}
		if (alreadyFinalized) {
			return;
		}
		try {
			enableKeepAlive = false;
			keepAliveRunning = false;
			alreadyFinalized = true;
			destroyNative();
		} catch (UnsatisfiedLinkError ule) {
			throw new PhoneException(NATIVE_LIB_ERR, ule);
		}
	}

	void sendMessage(final String message) throws PhoneException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkDialer.sendMessage() called"); //$NON-NLS-1$
			System.out.flush();
		}
		try {
			if (!sendNative(message)) {
				throw new PhoneException(Messages.SkDialer_10);
			}
			if (message.startsWith("#tellme")) { //$NON-NLS-1$
				if (!sendNative("minimize")) { //$NON-NLS-1$
					throw new PhoneException(Messages.SkDialer_11);
				}
			}
		} catch (UnsatisfiedLinkError ule) {
			throw new PhoneException(NATIVE_LIB_ERR, ule);
		}
	}

	// initializes the dll. In particular detects whether Skype client
	// exists on the machine. Makes sure that Skype is running.
	protected native boolean initializeNative() throws UnsatisfiedLinkError;

	// connectNative creates the dll window and the listening thread
	// sends the first broadcast message to Skype
	// returns: -1 on failure, 0 on success first time, 1 if window already
	// created
	protected native int connectNative() throws UnsatisfiedLinkError;

	// // just sends another broadcast message to Skype. Used when the window is
	// in
	// // place but the connection did not complete.
	// protected native void discoverSkypeNative() throws UnsatisfiedLinkError;

	// tears down the windows structures in the dll. must do initialize again
	protected native void destroyNative() throws UnsatisfiedLinkError;

	// sendNative sends one string to Skype. caller formats
	synchronized protected native boolean sendNative(String message)
			throws UnsatisfiedLinkError;

	// pattern to recognize special messages from the DLL, not Skype
	private final Pattern dllMessage = Pattern.compile("DLL Message (\\d+):"); //$NON-NLS-1$

	// this is the principal callback method from the dll.
	// Messages from Skype are forwarded to the caller via the listener
	protected final void receiveMessageFromSkype(final String message)
			throws PhoneException {
		synchronized (synchObj) {
			if (DEBUGGING) {
				System.out
						.println(time.format(new Date())
								+ ": SkDialer.receiveMessageFromSkype() message: " + message); //$NON-NLS-1$
				System.out.flush();
			}
			if ("PONG".equals(message) && !pongAfterPing) { //$NON-NLS-1$
				pongAfterPing = true;
				return;
			}
			Matcher matcher = dllMessage.matcher(message);
			if (matcher.lookingAt()) {
				int dllMsgNumber = Integer.parseInt(matcher.group(1));
				String dllMsg = message.substring(matcher.end()).trim();
				initFailReason = dllMsg;
				return;
			}
			if (listener != null) {
				listener.receiveMessage(message);
			}
		}
	}

	// this method gets the callback from Skype when attempting to connect
	// Skype may indicate success or failure, or something in between.
	// If full success or full failure are encountered, messages are passed to
	// the caller via the listener.
	protected final void receiveConnectMessageFromSkype(final String message) {
		// "ATTACH CONNECTED" is full success. return message to caller
		if (DEBUGGING) {
			System.out.println(time.format(new Date()) + ": ConnectMessage: " //$NON-NLS-1$
					+ message);
			System.out.flush();
		}
		if ("ATTACH CONNECTED".equals(message)) { //$NON-NLS-1$
			synchronized (connectResult) {
				connectResult = message;
			}
		}
		// "ATTACH REFUSED" means user has refused access to Skype. send message
		// to caller
		if ("ATTACH REFUSED".equals(message)) { //$NON-NLS-1$
			synchronized (connectResult) {
				connectResult = message;
			}
		}
	}

	// pinger implements a 5-second heartbeat, sending PING message to Skype
	// method receiveMessageFromSkype partners to listen for the PONG response
	private String synchObj = ""; //$NON-NLS-1$

	private boolean pongAfterPing = true;

	private boolean enableKeepAlive = true;

	Runnable pinger = new Runnable() {
		public void run() {
			synchronized (synchObj) {
				while (true) {
					if (enableKeepAlive) {
						// method receiveMessageFromSkype sets pongAfterPing to
						// true when it hears a PONG message from Skype
						if (pongAfterPing) {
							pongAfterPing = false;
							// System.out.println(time.format(new Date())+":
							// Sending PING");
							if (!sendNative("PING")) { //$NON-NLS-1$
								if (DEBUGGING) {
									System.out
											.println("sendNative returns false"); //$NON-NLS-1$
								}
							}
							try {
								synchObj.wait(10000);
							} catch (InterruptedException e) {
							}
						} else {
							if (DEBUGGING) {
								System.out
										.println("No PONG received after PING"); //$NON-NLS-1$
								System.out.flush();
							}
							try {
								receiveMessageFromSkype("SKYPE DEAD"); //$NON-NLS-1$
							} catch (PhoneException exc) {
								// receiveMessageFromSkype will not throw
								// exception for this message
							}
							return;
						}

					} else {
						break;
					}
				}
			}
		}
	};

}
