/*******************************************************************************
 * Copyright (c) 2006 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.internal.tellme;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.vtp.launching.IVoiceXMLBrowserConstants;
import org.eclipse.vtp.launching.VoiceXMLBrowserProcess;
import org.eclipse.vtp.launching.VoiceXMLLogMessage;

/**
 * class CallState provides a shared resource that allows the independent
 * threads watching for call completion and fetching log entries to report their
 * status to the other.
 *
 * The launch is truly done when both threads have finished
 *
 * @author mgreenawalt
 *
 */
public class CallState {
    private static boolean callWatchThreadDone = false;
    private static boolean logFetchThreadDone = false;
    private static boolean logListThreadDone = false;
    private static boolean callWatchSuccessful = true;
    private static boolean logFetchSuccessful = true;
    private static boolean logListSuccessful = true;
    private static String callWatchReason = ""; //$NON-NLS-1$
    private static String logFetchReason = ""; //$NON-NLS-1$
    private static String logListReason = ""; //$NON-NLS-1$
    private static String callID = null; // studio call id for use in finish
    private static CallLog log = null;     // hold log between threads
    private static boolean stopButtonPushed = false;
    
    final private static SimpleDateFormat time =
        new SimpleDateFormat("mm:ss.SSS", Locale.getDefault());		//$NON-NLS-1$
    private final static boolean DEBUGGING =
        TellmePlugin.getDefault().isDebugging();
    
    private static Object synchObj = new Object();

    public static boolean launchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.launchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            return callWatchThreadDone && logListThreadDone && logFetchThreadDone;
        }
    }

    public static void initCallWatch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initCallWatch() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchThreadDone = false;
            callWatchSuccessful = true;
            callWatchReason = ""; //$NON-NLS-1$
        }
    }

    public static void initLogFetch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initLogFetch() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchThreadDone = false;
            logFetchSuccessful = true;
            logFetchReason = ""; //$NON-NLS-1$
        }
    }

    public static void initLogList() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.initLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListThreadDone = false;
            logListSuccessful = true;
            logListReason = ""; //$NON-NLS-1$
        }
    }

    public static void setCallWatchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setCallWatchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchThreadDone = true;
        }
    }

    public static void setLogFetchDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogFetchDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchThreadDone = true;
        }
    }

    public static void setLogListDone() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogListDone() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListThreadDone = true;
        }
    }

    public static Object getSynch() {
        return synchObj;
    }

    public static void setLog(final CallLog loglist) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        log = loglist;
    }

    public static CallLog getLog() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.getLogList() called");		//$NON-NLS-1$
            System.out.flush();
        }
        return log;
    }

    public static void reportCallWatchFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportCallWatchFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchSuccessful = false;
            callWatchReason = reason;
        }
    }

    public static void reportCallWatchSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportCallWatchSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            callWatchSuccessful = true;
            callWatchReason = reason;
        }
    }

    public static void reportLogFetchFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogFetchFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchSuccessful = false;
            logFetchReason = reason;
        }
    }

    public static void reportLogFetchSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogFetchSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logFetchSuccessful = true;
            logFetchReason = reason;
        }
    }

    public static void reportLogListFail(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogListFail() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListSuccessful = false;
            logListReason = reason;
        }
    }

    public static void reportLogListSucceed(final String reason) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.reportLogListSucceed() called");		//$NON-NLS-1$
            System.out.flush();
        }
        synchronized (synchObj) {
            logListSuccessful = true;
            logListReason = reason;
        }
    }

    public static void setCallID(final String callid) {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.setCallID() called");		//$NON-NLS-1$
            System.out.flush();
        }
        callID = callid;
    }

    public static void endLaunch() {
        if (DEBUGGING) {
            System.out.println(time.format(new Date()) +
                    ": CallState.endLaunch() called");		//$NON-NLS-1$
            StackTraceElement[] whereami = new Throwable().getStackTrace();
            for (int i = 1; i < Math.min(whereami.length, 2); i++){
            	System.out.println(whereami[i].toString());
            }
            System.out.flush();
        }
        // locate this browserprocess
        VoiceXMLBrowserProcess activeBrowser = null;
        final ILaunch[] launches = DebugPlugin.getDefault().getLaunchManager()
            .getLaunches();
        if (launches != null) {
            for (int i = 0; i < launches.length; i++) {
                final IProcess[] threads = launches[i].getProcesses();
                if (threads != null && threads.length > 0
                    && threads[0] instanceof VoiceXMLBrowserProcess) {
                    activeBrowser = (VoiceXMLBrowserProcess) threads[0];
                }
            }
        }

        // we get here if all threads are done
        if (callWatchSuccessful && logFetchSuccessful && logListSuccessful) {
//            DisplayActions.showEndOfCall(callID);
            
            String checkedID = "".equals(callID) ? Messages.DisplayActions_0 : callID; //$NON-NLS-1$    
			DebugEvent event[] = new DebugEvent[1];
			event[0] = new DebugEvent(activeBrowser.getVoiceXMLBrowser(), DebugEvent.MODEL_SPECIFIC, IVoiceXMLBrowserConstants.EVENT_LOG_MESSAGE);
			event[0].setData(new VoiceXMLLogMessage(new Date(), Messages.DisplayActions_7 + checkedID + Messages.DisplayActions_8));
			DebugPlugin.getDefault().fireDebugEventSet(event);
        }
        else {
            // to here means some or all threads had problem
            DisplayActions.showError(("OK".equals(callWatchReason)?"":callWatchReason) //$NON-NLS-1$ //$NON-NLS-2$
            		+(("".equals(callWatchReason)||"OK".equals(callWatchReason))?"":"\n") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            		+logListReason+("".equals(logListReason)?"":"\n")+(logListReason.equals(logFetchReason)?"":logFetchReason)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
        }
        
        if (activeBrowser.getVoiceXMLBrowser() != null) {
        	activeBrowser.getVoiceXMLBrowser().stop();
        }

        activeBrowser.setTerminated(true);
        DebugEvent[] event = new DebugEvent[2];
        event[0] = new DebugEvent(activeBrowser, DebugEvent.TERMINATE);
		event[1] = new DebugEvent(activeBrowser.getLaunch(), DebugEvent.CHANGE);
        DebugPlugin.getDefault().fireDebugEventSet(event);
        
    }
    
    public static void setStopping(boolean state){
    	if (DEBUGGING){
    		System.out.println(time.format(new Date()) +": CallState.setStopping() called"); //$NON-NLS-1$
    		System.out.flush();
    	}
    	stopButtonPushed = state;
    }
    
    public static boolean isStopping(){
    	if (DEBUGGING){
    		System.out.println(time.format(new Date()) +": CallState.isStopping() called: value = "+stopButtonPushed); //$NON-NLS-1$
    		System.out.flush();
    	}
    	return stopButtonPushed;
    }
}
