/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.server.extensions.internal.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.UserDefinedType;
import org.eclipse.wst.rdb.internal.models.sql.routines.Routine;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.schema.Sequence;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;
import org.eclipse.wst.rdb.internal.models.sql.tables.Trigger;
import org.eclipse.wst.rdb.server.extensions.internal.ServerExtensionsPlugin;


public class Utility {
    
    private static ConnectionInfo conInfo = null;
   public static final String IBMCLOUDSCAPE = "IBM Cloudscape";  //$NON-NLS-1$
   public static final String INFORMIX      = "Informix";        //$NON-NLS-1$
   public static final String ORACLE        = "Oracle";          //$NON-NLS-1$
   public static final String SQLSERVER     = "SQL Server";      //$NON-NLS-1$
   public static final String DERBY         = "Derby";           //$NON-NLS-1$
   public static final String DB2_UDB       = "DB2 UDB";         //$NON-NLS-1$
   public static final String DB2_ISERIES   = "DB2 UDB iSeries"; //$NON-NLS-1$
   public static final String DB2_ZSERIES   = "DB2 UDB zSeries"; //$NON-NLS-1$ 
   public static final String SYBASE        = "Sybase";          //$NON-NLS-1$
    
    
    
    public static DatabaseDefinition getDBDef(Database database) {
    	if (database == null){
    		return null;
    	}
    	return RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
    }    
   
    public static boolean isDB2Server(Database database) {
    	if (database == null) 
    		return false;
    	return isDB2(getDBDef(database)) || isCloudscape(getDBDef(database));     	
    } 
    
    public static boolean isDB2(DatabaseDefinition dbDef) {
    	String dbname = dbDef.getProduct();
    	return dbname.equals(DB2_UDB) ||  dbname.equals(DB2_ISERIES) || dbname.equals(DB2_ZSERIES);
    }
    
    public static boolean isCloudscape(DatabaseDefinition dbDef) {
    	return dbDef.getProduct().equals(IBMCLOUDSCAPE);     	
    }
    
    public static boolean isSybase(DatabaseDefinition dbDef) {
        return dbDef.getProduct().equals(SYBASE);     	
    }
    public static boolean isInformix(DatabaseDefinition dbDef) {
        return dbDef.getProduct().equals(INFORMIX);     	
    }
    
    public static boolean isOracle(DatabaseDefinition dbDef) {
        return dbDef.getProduct().equals(ORACLE);     	
    }
  
    public static boolean isSQLServer(DatabaseDefinition dbDef) {
        return dbDef.getProduct().equals(SQLSERVER);     	
    }
  
    public static boolean isDerby(DatabaseDefinition dbDef) {
        return dbDef.getProduct().equals(DERBY);     	
    }
       
    public static int getVendorType(Database database) {
    	DatabaseDefinition dbdef = getDBDef(database);
    	if (isDB2(dbdef))
    		return DatabaseTypes.DB2;
        if (isOracle(dbdef))
            return DatabaseTypes.ORACLE;
        if (isSybase(dbdef))
        	return DatabaseTypes.SYBASE;
        if (isSQLServer(dbdef))
        	return DatabaseTypes.SQL_SERVER;
        if (isInformix(dbdef))
        	return DatabaseTypes.INFORMIX;
        if (isDerby(dbdef) || isCloudscape(dbdef) )
        	return DatabaseTypes.DERBY;
        
        return DatabaseTypes.SQL99;

    }
   
    public static ConnectionInfo getConnectionInfo(Database database) {
    	if (database == null) 
    		return null;
    	conInfo = RDBCorePlugin.getDefault().getConnectionManager().getConnectionInfo(database);
    	return conInfo;
    }
    /**
     * Ask if the user really wants to drop the specified object from the database.
     * <p>
     * @param objName The name of the object to be dropped.
     * @param conName The database to drop it from.
     * @return True if we may drop the object.
     */
    public static int isOkayToDrop(String objName)
    {	
   	String[] options =
   	{
               IDialogConstants.YES_LABEL,
               IDialogConstants.YES_TO_ALL_LABEL,
               IDialogConstants.NO_LABEL,
               IDialogConstants.CANCEL_LABEL          
   	};
   	
   	MessageDialog mb = new MessageDialog(
   						new Shell(),
   						ServerExtensionsPlugin.getString("DROP_ITEM_TITLE", new Object[]{objName}), //$NON-NLS-1$
   						null,
   						ServerExtensionsPlugin.getString("DROP_MSG", new Object[]{objName}), //$NON-NLS-1$
   						MessageDialog.QUESTION,
   						options,
   						0);   
   	mb.open();
   	return mb.getReturnCode(); 
    }

    /**
     * return the fully qualified name of the given sql object
     * @param sqlObj
     * @return
     */
   public static String getSchemaQualifiedName(SQLObject sqlObj)
 	{
 		StringBuffer sb = new StringBuffer();
 		Schema schema = null;
 		if (sqlObj instanceof Table) 
 		{
 		   schema = ((Table)sqlObj).getSchema();
		}
		else if (sqlObj instanceof Routine)
		{
		   schema = ((Routine)sqlObj).getSchema();
		}
		else if (sqlObj instanceof Sequence)
		{
			schema = ((Sequence)sqlObj).getSchema();
		}
		else if (sqlObj instanceof UserDefinedType)
		{
		   schema = ((UserDefinedType)sqlObj).getSchema();
		}
 		
 		if( schema != null )
 			sb.append( schema.getName() ).append( "." ); //$NON-NLS-1$
 		
 		sb.append( sqlObj.getName() );
 		return sb.toString();
 	}
     
   /**
    * Walk up the sqlmodel containment tree to find the owning database.
    * <p>
    * @param aSqlObj Could be a table, routine, sequence or user defined type
    * @return the database containing the sql object.
    */
   public static Database navigateToDatabase(SQLObject aSqlObj) 
   {     	
      Database db = null;
      Schema s = null;
      if (aSqlObj != null) 
      {
         if (aSqlObj instanceof Table) 
         {
            s = ((Table)aSqlObj).getSchema();
         }
         else if (aSqlObj instanceof Routine) 
         {
            s = ((Routine)aSqlObj).getSchema();
         }
         else if (aSqlObj instanceof Sequence) 
         {
            s = ((Sequence)aSqlObj).getSchema();        			
         }
         else if (aSqlObj instanceof UserDefinedType) 
         {
            s = ((UserDefinedType)aSqlObj).getSchema();    			
         }   
         else if (aSqlObj instanceof Trigger) 
         {
            s = ((Trigger)aSqlObj).getSchema();    			
         }
         else if (aSqlObj instanceof Index) 
         {
            s = ((Index)aSqlObj).getSchema();    			
         }
         if (s != null) 
         {
            db = s.getDatabase();
         }
      }
      return db;
   }
     
    /**
     * Gets the source from a File.
     * Supports multiple encodings according to the introductory bytes
     * known as the signature or Byte Order Mark (BOM):
     * <dl>
     * <dt>FEFF   <dd>UTF-16BE (big endian), the default for writing UTF-16
     * <dt>FFFE   <dd>UTF-16LE (little endian)
     * <dt>EFBBBF <dd>UTF-8
     * <dt>Anything else <dd>The default encoding
     * </dl>
     * <p>
     * @param source A File object that should already be known to exist.
     */
   public static String getSourceFromFile(File source) {
      try {
         InputStream in = new FileInputStream(source);
         byte[] buf = new byte[4];
         int len = in.read(buf);
         in.close();
         if (len < 3) {
            // What about 1-2 byte files in the default encoding?
            if (len > 0 && buf[0] != (byte)254 && buf[0] != (byte)255) {
               char[] buffer1 = new char[4];
               StringBuffer sb1 = new StringBuffer(512);
               InputStream in1 = new FileInputStream(source);
               InputStreamReader sr1 = new InputStreamReader(in1, System.getProperty("file.encoding"));//$NON-NLS-1$
               Reader r1 = new BufferedReader(sr1);
               if ((len = r1.read(buffer1)) != -1) {
                  sb1.append(buffer1, 0, len);
               }
               in1.close();
               sr1.close();
               r1.close();
               return sb1.toString();
            }
            return null;
         }
         String encoding = null;
         if (buf[0] == (byte)254 && buf[1] == (byte)255) {
            encoding = "UTF-16";//$NON-NLS-1$
         }
         else if (buf[0] == (byte)255 && buf[1] == (byte)254) {
            encoding = "UTF-16";//$NON-NLS-1$
         }
         else if (buf[0] == (byte)239 && buf[1] == (byte)187 && buf[2] == (byte)191) {
            encoding = "UTF-8";//$NON-NLS-1$
         }
         else {
            encoding = System.getProperty("file.encoding");//$NON-NLS-1$
         }
         char[] buffer2 = new char[4096];
         StringBuffer sb2 = new StringBuffer(4096);
         InputStream in2 = new FileInputStream(source);
         InputStreamReader sr2 = new InputStreamReader(in2, encoding);
         Reader r2 = new BufferedReader(sr2);
         while ((len = r2.read(buffer2)) != -1) {
            sb2.append(buffer2, 0, len);
         }
         in2.close();
         sr2.close();
         r2.close();
         return sb2.toString();
      }
      catch (Exception e) {
         return null;
      }
   } 

}
