/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.connection;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Savepoint;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.JDBCManager.IConnectionThread;

public class ConnectionAdapter implements Connection {
	
	private static final HashMap testMap = new HashMap ();
	
	private static final String DEBUG_INITIALIZE = "Initialize"; //$NON-NLS-1$
	private static final String DEBUG_RELEASE = "Release"; //$NON-NLS-1$
	
	static
	{
		IExtensionRegistry pluginRegistry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = pluginRegistry.getExtensionPoint("org.eclipse.wst.rdb.core", "connectionTest"); //$NON-NLS-1$ //$NON-NLS-2$
		IExtension[] extensions = extensionPoint.getExtensions();
		for (int i = 0; i < extensions.length; ++i)
		{
			IConfigurationElement[] configElements = extensions[i].getConfigurationElements();
			for (int j = 0; j < configElements.length; ++j)
			{
				if (configElements[j].getName().equals("provider")) { //$NON-NLS-1$
					String product = configElements[j].getAttribute("product"); //$NON-NLS-1$
					try
					{
						IConnectionTest test = (IConnectionTest) configElements[j].createExecutableExtension("class"); //$NON-NLS-1$
						testMap.put(product, test);
					}
					catch (CoreException e)
					{
						IStatus status = new Status(
								IStatus.ERROR,
								RDBCorePlugin.getDefault().getBundle().getSymbolicName(),
								IStatus.ERROR,
								"Error loading Database Connection tests", //$NON-NLS-1$
								e);
						RDBCorePlugin.getDefault().getLog().log(status);
						continue;
					}
				}
			}
		}
	}
	
	private ConnectionInfoImpl info;
	private Connection connection;
	private int statementCount;
	private IConnectionThread connectionThread;
	private boolean debug;
	private boolean debugStatement;
	private boolean debugCount;
	private boolean forceOpen;
	
	private void debug (String statement)
	{
		if (this.debug)
		{
			System.out.println(statement);
		}
	}
	
	private void printStack (String message)
	{
		if (this.debugStatement)
		{
			Thread.dumpStack();
		}
		if (this.debugCount)
		{
			System.out.println(message + " ** Count -> " + this.statementCount);
		}
	}
	
	private void reopenNativeConnection () throws Exception
	{
		debug("Start Reopen Native -> " + this.info.getName());
		this.connectionThread = (IConnectionThread)this.connectionThread.clone();
		Connection tempConnection = this.connectionThread.activateConnection(this);
		if (tempConnection != null)
		{
			this.connection = tempConnection;
		}
		debug("End Reopen Native -> " + this.info.getName());
	}
	
	private void checkConnectionOpen ()
	{
		Statement statement = null;
		try
		{
			String product = this.info.getDatabaseDefinition().getProduct();
			if (testMap.containsKey(product))
			{
				IConnectionTest connectionTest = (IConnectionTest) testMap.get(product);
				statement = connection.createStatement();
				statement.executeQuery(connectionTest.getTestStatement());
			}
		}
		catch (SQLException e)
		{
			try
			{
				statement.close();
				statement = null;
				if (this.connection.isClosed())
				{
					debug("Start Reopen Native [checkConnectionOpen]");
					reopenNativeConnection ();
				}
			}
			catch (Exception e1)
			{
			}
		}
		finally
		{
			if (statement != null)
			{
				try
				{
					statement.close();
				}
				catch (SQLException e)
				{
				}
			}
		}
	}
    
    private void initializeStatementIntegrity () throws SQLException
    {
        statementCount++;
        try
        {
            initializeConnectionIntegrity();
        }
        catch (SQLException e)
        {
            statementCount--;
            throw e;
        }
    }
	
	private void initializeConnectionIntegrity () throws SQLException
	{
		try
		{
			if (this.info.hasTimedOut())
			{
				debug("Start Reopen Native [this.info.hasTimedOut()]");
				reopenNativeConnection();
			}
			else if (this.connection != null && connection.isClosed())
			{
				debug("Start Reopen Native [this.connection != null && connection.isClosed()]");
				reopenNativeConnection();
			}
			else
			{
				checkConnectionOpen();
			}
		}
		catch (Exception e)
		{
			throw new SQLException (e.getMessage());
		}
	}
	
	private synchronized void cleanUpConnectionThread ()
	{
		if (this.connectionThread != null)
		{
			this.connectionThread.interrupt();
		}
	}
	
	private synchronized Statement createStatementAdapter (Statement s) throws SQLException
	{
        return new StatementAdapter(this, s);
    }
	
	private synchronized CallableStatement createCallableStatementAdapter (CallableStatement s) throws SQLException
	{
        return new CallableStatementAdapter(this, s);
    }
	
	private synchronized PreparedStatement createPreparedStatementAdapter (PreparedStatement s) throws SQLException
	{
        return new PreparedStatementAdapter(this, s);
    }
	
	private void logException (SQLException e)
	{
	    IStatus status = new Status(IStatus.ERROR, RDBCorePlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR,
	            e.getClass().getName(),
	            e);
		RDBCorePlugin.getDefault().getLog().log(status);
		this.info.onSQLException(this, e);
	}
	
	public ConnectionAdapter(ConnectionInfo info, Connection connection) {
		this.info = (ConnectionInfoImpl) info;
		this.connection = connection;
		this.statementCount = 0;
		this.forceOpen = false;
		this.debug = ((ConnectionManagerImpl)RDBCorePlugin.getDefault().getConnectionManager()).isDebugging();
		this.debugStatement = ((ConnectionManagerImpl)RDBCorePlugin.getDefault().getConnectionManager()).isDebuggingStatement();
		this.debugCount = ((ConnectionManagerImpl)RDBCorePlugin.getDefault().getConnectionManager()).isDebuggingCountStatement();
	}

	public void setNativeConnection (Connection connection)
	{
		this.connection = connection;
	}
	
	public void setConnectionThread (IConnectionThread connectionThread)
	{
		this.connectionThread = connectionThread;
	}
	
	public INativeConnectionLock getNativeConnectionLock ()
	{
	    return new INativeConnectionLock ()
	    {
            public Connection lockJDBCConnection() throws SQLException
            {
                initializeConnectionIntegrity();
                debug (" -- Acquiring Native Connection Lock -- ");
                forceOpen = true;
                return getNativeConnection();
            }

            public void releaseLock()
            {
                forceOpen = false;
                debug (" -- Releasing Native Connection Lock -- ");
            }
	    };
	}
	
	public ConnectionInfo getConnectionInfo() {
		return this.info;
	}
	
	public Connection getNativeConnection() {
		return this.connection;
	}

	public synchronized void releaseStatement ()
	{
		if (this.statementCount > 0)
		{
			this.statementCount--;
		}
    	printStack (DEBUG_RELEASE);
	}

	public String nativeSQL(String arg0) throws SQLException 
    {
        initializeConnectionIntegrity();
		return nativeSQL(arg0);
	}

	public void setAutoCommit(boolean arg0) throws SQLException 
    {
        initializeConnectionIntegrity();
		connection.setAutoCommit(arg0);
	}

	public boolean getAutoCommit() throws SQLException 
    {
        initializeConnectionIntegrity();
		return connection.getAutoCommit();
	}

	public void commit() throws SQLException {
 		connection.commit();
	}

	public void rollback() throws SQLException {
		connection.rollback();
	}
	
	public synchronized boolean timeOut () throws SQLException
	{
		if (this.statementCount == 0 && !isClosed() && !forceOpen)
		{
			debug("Timed-out and Close JDBC Connection...");
			this.connection.close();
			cleanUpConnectionThread ();
			return true;
		}
		else if (this.statementCount != 0 && !forceOpen)
		{
			debug("Cannot close - Statement count -> " + this.statementCount);
		}
		else if (forceOpen)
		{
		    debug("** Force Open **");
		}
		return false;
	}

	public synchronized void close() throws SQLException {
	
		try
		{
			if (!isClosed())
			{
				debug ("*** Close JDBC Connection... *** Connection [" + this.connection.toString() + "] - Connection Name -> " + this.info.getName());
				this.connection.close();
				cleanUpConnectionThread ();
			}
		}
		finally
		{
			ConnectionManagerImpl.INSTANCE.disconnect(info, this);
			this.connectionThread = null;
			this.connection = null;
		}
	}

	public boolean isClosed() throws SQLException {
		return this.connection == null || connection.isClosed();
	}

	public DatabaseMetaData getMetaData() throws SQLException {
	    initializeConnectionIntegrity();
	    return connection.getMetaData();
	}

	public void setReadOnly(boolean arg0) throws SQLException {
		connection.setReadOnly(arg0);
	}

	public boolean isReadOnly() throws SQLException {
		return connection.isReadOnly();
	}

	public void setCatalog(String arg0) throws SQLException {
		connection.setCatalog(arg0);
	}

	public String getCatalog() throws SQLException {
		return connection.getCatalog();
	}

	public void setTransactionIsolation(int arg0) throws SQLException {
		connection.setTransactionIsolation(arg0);
	}

	public int getTransactionIsolation() throws SQLException {
		return connection.getTransactionIsolation();
	}

	public SQLWarning getWarnings() throws SQLException {
		return connection.getWarnings();
	}

	public void clearWarnings() throws SQLException {
		connection.clearWarnings();
	}

	public Map getTypeMap() throws SQLException {
		return connection.getTypeMap();
	}

	public void setTypeMap(Map arg0) throws SQLException {
		connection.setTypeMap(arg0);
	}

	public void setHoldability(int arg0) throws SQLException {
		connection.setHoldability(arg0);
	}

	public int getHoldability() throws SQLException {
		return connection.getHoldability();
	}

	public Savepoint setSavepoint() throws SQLException {
		return connection.setSavepoint();
	}

	public Savepoint setSavepoint(String arg0) throws SQLException {
		return connection.setSavepoint(arg0);
	}

	public void rollback(Savepoint arg0) throws SQLException {
		connection.rollback(arg0);
	}

	public void releaseSavepoint(Savepoint arg0) throws SQLException {
		connection.releaseSavepoint(arg0);
	}
    
    public synchronized Statement createStatement() throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createStatementAdapter(connection.createStatement());
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

    public synchronized PreparedStatement prepareStatement(String arg0) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

    public synchronized CallableStatement prepareCall(String arg0) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createCallableStatementAdapter(connection.prepareCall(arg0));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }
	
    public Statement createStatement(int arg0, int arg1) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createStatementAdapter(connection.createStatement(arg0, arg1));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

    public PreparedStatement prepareStatement(String arg0, int arg1, int arg2) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0, arg1, arg2));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

    public CallableStatement prepareCall(String arg0, int arg1, int arg2) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createCallableStatementAdapter(connection.prepareCall(arg0, arg1, arg2));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public Statement createStatement(int arg0, int arg1, int arg2) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createStatementAdapter(connection.createStatement(arg0, arg1, arg2));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public PreparedStatement prepareStatement(String arg0, int arg1, int arg2, int arg3) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0, arg1, arg2, arg3));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public CallableStatement prepareCall(String arg0, int arg1, int arg2, int arg3) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createCallableStatementAdapter(connection.prepareCall(arg0, arg1, arg2, arg3));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public PreparedStatement prepareStatement(String arg0, int arg1) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0, arg1));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public PreparedStatement prepareStatement(String arg0, int[] arg1) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0, arg1));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }

	public PreparedStatement prepareStatement(String arg0, String[] arg1) throws SQLException
    {
        try
        {
            initializeStatementIntegrity();
            printStack(DEBUG_INITIALIZE);
            return createPreparedStatementAdapter(connection.prepareStatement(arg0, arg1));
        }
        catch (SQLException e)
        {
            releaseStatement();
            logException(e);
            throw e;
        }
    }
}
