/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.connection;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.Map;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;

class JDBCManager
{
	private static final ConnectionManager manager = RDBCorePlugin.getDefault().getConnectionManager(); 
	
	public static final JDBCManager INSTANCE = new JDBCManager();
	
	private boolean debug;
	
	interface IConnectionThread extends Cloneable
	{
		public void connect () throws Exception;
		public void startTimeOutProcess () throws SQLException;
		public boolean timeOut () throws SQLException;
		public void interrupt();
		public Connection activateConnectionAdapter() throws Exception;
		public Connection activateConnection(ConnectionAdapter adapter) throws Exception;
		public Connection getConnection ();
		public boolean isAlive();
		public Object clone();
	}

	private class ConnectionThread extends Thread implements IConnectionThread
	{
		private ConnectionInfo info = null;
		private Driver driver = null;
		private Map loaderMap = null;
		private ConnectionAdapter connectionAdapter = null;
		private ConnectionAdapter startAdapter = null;
		private Connection nativeConnection = null;
		private Exception connectionException = null;
		private boolean stopped = false;
				
		private Connection getNewConnection () throws Exception
		{
			debug ("Create New Connection -> " + info.getName());
			Connection cnx = RDBCorePlugin.getDefault().getConnectionFactoryRegistry().getConnectionFactory(
					info.getDriverClassName()).connect(driver, (ClassLoader) loaderMap.get(info.getURL()), info);
			debug ("*** Open JDBC Connection... *** Connection [" + cnx.toString() + "] - Connection Name -> " + this.info.getName());
			return cnx;
		}
		
		private Connection waitForConnection () throws Exception
		{
			synchronized (this)
			{
				while (this.nativeConnection == null && !this.stopped)
				{
					try
					{
						wait(1000);
						debug("Waiting for Connection initialization...");
					}
					catch (InterruptedException e)
					{
					}
				}
			}
			
			if (this.nativeConnection != null)
			{
				debug ("Connection initialized!");
			}
			else
			{
				debug ("Connection failed to initialize!");
			}
			
			if (this.connectionException != null)
			{
				throw this.connectionException;
			}
			
			return getConnection();
		}
		
		public synchronized void connect () throws Exception
		{
			try
			{
				synchronized (this)
				{
					Connection cnn = getNewConnection();
					if (this.startAdapter == null)
					{
						this.connectionAdapter = new ConnectionAdapter (this.info, cnn);
						this.connectionAdapter.setConnectionThread(this);
					}
					else
					{
						this.connectionAdapter = this.startAdapter;
					}
					this.info.setTimedOut(false);
					this.nativeConnection = cnn;
					this.notifyAll();
				}
			}
			catch (Exception e)
			{
				this.connectionException = e;
				debug ("Connection interruption!");
				interrupt();
			}
		}
		
		public ConnectionThread (ConnectionInfo info, Driver driver, Map loaderMap)
		{
			super();
			this.info = info;
			this.driver = driver;
			this.loaderMap = loaderMap;
		}
		
		public Object clone()
		{
			return new ConnectionThread(this.info, this.driver, this.loaderMap);
		}
		
		public Connection activateConnection(ConnectionAdapter adapter) throws Exception
		{
			this.startAdapter = adapter;
			this.start();
			waitForConnection();
			return this.nativeConnection;
		}
		
		public Connection activateConnectionAdapter() throws Exception
		{
			this.start();
			return waitForConnection();
		}
		
		public void interrupt()
		{
			debug ("Interruption in progress for Connection -> " + info.getName());
			super.interrupt();
		}
		
		public synchronized boolean timeOut() throws SQLException
		{
			return this.connectionAdapter.timeOut ();
		}

		public synchronized void startTimeOutProcess() throws SQLException
		{
			synchronized (this)
			{
				try
				{
					debug ("Timed-out requested...");
					while (!timeOut())
					{
						wait(1000);
					}
					this.info.setTimedOut(true);
				}
				catch (InterruptedException e)
				{
					debug(e);
				}
			}
		}
		
		public synchronized Connection getConnection ()
		{
			return this.connectionAdapter;
		}
		
		public void run()
		{
			try
			{
				connect();
				int timeout = manager.getTimeout() != 0 ? manager.getTimeout() : manager.getDefaultTimeout();
				
				synchronized (this)
				{
					debug("Start wait...");
					wait(1000 * timeout);
					debug("Stop wait...");
					
					debug("Time expired - Start time out process...");
					startTimeOutProcess ();
				}
			}
			catch (InterruptedException e)
			{
				this.stopped = true;
				debug("*** Wait interrupted! force Close ***");
				debug(e);
				try
				{
					if (this.connectionAdapter != null)
					{
						this.connectionAdapter.close();
					}
				}
				catch (SQLException e1)
				{
					debug(e1);
				}
			}
			catch (Exception e)
			{
				debug("*** Run Exception! force Close ***");
				debug(e);
				try
				{
					if (this.connectionAdapter != null)
					{
						this.connectionAdapter.close();
					}
				}
				catch (SQLException e1)
				{
					debug(e1);
				}
			}
		}
	}
	
	private void debug (Exception e)
	{
		if (this.debug)
		{
			e.printStackTrace();
		}
	}
	
	private void debug (String statement)
	{
		if (this.debug)
		{
			System.out.println(statement);
		}
	}
	
	private JDBCManager()
	{
		this.debug = ((ConnectionManagerImpl)manager).isDebugging();
	}
	
	public Connection createConnection(final ConnectionInfo info, final Driver driver, final Map loaderMap) throws Exception
	{
		final ConnectionThread thread = new ConnectionThread(info, driver, loaderMap);
		return thread.activateConnectionAdapter();
	}
}
