/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.server.internal.ui.explorer.content;

import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.text.MessageFormat;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.widgets.Display;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.shared.UserIdentification;
import org.eclipse.wst.rdb.core.internal.ui.explorer.virtual.IConnectionNode;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.internal.core.util.RDBCorePluginConstants;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.server.internal.ui.explorer.actions.popup.WorkOffline;
import org.eclipse.wst.rdb.server.internal.ui.services.IServicesManager;
import org.eclipse.wst.rdb.server.internal.ui.util.resources.ResourceLoader;


/**
 * @author ljulien
 */
public class ServerExplorerConfiguration
{
	private static final RDBCorePlugin corePlugin = RDBCorePlugin.getDefault();
	
	private static final String ERROR_CONNECTION = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.CONNECTION.ERROR.MESSAGE"); //$NON-NLS-1$
	private static final String ERROR_CONN_TITLE = ResourceLoader.INSTANCE.queryString("DATATOOLS.SERVER.UI.CONNECTION.ERROR.TITLE"); //$NON-NLS-1$
	private static final String USER_NAME = "user";//$NON-NLS-1$
	private static final String PASSWORD = "password";//$NON-NLS-1$
	
	private UserNotLoggedException userException;
    private Connection userConnection = null;
	private Exception connectException = null;
    
	/**
	 * Will reidentify the user
	 */
	private void getLoginInformation (final IConnectionNode server, final ConnectionInfo info, final String userName) throws UserNotLoggedException
	{
	    userException = null;
	    Display.getDefault().asyncExec(new Runnable ()
        {
            public void run()
            {
        		UserIdentification idDialog = new UserIdentification (userName, null, info.getName());
        		if (idDialog.open() == Window.OK)
        		{
        		    String us = idDialog.getUserNameInformation();
        		    String pa = idDialog.getPasswordInformation();
        			info.getProperties().setProperty(USER_NAME, us == null ? "" : us); //$NON-NLS-1$
        			info.getProperties().setProperty(PASSWORD, pa == null ? "" : pa); //$NON-NLS-1$
        			
        				try
        				{
                            BusyIndicator.showWhile(Display.getCurrent(), new Runnable ()
                            {
                                public void run ()
                                {
                                    try
                                    {
                                        userConnection = info.connect();
                                    }
                                    catch (Exception e)
                                    {
                                        connectException = e;
                                    }
                                }
                            });
        				    if (userConnection == null)
        				    {
        				        throw new Exception(connectException);
        				    }
        				    info.setSharedConnection(userConnection);
        				    new DatabaseProviderHelper().setDatabase(userConnection, info, info.getDatabaseName());
        				    info.saveConnectionInfo();
        			        IServicesManager.INSTANCE.getServerExplorerContentService().updateSelection(new StructuredSelection (server));
        				}
        				catch (Exception e)
        				{
                            String s = ""; //$NON-NLS-1$
                            Throwable t = e;
                            while (t!=null) {
                                s += System.getProperties().getProperty("line.separator") + t.getMessage(); //$NON-NLS-1$
                                if (t instanceof SQLException && ((SQLException)t).getNextException()!=t)
                                    t = ((SQLException)t).getNextException();                                
                                else if (t.getCause()!=t)
                                    t = t.getCause();
                                else 
                                    t = null;
                            }
        				    String message = MessageFormat.format(ERROR_CONNECTION, new String [] {s});
        					if (MessageDialog.openQuestion(Display.getCurrent().getActiveShell(), ERROR_CONN_TITLE, message))
        					{
        					    WorkOffline offline = new WorkOffline ();
        					    offline.selectionChanged(null, new StructuredSelection (server));
        					    offline.run(null);
        					}
        					else
        					{
        					    userException = new UserNotLoggedException ();
        					}
        				}
        		}
        		else 
        		{
        			userException = new UserNotLoggedException ();
        		}
            }
       	});
	    if (userException != null)
	    {
	        throw userException;
	    }
	}
	
	/**
	 * Will save existing configurations available in the server Explorer
	 * @param connectionInfo - The connection info to save
	 * @param definition - Will be used to store product and version
	 * @param databases - The databases to store 
	 * @param serverNode - The server node
	 * @throws IOException 
	 */
	public void save (ConnectionInfo connectionInfo)
			throws IOException
	{
	    connectionInfo.saveConnectionInfo();
	}

	/**
	 * Will remove the Configuration file for this server
	 * @param server
	 */
	public void deleteServer (IConnectionNode serverNode)
	{
	    ConnectionInfo info = corePlugin.getConnectionManager().getConnectionInfo(serverNode.getName());
	    if (info != null)
	    {
	        corePlugin.getConnectionManager().removeConnectionInfo(serverNode.getName());
	    }
   	}
	
	public ConnectionInfo restoreConnectionInfo (IConnectionNode server, ConnectionInfo connectionInfo)
	{
        try
        {
            Database database = connectionInfo.getSharedDatabase();
            if (database == null)
            {
            	userConnection = this.reestablishConnect(server,connectionInfo);
            	if (userConnection == null) {
            		getLoginInformation (server, connectionInfo, connectionInfo.getUserName());
            	}
            }
            return connectionInfo;
        }
        catch (UserNotLoggedException e)
        {
            return null;
        }
	}
	
	/**
	 * @return The list of previously created Connections
	 */
	public ConnectionInfo [] getRestoredConnections ()
	{	    
	    return corePlugin.getConnectionManager().getAllNamedConnectionInfo();
	}
	
	/**
	 * @author ljulien
	 */
	private class UserNotLoggedException extends Exception
	{

		/**
		 * Comment for <code>serialVersionUID</code>
		 */
		private static final long serialVersionUID = -5743507206244947582L;
	}
	
	public Connection reestablishConnect(IConnectionNode server,ConnectionInfo connectionInfo)  {
		Connection connection = null;
		if (this.isPromptNeeded(connectionInfo)){
			return connection;
		}
		
		try {
			connection = connectionInfo.connect();
			connectionInfo.setSharedConnection(connection);
		    new DatabaseProviderHelper().setDatabase(connection, connectionInfo, connectionInfo.getDatabaseName());
		    connectionInfo.saveConnectionInfo();
	        IServicesManager.INSTANCE.getServerExplorerContentService().updateSelection(new StructuredSelection (server));
		} catch (Exception e) {
			
		}
		return connection;
	}
	
	private boolean isPromptNeeded(ConnectionInfo connectionInfo){
		if (RDBCorePlugin.getDefault().getPluginPreferences().getInt(RDBCorePluginConstants.PASSWORD_SCOPE)== RDBCorePluginConstants.CONNECTION_SCOPE
			|| connectionInfo.getPassword() == null ){
			return true;		
		}
		return false;
	}
}
