/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.derby.catalog;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Constraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.ForeignKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.SQLConstraintsPackage;
import org.eclipse.wst.rdb.internal.models.sql.constraints.UniqueConstraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.impl.ForeignKeyImpl;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;


public class DerbyCatalogForeignKey extends ForeignKeyImpl implements ICatalogObject {
	private static final long serialVersionUID = 3257285837955086384L;

	public void refresh() {
		this.eAnnotationLoaded = false;
		this.uniqueConstraintLoaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public EList getEAnnotations() {
		if(!this.eAnnotationLoaded) this.loadEAnnotations();
		return this.eAnnotations;
	}
	
	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}

	public Database getCatalogDatabase() {
		return this.getBaseTable().getSchema().getDatabase();
	}
	
	public UniqueConstraint getUniqueConstraint() {
		if (!this.uniqueConstraintLoaded) this.loadUniqueConstraint();
		return this.uniqueConstraint;
	}
	
	public EList getMembers() {
		if (!this.uniqueConstraintLoaded) this.loadUniqueConstraint();
		return this.members;
	}
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLConstraintsPackage.FOREIGN_KEY__EANNOTATIONS) {
			this.getEAnnotations();
		} else if(id == SQLConstraintsPackage.FOREIGN_KEY__UNIQUE_CONSTRAINT) {
			this.getUniqueConstraint();
		} else if(id == SQLConstraintsPackage.FOREIGN_KEY__MEMBERS) {
			this.getMembers();
		}
		
		return super.eIsSet(eFeature);
	}

	private synchronized void loadEAnnotations() {
		if(this.eAnnotationLoaded) return;
		this.eAnnotationLoaded = true;
		EList memberList = super.getEAnnotations();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		DerbyCatalogForeignKey.setAsIdentifyingRelatinship(this,this.isIdentifyingRelationship(this.getMembers()));
		
		this.eSetDeliver(deliver);		
	}
	
	private synchronized void loadUniqueConstraint() {
		if(this.uniqueConstraintLoaded) return;
		this.uniqueConstraintLoaded = true;
		Connection connection = this.getConnection();
		if(connection == null) return;
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			BaseTable table = this.getBaseTable();
			Schema schema = table.getSchema();
			String query = "SELECT PK.CONSTRAINTNAME, PT.TABLENAME, PS.SCHEMANAME"+ //$NON-NLS-1$
					" FROM SYS.SYSFOREIGNKEYS R, SYS.SYSCONSTRAINTS FK, SYS.SYSTABLES FT, SYS.SYSSCHEMAS FS," + //$NON-NLS-1$
					" SYS.SYSCONSTRAINTS PK, SYS.SYSTABLES PT, SYS.SYSSCHEMAS PS" + //$NON-NLS-1$
					" WHERE R.CONSTRAINTID = FK.CONSTRAINTID" + //$NON-NLS-1$
					" AND R.KEYCONSTRAINTID = PK.CONSTRAINTID" + //$NON-NLS-1$
					" AND FK.TABLEID = FT.TABLEID" + //$NON-NLS-1$
					" AND FT.SCHEMAID = FS.SCHEMAID" + //$NON-NLS-1$
					" AND FS.SCHEMANAME='" + schema.getName() + "'" + //$NON-NLS-1$ //$NON-NLS-2$
					" AND FT.TABLENAME='" + table.getName() + "'" + //$NON-NLS-1$ //$NON-NLS-2$
					" AND FK.CONSTRAINTNAME='" + this.getName() + "'"+ //$NON-NLS-1$
					" AND PK.TABLEID = PT.TABLEID" + //$NON-NLS-1$
					" AND PT.SCHEMAID = PS.SCHEMAID"; //$NON-NLS-1$
				
					
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query);
			while (r.next()) {
				final String refKeyName   = r.getString("CONSTRAINTNAME"); //$NON-NLS-1$
				final String refTabSchema = r.getString("SCHEMANAME").trim(); //$NON-NLS-1$
				final String refTabName   = r.getString("TABLENAME"); //$NON-NLS-1$
				UniqueConstraint uk = this.getConstraint(refTabSchema, refTabName, refKeyName);
				this.setUniqueConstraint(uk);
				this.setReferencedTable(uk.getBaseTable());
				loadMembers(connection,uk);
			}
			r.close();
			s.close();
			loadEAnnotations();
		}
		catch (Exception e) {
			System.out.println(e.toString());
		}
		
		this.eSetDeliver(deliver);		
	}
	
	private synchronized void loadMembers(Connection connection, UniqueConstraint uk){
		try {
			EList members = super.getMembers();
			ResultSet r = connection.getMetaData().getExportedKeys(null, uk.getBaseTable().getSchema().getName(),uk.getBaseTable().getName());
			Table table = this.getBaseTable();
			Schema schema = table.getSchema();

			while(r.next()) {
				final String fkSchema_Name= r.getString(6);
				final String fkTable_Name = r.getString(7);
				final String fkName = r.getString(12);
				if(fkSchema_Name.equals(schema.getName())
					&& fkTable_Name.equals(table.getName())
					&& fkName.equals(this.getName())) {
					String columnName = r.getString(8);
					Column column =DerbyCatalogTable.getColumn(table,columnName);
					members.add(column);
				}
			}
			r.close();
		}catch (Exception e){
		}
	}
	
	public static void setAsIdentifyingRelatinship(ForeignKey fk,boolean identifying){
		EAnnotation eAnnotation = fk.addEAnnotation(RDBCorePlugin.FK_MODELING_RELATIONSHIP);
		fk.addEAnnotationDetail(eAnnotation,RDBCorePlugin.FK_IS_IDENTIFYING_RELATIONSHIP,new Boolean(identifying).toString());

		fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_MULTIPLICITY, RDBCorePlugin.MANY);
		fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_ROLE_NAME, new String ());
		fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_MULTIPLICITY, (fk.getMembers().size() > 0) ? RDBCorePlugin.ZERO_TO_ONE : RDBCorePlugin.ONE);
		fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_ROLE_NAME, new String ());
	}

	
	private boolean isIdentifyingRelationship(EList columns){
		boolean isIdentifying = true;
		Iterator it = columns.iterator();
		while(it.hasNext()) {
			Column column = (Column) it.next();
			if(!column.isPartOfPrimaryKey()){
				isIdentifying = false;
				break;
			}
		}
		return isIdentifying;
	}
	
	
	private Schema getSchema(String schemaName) {
		Schema s = this.getBaseTable().getSchema();
		if(s.getName().equals(schemaName)) return s;
		Database d = s.getDatabase();
		Iterator it = d.getSchemas().iterator();
		while(it.hasNext()) {
			s = (Schema) it.next();
			if(s.getName().equals(schemaName)) return s;			
		}

		Schema schema = new DerbyCatalogSchema();
		schema.setName(schemaName);
		schema.setDatabase(d);

		return schema;
	}
	
	private Table getTable(String schemaName, String tableName) {
		Schema schema = this.getSchema(schemaName);
		Iterator it = schema.getTables().iterator();
		while(it.hasNext()) {
			Table table = (Table) it.next();
			if(table.getName().equals(tableName)) return table;			
		}

		Table table = new DerbyCatalogTable();
		table.setName(tableName);
		table.setSchema(schema);

		return table;
	}
	
	private UniqueConstraint getConstraint(String schemaName, String tableName, String constraintName) {
		BaseTable table = (BaseTable) this.getTable(schemaName, tableName);
		Iterator it = table.getConstraints().iterator();
		while(it.hasNext()) {
			Constraint constraint = (Constraint) it.next();
			if(constraint.getName().equals(constraintName)) return (UniqueConstraint) constraint;			
		}
		
		UniqueConstraint uniqueConstrain = new DerbyCatalogPrimaryKey();
		uniqueConstrain.setName(constraintName);
		uniqueConstrain.setBaseTable(table);
		
		return uniqueConstrain;		
	}
	
	private boolean eAnnotationLoaded=false;
	private boolean uniqueConstraintLoaded = false;

}
