/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.derby.catalog;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionFilter;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.models.sql.routines.Routine;
import org.eclipse.wst.rdb.internal.models.sql.routines.SQLRoutinesPackage;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLSchemaPackage;
import org.eclipse.wst.rdb.internal.models.sql.schema.impl.SchemaImpl;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;

public class DerbyCatalogSchema extends SchemaImpl implements ICatalogObject {
	private static final long serialVersionUID = 3257285846544429880L;
	
	public synchronized void refresh() {
		this.tablesLoaded = false;
		this.routineLoaded = false;
		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getDatabase();
		return ((DerbyCatalogDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getDatabase();
	}

	public EList getTables() {
		if(!this.tablesLoaded) this.loadTables();
		return this.tables;
	}
	
	public EList getRoutines() {
		if(!this.routineLoaded) this.loadRoutines();
		return this.routines;
	}
	

	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLSchemaPackage.SCHEMA__TABLES) {
			this.getTables();
		}
		else if(id == SQLSchemaPackage.SCHEMA__ROUTINES) {
			this.getRoutines();
		}
		return super.eIsSet(eFeature);
	}
	
	private synchronized void loadTables() {
		if(this.tablesLoaded) return;
		EList tableList = super.getTables();
		Object[] oldList = tableList.toArray();
		tableList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		ConnectionInfo connectionInfo = RDBCorePlugin.getDefault().getConnectionManager().getConnectionInfo(this.getDatabase());
		ConnectionFilter tableFilter = connectionInfo.getFilter(this.getName()+"::"+ConnectionFilter.TABLE_FILTER); //$NON-NLS-1$
		if (tableFilter == null) {  //default table filter
			tableFilter = connectionInfo.getFilter(ConnectionFilter.TABLE_FILTER);
		}
		ConnectionFilter viewFilter = connectionInfo.getFilter(this.getName()+"::"+ConnectionFilter.VIEW_FILTER); //$NON-NLS-1$
		if (viewFilter == null) {   //default view filter
			viewFilter = connectionInfo.getFilter(ConnectionFilter.VIEW_FILTER);
		}

		String filterStr = ""; //$NON-NLS-1$
		boolean hasFilter = tableFilter != null || viewFilter != null;
		if (hasFilter) {
			if (tableFilter != null) {
				filterStr += "TABLENAME " + tableFilter.getPredicate()  + " AND TABLETYPE='T' OR TABLETYPE='S'";  //$NON-NLS-1$//$NON-NLS-2$
			}
			else
			{
				filterStr += "TABLETYPE='T' OR TABLETYPE='S'"; //$NON-NLS-1$
			}
			
			if (viewFilter != null) {
				if (filterStr.length() >0) filterStr += " OR "; //$NON-NLS-1$
				filterStr += "TABLENAME " + viewFilter.getPredicate()  + " AND TABLETYPE='V' "; //$NON-NLS-1$ //$NON-NLS-2$
			}
			else
			{
				filterStr += " OR TABLETYPE='V' "; //$NON-NLS-1$
			}
			filterStr =  " AND (" + filterStr; //$NON-NLS-1$
			filterStr += ") "; //$NON-NLS-1$
		}

		String query = "SELECT TABLENAME,TABLETYPE" + //$NON-NLS-1$
		" FROM SYS.SYSTABLES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
		" WHERE A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
		" AND B.SCHEMANAME='"+ this.getName()+"'"; //$NON-NLS-1$ //$NON-NLS-2$

		if (hasFilter) {
			query += filterStr;
		}
		try {

			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
			
			while(r.next()) {				
				String tableName = r.getString("TABLENAME"); //$NON-NLS-1$
				String type      = r.getString("TABLETYPE"); //$NON-NLS-1$
				Table table;

				EClass metaclass=null;
				if (type.equals("T") ||type.equals("S")){  //$NON-NLS-1$//$NON-NLS-2$
					metaclass = SQLTablesPackage.eINSTANCE.getPersistentTable();
				} else if (type.equals("V")){ //$NON-NLS-1$
					metaclass = SQLTablesPackage.eINSTANCE.getViewTable();
				}

				Object element = DerbyCatalogSchema.findElement(oldList,tableName,metaclass);

				if (element != null) {
					table = (Table) element;
					tableList.add(table);
					((ICatalogObject)table).refresh();
				} else {
					if(type.equals("T") ||type.equals("S") ) {  //$NON-NLS-1$//$NON-NLS-2$
						table = new DerbyCatalogTable();
					}
					else if(type.equals("V")) { //$NON-NLS-1$
						table = new DerbyCatalogView();
					}
					else continue;
					table.setName(tableName);
					tableList.add(table);
				}
				
			}
			this.tablesLoaded = true;
			r.close();
			s.close();
		}
		catch (Exception e) {
		}
		
		this.eSetDeliver(deliver);
	}	

	private synchronized void loadRoutines() {
		if(this.routineLoaded) return;
		EList routineList = super.getRoutines();
		Object[] oldList = routineList.toArray();
		routineList.clear();
		Connection connection = this.getConnection();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			final String query = "SELECT ALIAS,ALIASTYPE" + //$NON-NLS-1$
								" FROM SYS.SYSALIASES A,SYS.SYSSCHEMAS B"+ //$NON-NLS-1$
								" WHERE A.ALIASTYPE IN ('P','F')" + //$NON-NLS-1$
								" AND A.SCHEMAID=B.SCHEMAID" + //$NON-NLS-1$
								" AND B.SCHEMANAME='"+ this.getName()+"'";  //$NON-NLS-1$//$NON-NLS-2$
			Statement s = connection.createStatement();
			ResultSet r = s.executeQuery(query); 
					
			while(r.next()) {
				String routineName = r.getString("ALIAS"); //$NON-NLS-1$
				String type      = r.getString("ALIASTYPE"); //$NON-NLS-1$
				Routine routine;

				EClass metaclass=null;
				if (type.equals("P")){ //$NON-NLS-1$
					metaclass = SQLRoutinesPackage.eINSTANCE.getProcedure();
				} else if (type.equals("F")){ //$NON-NLS-1$
					metaclass = SQLRoutinesPackage.eINSTANCE.getUserDefinedFunction();
				}

				Object element = DerbyCatalogSchema.findElement(oldList,routineName,metaclass);
				if (element != null) {
					routine = (Routine) element;
					((ICatalogObject)routine).refresh();
				} else {
				
					if(type.equals("P"))  //$NON-NLS-1$
						routine = new DerbyCatalogProcedure();
					else if(type.equals("F"))  //$NON-NLS-1$
						routine = new DerbyCatalogUserDefinedFunction();
					else continue;
					
					routine.setName(routineName);					
				}
				routineList.add(routine);
			}
			this.routineLoaded = true;
			r.close();
			s.close();
		}
		catch (Exception e) {
		}
		
		this.eSetDeliver(deliver);
	}	
	
	protected static Object findElement(Object[] list, String name,EClass metaclass){
		Object object = null;
		for (int i = 0; i < list.length; i++){
			SQLObject sqlObject = (SQLObject) list[i];
			if (sqlObject.getName().equals(name) && sqlObject.eClass() == metaclass){
				object = list[i];
				break;
			}
		}
		return object;
	}
	
	private boolean tablesLoaded = false;
	private boolean routineLoaded = false;
}
