/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.dialer.skype;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.vtp.internal.dialer.IVoipCall;
import org.eclipse.vtp.internal.dialer.IVoipConnection;
import org.eclipse.vtp.internal.dialer.IVoipDialerListener;
import org.eclipse.vtp.internal.dialer.VoipDialerException;

/**
 * 
 * 
 * @author rjray
 */
public class SkypeConnection implements IVoipConnection {
	final private IVoipDialerListener userListener;

	final private SkDialer skDialer;

	private int connectionStatus;

	private IVoipCall currentCall = null;

	// Format for debug printout timestamp
	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	// Control for debugging printout
	private static final boolean DEBUGGING = SkypePlugin.getDefault()
			.isDebugging();

	/**
	 * @param listener
	 * @param dialer
	 */
	SkypeConnection(IVoipDialerListener listener, SkDialer dialer) {
		super();
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeConnection create"); //$NON-NLS-1$
			System.out.flush();
		}

		userListener = listener;
		skDialer = dialer;
		connectionStatus = -1;
	}

	/**
	 * @throws VoipDialerException
	 */
	void connect() throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeConnection.connect() called"); //$NON-NLS-1$
			System.out.flush();
		}

		try {
			skDialer.connectSkDialer();
		} catch (PhoneException pe) {
			// Pass it up the chain with a local annotation, using the
			// user-listener
			connectionStatus = IVoipConnection.CONNECTION_ERROR;
			throw new VoipDialerException(userListener,
					"Unable to init SkDialer", pe);
		}

		connectionStatus = IVoipConnection.CONNECTION_OK;		
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipConnection#dial(java.lang.String)
	 */
	public IVoipCall dial(final String number) throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeConnection.dial() called"); //$NON-NLS-1$
			System.out.flush();
		}
		SkypeCall call = null;

		connectionStatus = IVoipConnection.CONNECTION_DIALING;
		try {
			call = new SkypeCall(userListener, skDialer, this);
			call.dialNumber(number);
		} catch (VoipDialerException ve) {
			connectionStatus = IVoipConnection.CONNECTION_ERROR;
			throw ve;
		}
		connectionStatus = IVoipConnection.CONNECTION_OK;

		currentCall = call;
		return call;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipConnection#status()
	 */
	public int status() {
		return connectionStatus;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipConnection#disconnect()
	 */
	public void disconnect() throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeConnection.disconnect() called"); //$NON-NLS-1$
			System.out.flush();
		}
		// If there is a call currently active, hang it up
		if ((currentCall != null)
				&& (currentCall.status() == IVoipCall.CALL_ACTIVE)) {
			currentCall.hangUp();
		}
		// tear down the dialer library
		try {
			skDialer.destroySkDialer();
		} catch (PhoneException exc) {
			throw new VoipDialerException(userListener,
					"Library linking error", exc);
		}
	}
}
