/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.internal.dialer.nullphone;

import java.io.PrintStream;

import org.eclipse.vtp.internal.dialer.IVoipCall;
import org.eclipse.vtp.internal.dialer.IVoipConnection;
import org.eclipse.vtp.internal.dialer.IVoipDialer;
import org.eclipse.vtp.internal.dialer.IVoipDialerFactory;
import org.eclipse.vtp.internal.dialer.IVoipDialerListener;
import org.eclipse.vtp.internal.dialer.VoipDialerException;
import org.eclipse.vtp.internal.dialer.VoipDialerListener;

/**
 * A null implementation of a "phone", based on the interfaces in the
 * org.eclipse.vtp.dialer namespace.
 *
 * This class provides simple implementations of the IVoipCall, IVoipConnection
 * and IVoipDialer interfaces. Each of the methods being implemented uses a
 * listener to log their steps. The listener is an extension of the
 * VoipDialerListener class that allows the user to provide a listener of their
 * own, while the internal one quietly gathers all logged messages in an
 * internal array.
 *
 * This class does nothing else, it exists only as a way for users to ensure
 * that their code is calling the set of dialer-related steps in the correct
 * order.
 *
 * @author rjray
 */
public class NullPhone implements IVoipCall, IVoipConnection, IVoipDialer,
    IVoipDialerFactory {
    private int currentStatus;
    private VoipDialerListener listener;
    private final int thisId;
    private final int thisRole;

    private static int idBase = 0;
    private static final int ROLE_DIALER     = 0;
    private static final int ROLE_CONNECTION = 1;
    private static final int ROLE_CALL       = 2;

    /**
     * Create an object in the "dialer" role. This is the only visible
     * constructor, as the user should utilize the connect() and dial()
     * methods to get objects in the other roles.
     */
    public NullPhone() {
        this(ROLE_DIALER);
    }

    /**
     * Create an object with the specified role.
     *
     * @param role
     */
    private NullPhone(final int role) {
        super();
        thisRole = role;
        thisId = ++idBase;

        switch (role) {
            case ROLE_DIALER:
                currentStatus = IVoipDialer.DIALER_NO_INIT;
                break;
            case ROLE_CONNECTION:
                currentStatus = IVoipConnection.CONNECTION_OK;
                break;
            case ROLE_CALL:
                currentStatus = IVoipCall.CALL_OK;
                break;
            default:
                break;
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipDialerFactory#getDialer()
     */
    public IVoipDialer getDialer() {
        return this;
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipCall#status()
     */
    public int status() {
        return currentStatus;
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipCall#sendDTMF(java.lang.String)
     */
    public void sendDTMF(final String dtmf) throws VoipDialerException {
        if (thisRole != ROLE_CALL)
        {
            throw new VoipDialerException(listener,
                    "Attempt to sendDTMF from non-call object (id: " + thisId +
                    ")");
        }
        listener.dialerMessage("DTMF " + dtmf + " <" + thisId + ">");
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipCall#sendDTMF(java.lang.String, int)
     */
    public void sendDTMF(final String dtmf, final int pause)
        throws VoipDialerException {
        if (thisRole != ROLE_CALL)
        {
            throw new VoipDialerException(listener,
                    "Attempt to sendDTMF from non-call object (id: " + thisId +
                    ")");
        }
        listener.dialerMessage("DTMF " + dtmf + " <" + thisId +
                "> (explicit pause of " + pause + "ms)");
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipCall#hangUp()
     */
    public void hangUp() throws VoipDialerException {
        if (thisRole != ROLE_CALL)
        {
            throw new VoipDialerException(listener,
                    "Attempt to hangUp from non-call object (id: " + thisId +
                    ")");
        }
        listener.dialerMessage("HANGUP <" + thisId + ">");
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipConnection#dial(java.lang.String)
     */
    public IVoipCall dial(final String number) throws VoipDialerException {
        NullPhone callObj;

        if (thisRole != ROLE_CONNECTION)
        {
            throw new VoipDialerException(listener,
                    "Attempt to dial from non-connection object (id: " +
                    thisId + ")");
        }

        callObj = new NullPhone(ROLE_CALL);
        listener.dialerMessage("DIAL " + number + "<" + thisId +
                ">, new call has ID: " + callObj.thisId);

        return callObj;
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipConnection#disconnect()
     */
    public void disconnect() throws VoipDialerException {
        if (thisRole != ROLE_CONNECTION)
        {
            throw new VoipDialerException(listener,
                    "Attempt to disconnect from non-connection object (id: " +
                    thisId + ")");
        }

        listener.dialerMessage("DISCONNECT <" + thisId + ">");
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipDialer#initialize()
     */
    public void initialize() throws VoipDialerException {
        initialize(new IVoipDialerListener() {
            public void dialerMessage(final String msg) {} });
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipDialer#initialize(org.eclipse.vtp.dialer.IVoipDialerListener)
     */
    public void initialize(final IVoipDialerListener theirListener)
            throws VoipDialerException {
        if (thisRole != ROLE_DIALER)
        {
            throw new VoipDialerException(listener,
                    "Attempt to initialize from non-dialer object (id: " +
                    thisId + ")");
        }

        currentStatus = IVoipDialer.DIALER_OK;
        listener = new LocalListener(theirListener);
        listener.dialerMessage("INIT <" + thisId + ">");
    }

    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipDialer#connect()
     */
    public IVoipConnection connect() throws VoipDialerException {
        NullPhone connectionObj;
        if (thisRole != ROLE_DIALER)
        {
            throw new VoipDialerException(listener,
                    "Attempt to connect from non-dialer object (id: " +
                    thisId + ")");
        }

        connectionObj = new NullPhone(ROLE_CONNECTION);
        listener.dialerMessage("CONNECT " + "<" + thisId +
                ">, new connection has ID: " + connectionObj.thisId);

        return connectionObj;
    }

    /**
     * Write all the messages that passed through the listener to the given
     * PrintStream. The full message history from the internal listener is
     * printed to the stream one message at a time, one line per message.
     *
     * @param dest The PrintStream to send lines to
     */
    public void writeMessageLog(final PrintStream dest)
    {
        final String[] messageHistory = listener.messageHistory();

        for (int idx = 0; idx < messageHistory.length; idx++)
        {
            dest.println(messageHistory[idx]);
        }
    }
    /* (non-Javadoc)
     * @see org.eclipse.vtp.dialer.IVoipCall#getConnection()
     */

    public IVoipConnection getConnection() {
    	return new NullPhone(ROLE_CONNECTION);
    }
    
    private class LocalListener extends VoipDialerListener {
        private IVoipDialerListener theirListener = null;

        LocalListener() {
            super();
        }

        LocalListener(IVoipDialerListener otherListener) {
            super();
            theirListener = otherListener;
        }

        public void dialerMessage(final String message) {
            super.dialerMessage(message);
            if (theirListener != null)
            {
                theirListener.dialerMessage(message);
            }
        }
    }
    
}
