/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.launching.tellme;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.vtp.launching.IVoiceXMLBrowser;
import org.eclipse.vtp.launching.IVoiceXMLBrowserConstants;
import org.eclipse.vtp.launching.VoiceXMLBrowserInput;
import org.eclipse.vtp.launching.VoiceXMLBrowserProcess;
import org.eclipse.vtp.launching.VoiceXMLLogMessage;
import org.eclipse.vtp.launching.tellme.CallLog;
import org.eclipse.vtp.launching.tellme.LogItem;
import org.eclipse.vtp.launching.tellme.LogItemReader;
import org.eclipse.vtp.launching.tellme.TellmeBrowserException;

/**
 * @author mgreenawalt
 * 
 */

public class GetLogItems implements Runnable {

	final private static SimpleDateFormat logDF = new SimpleDateFormat(
			"MM/dd/yy HH:mm:ss", Locale.getDefault()); //$NON-NLS-1$

	final private static SimpleDateFormat time = new SimpleDateFormat(
			"mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	private final static boolean DEBUGGING = TellmePlugin.getDefault()
			.isDebugging();

	public void run() {
		CallState.initLogFetch();
		IVoiceXMLBrowser activeBrowser = null;
		final ILaunch[] launches = DebugPlugin.getDefault().getLaunchManager()
				.getLaunches();
		if (launches == null) {
			CallState
					.reportLogFetchFail("Fatal problem, no launches in the LaunchManager");
			synchronized (CallState.getSynch()) {
				CallState.setLogFetchDone();
				if (DEBUGGING) {
					System.out.println(time.format(new Date())
							+ "calling launchDone from logFetch on failure"); //$NON-NLS-1$
					System.out.flush();
				}
				if (CallState.launchDone()) {
					CallState.endLaunch();
				}
			}
			return;
		} else {
			for (int i = 0; i < launches.length; i++) {
				IProcess[] t = launches[i].getProcesses();
				if (t != null && t.length > 0
						&& t[0] instanceof VoiceXMLBrowserProcess) {
					activeBrowser = ((VoiceXMLBrowserProcess) t[0])
							.getVoiceXMLBrowser();
				}
			}
		}

		/**
		 * if the call to Studio was successful, pull the debug log over
		 * 
		 * first, get the list of available logs
		 */
		CallLog theLog = null;

		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": GetLogItems started"); //$NON-NLS-1$
			System.out.flush();
		}
		final StringBuffer logList = CallState.getLogList();
		if (logList == null) {
			CallState.reportLogFetchFail("No log created by Studio");
			synchronized (CallState.getSynch()) {
				CallState.setLogFetchDone();
				if (DEBUGGING) {
					System.out
							.println(time.format(new Date())
									+ "calling launchDone from logFetch on failure (no log)"); //$NON-NLS-1$
					System.out.flush();
				}
				if (CallState.launchDone()) {
					CallState.endLaunch();
				}
			}
			return;
		} else {
			theLog = StudioInteractions.getFirstLogFromList(logList);
			try {
				final LogItemReader theLogReader = StudioInteractions
						.getLogItemReader(theLog);
				LogItem theItem = theLogReader.nextLogItem();
				if (theItem == null) {
					DebugEvent event[] = new DebugEvent[1];
					event[0] = new DebugEvent(activeBrowser,
							DebugEvent.MODEL_SPECIFIC,
							IVoiceXMLBrowserConstants.EVENT_LOG_MESSAGE);
					event[0].setData(new VoiceXMLLogMessage(new Date(),
							"No log events returned from Studio"));
					DebugPlugin.getDefault().fireDebugEventSet(event);
					CallState
							.reportLogFetchFail("No log events returned from Studio");
					synchronized (CallState.getSynch()) {
						CallState.setLogFetchDone();
						if (DEBUGGING) {
							System.out
									.println(time.format(new Date())
											+ "calling launchDone from logFetch on failure (no log events)"); //$NON-NLS-1$
							System.out.flush();
						}
						if (CallState.launchDone()) {
							CallState.endLaunch();
						}
					}
					return;
				}
				Date logDate = null;
				while (theItem != null && !CallState.isStopping()) {
					if (DEBUGGING) {
						System.out.println(time.format(new Date())
								+ ": log item: \t" + theItem.getKind() + "\t" //$NON-NLS-1$//$NON-NLS-2$
								+ theItem.getTime() + "\t" + theItem.getBody()); //$NON-NLS-1$
						System.out.flush();
					}
					// convert string timestamp to Date object
					try {
						logDate = logDF.parse(theItem.getTime());
					} catch (ParseException e) {
						DebugPlugin.log(new Status(IStatus.ERROR,
								"org.eclipse.vtp.launching.tellme", //$NON-NLS-1$
								IStatus.ERROR,
								"Failed to parse datetime in log message", e));
					}

					DebugEvent event[] = new DebugEvent[1];
					event[0] = new DebugEvent(activeBrowser,
							DebugEvent.MODEL_SPECIFIC,
							IVoiceXMLBrowserConstants.EVENT_LOG_MESSAGE);
					event[0].setData(new VoiceXMLLogMessage(logDate, theItem
							.getBody()));
					DebugPlugin.getDefault().fireDebugEventSet(event);

					if ("SESSION_DONE".equals(theItem.getBody())) { //$NON-NLS-1$
						break;
					} else {
						theItem = theLogReader.nextLogItem();
					}
				}
			} catch (TellmeBrowserException e) {
				DebugPlugin.log(new Status(IStatus.ERROR,
						"org.eclipse.vtp.launching.tellme", IStatus.ERROR, //$NON-NLS-1$
						"Possible HTTP failure of some kind", e));
			}

			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": no more log items"); //$NON-NLS-1$
				System.out.flush();
			}

			// DisplayActions.showDebugLogURL(theLog);

		}

		if (CallState.isStopping()){
			CallState.reportLogFetchFail("Stop button pushed");
		} else if (theLog == null) {
			CallState.reportLogFetchFail("Failed to connect");
			// DisplayActions.showError("Failed to connect");
		} else  {
			CallState.setCallID(theLog.getCallID());
			CallState.reportLogFetchSucceed("");
			// DisplayActions.showEndOfCall(theLog.getCallID());
		}
		synchronized (CallState.getSynch()) {
			CallState.setLogFetchDone();
			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": calling launchDone from logFetch at end"); //$NON-NLS-1$
				System.out.flush();
			}
			if (CallState.launchDone()) {
				CallState.endLaunch();
			}
		}
	}
}
