/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/


#include <windows.h>
#include <process.h>
#include <stdio.h>
#include <tchar.h>
#include <psapi.h>

#include "org_eclipse_vtp_dialer_skype_SkDialer.h"

enum {
    SKYPECONTROLAPI_ATTACH_SUCCESS=0,               // Successful attachment, Skype window handle is in WPARAM
    SKYPECONTROLAPI_ATTACH_PENDING_AUTHORIZATION=1, // Waiting for user to say OK
                                                    // wait for SKYPECONTROLAPI_ATTACH_SUCCESS to be received
    SKYPECONTROLAPI_ATTACH_REFUSED=2,               // User has said NO
    SKYPECONTROLAPI_ATTACH_NOT_AVAILABLE=3,         // API is not available now 
                                                    // wait for SKYPECONTROLAPI_ATTACH_API_AVAILABLE message before 
                                                    // trying to connect again
    SKYPECONTROLAPI_ATTACH_API_AVAILABLE=0x8001
};

bool debug = false;             // set in main method so can be run as exe, not just dll
bool dllInitialized = false;    // set to note that initializeNative has been called successfully
bool dllConnected = false;      // set to note that connectNative has been called successfully

HINSTANCE processHandle;
HWND skypeAPIWindowHandle=NULL;
HWND mainWindow = NULL;
HANDLE msgThread;

CHAR skypefile[1024];       // will hold the Skype executable path
DWORD skypefilelen = sizeof(skypefile);

// following two vars will hold Skype message ids used for connection
UINT msgSkypeControlAPIAttach;
UINT msgSkypeControlAPIDiscover;

// ID of the thread that handles the messages from Skype
UINT msgThreadID;

// JNI-related global variables caching useful references
jmethodID receiveMessageFromSkypeID = NULL; // remember methodID for callback to Java
jmethodID receiveConnectMessageFromSkypeID = NULL; 

jclass clscache = NULL;
JavaVM *jvm = NULL;
JNIEnv *envCache = NULL;                    // cache the JVM environment pointer
jobject objCache;                           // cache the calling object

// Utility functions
void ErrorBox(LPCTSTR errorMessage) {
    MessageBox(NULL, errorMessage, _T("Error"), MB_OK | MB_ICONERROR);
}

void InfoBox(LPCTSTR infoMessage) {
    MessageBox(NULL, infoMessage, _T("Information"), MB_OK | MB_APPLMODAL | MB_ICONEXCLAMATION);
}

/**
 * skypeIsInstalled determines whether the Skype application is installed on the machine
 * It queries the registry. It asks for the value of SkypePath under Software\Skype\Phone
 * in HKEY CURRENT USER first and in HKEY LOCAL MACHINE second. If there is no value under
 * either top-level key, Skype is not installed.
 * 
 * The pathname of the Skype application is stored in the global: skypefile
 */
bool skypeIsInstalled() {
    HKEY skypeRegKey = NULL;
    if (RegOpenKeyEx(HKEY_CURRENT_USER, "SOFTWARE\\Skype\\Phone", 0, 
            KEY_QUERY_VALUE, &skypeRegKey) == ERROR_SUCCESS) {
        if (RegQueryValueEx(skypeRegKey, "SkypePath", NULL, NULL, (BYTE *)&skypefile, &skypefilelen) == ERROR_SUCCESS) {
            if (!(skypefilelen == 0 || skypefilelen == 1)) {
                RegCloseKey(skypeRegKey);
                return true;
            }
        }
        RegCloseKey(skypeRegKey);
    }
    skypefilelen = sizeof(skypefile);   // reinitialize before using again
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "SOFTWARE\\Skype\\Phone", 0, 
            KEY_QUERY_VALUE, &skypeRegKey) == ERROR_SUCCESS) {
        if (RegQueryValueEx(skypeRegKey, "SkypePath", NULL, NULL, (BYTE *)&skypefile, &skypefilelen) == ERROR_SUCCESS) {
            if (!(skypefilelen == 0 || skypefilelen == 1)) {
                RegCloseKey(skypeRegKey);
                return true;
            }
        }
        RegCloseKey(skypeRegKey);
    }
    return false;     
}

// start Skype from the location stored in global skypefile     
//bool startSkype() {
//    // start Skype if not found in processes
//    printf("Starting Skype client because it was not running\n"); fflush(stdout);
//    HINSTANCE shresult = ShellExecute(NULL, "open", skypefile, "", ".", SW_SHOWNORMAL);
//    if ((int)shresult <= 32) {
//        char buf[64];
//        sprintf(buf, "DLL Message 2: Skype client did not start\nReason: %d", (int)shresult);
//        jstring skypemsg = envCache->NewStringUTF(buf);
//        envCache->CallVoidMethod(objCache, receiveMessageFromSkypeID, skypemsg);
//        return false;
//    }
//    return true;
//}


// Locate Skype.exe process in Windows, start Skype if it is not running

bool skypeIsRunning() {
    
    DWORD processIDs[1024];      // will have array of all process IDs in system

    DWORD count = 0;            // receives number of bytes returned from EnumProcesses
    HMODULE processModules[1];  // will have pointer to a module of the process
                                // the first module is always the one we want 
    DWORD modCount = 0;         // should always be sizeof(HMODULE) at end
    
    TCHAR processFilename[256]; // receives the module filename for a process

    // first ask if Skype is installed
    if (!skypeIsInstalled()) {
        jstring skypemsg = envCache->NewStringUTF("DLL Message 1: Skype is not installed on this machine!\nYou must install Skype before you can continue.");
        envCache->CallVoidMethod(objCache, receiveMessageFromSkypeID, skypemsg);
        
        return false;
    }
    // OK, so Skype is installed. Is it running?
    // get all the processes on the machine
    if (EnumProcesses(processIDs, sizeof(processIDs), &count)) {
        count /= sizeof(DWORD);
    }
    // if EnumProcesses fails, or we do not get all the processes, 
    // go ahead and look for it and start it again, even if it is really running
    // That is an OK thing to do, though it causes some annoying window popping    

    // iterate over the processIDs, find modules and the file name of the module
    // quit when file name Skype.exe is found
    for (unsigned i = 0; i < count; i++) {
        if (processIDs[i] < 10) continue;   // ignore early system processes
        
        HANDLE processHandle = OpenProcess( PROCESS_QUERY_INFORMATION |
                                   PROCESS_VM_READ,
                                   FALSE, processIDs[i] );
        
        EnumProcessModules(processHandle, processModules,
                sizeof(HMODULE), &modCount);
        // the first module returned will be the one for the main window        
        GetModuleBaseName( processHandle, processModules[0], processFilename, 
                               sizeof(processFilename)/sizeof(TCHAR) );        
        if (strcmp("Skype.exe", processFilename) == 0) {
            return true;
        }
    }
        jstring skypemsg = envCache->NewStringUTF("DLL Message 4: Skype is not currently running on this machine!\nYou must start it and login before continuing.");
        envCache->CallVoidMethod(objCache, receiveMessageFromSkypeID, skypemsg);
        
        return false;
}

/**
 * Set up Window and handle window communications.
 */
static LRESULT CALLBACK processIncomingMessages(
    HWND windowHandle, UINT message, WPARAM intparam, LPARAM longparam) {
    LRESULT returnCode = 0;

	JNIEnv *env;
	jvm->AttachCurrentThread( (void**) &env, NULL );

    switch(message) {
        case WM_DESTROY:
        	PostQuitMessage(0);
            returnCode = 0;
            break;
        case WM_COPYDATA:
            if( skypeAPIWindowHandle == (HWND) intparam ) {  // did this come from Skype?
                PCOPYDATASTRUCT msgData =(PCOPYDATASTRUCT)longparam;
                
                // send message string from the Windows message to the Java callback
                jstring skypemsg = env->NewStringUTF((const char*)msgData->lpData);
                env->CallVoidMethod(objCache, receiveMessageFromSkypeID, skypemsg);
                returnCode=1;
            }
            break;
        default:
            if( message==msgSkypeControlAPIAttach ) {
                jstring skypemsg;
                switch(longparam) {
                    case SKYPECONTROLAPI_ATTACH_SUCCESS:
//                        printf("ATTACH CONNECTED\n"); fflush(stdout);                    
                        skypemsg = env->NewStringUTF("ATTACH CONNECTED");
                        skypeAPIWindowHandle=(HWND)intparam;
                        break;
                    case SKYPECONTROLAPI_ATTACH_REFUSED:
//                        printf("ATTACH REFUSED\n"); fflush(stdout);  
                        skypemsg = env->NewStringUTF("ATTACH REFUSED");
                        break;
                    case SKYPECONTROLAPI_ATTACH_API_AVAILABLE:
//                        printf("ATTACH AVAILABLE\n"); fflush(stdout);             
						jvm->DetachCurrentThread();
				        SendMessage( HWND_BROADCAST, msgSkypeControlAPIDiscover, (WPARAM)mainWindow, 0);						
						return 1;
                        break;
                    case SKYPECONTROLAPI_ATTACH_NOT_AVAILABLE:
//                        printf("ATTACH NOT AVAILABLE\n"); fflush(stdout);
						jvm->DetachCurrentThread();
						// TODO send the ATTACH NOT AVAILABLE message back to Java
						// let Java code deal with what to tell the user. End the call
						InfoBox("There may be no user logged into Skype");                 
						return 1;
						break;
                    case SKYPECONTROLAPI_ATTACH_PENDING_AUTHORIZATION:
//                        printf("ATTACH PENDING\n"); fflush(stdout); 
						jvm->DetachCurrentThread();
						return 1;
                        break;
                }
                // send connect message to Java
                env->CallVoidMethod(objCache, receiveConnectMessageFromSkypeID, skypemsg);
            }
            else {
            	
                returnCode = DefWindowProc(windowHandle, message, intparam, longparam);
            }
    }
	jvm->DetachCurrentThread();

    return returnCode;
}
    
bool createMainWindowClass() {
    WNDCLASSEX mainWndClass;
    
    bool returnStatus = false;
    processHandle = (HINSTANCE)OpenProcess( PROCESS_DUP_HANDLE, FALSE, GetCurrentProcessId());
    
    mainWndClass.cbSize = sizeof(WNDCLASSEX);
    mainWndClass.style = CS_VREDRAW | CS_HREDRAW;
    mainWndClass.lpfnWndProc = &processIncomingMessages;
    mainWndClass.cbClsExtra = 0;
    mainWndClass.cbWndExtra = 0;
    mainWndClass.hInstance = processHandle;
    mainWndClass.hIcon = NULL;
    mainWndClass.hCursor = NULL;
    mainWndClass.hbrBackground = NULL;
    mainWndClass.lpszMenuName = 0;
    mainWndClass.lpszClassName = "SkDialer";
    mainWndClass.hIconSm = NULL;
    
    if (RegisterClassEx(&mainWndClass) != 0) {
    	returnStatus = true;
    } else {
    	if (GetLastError() == 1410) returnStatus = true; // case where class exists
//    	printf("inside createMainWindowClass: error code = %ld\n", GetLastError()); fflush(stdout);
    }
    
    if (!returnStatus) {
        CloseHandle(processHandle);
        processHandle = NULL;
    }
    return returnStatus;
}

void teardownMainClass() {
    if(!UnregisterClass( "SkDialer", processHandle)) {
//    	printf("error code for UnregisterClass = %ld\n", GetLastError()); fflush(stdout);
    }
    CloseHandle(processHandle),processHandle=NULL;
}

bool createMainWindow() {
    mainWindow = CreateWindowEx(WS_EX_NOPARENTNOTIFY, 
        "SkDialer", "SkDialer", WS_MINIMIZE, 
        CW_USEDEFAULT, CW_USEDEFAULT, 0, 
        0, HWND_MESSAGE, NULL,
        processHandle, NULL);
    return (mainWindow != NULL);
}

unsigned WINAPI skypeInputMsgHandlerThread(LPVOID threadPtr) {
    createMainWindow();
//    printf("after createMainWindow\n");
//                    fflush(stdout);
    SendMessage( HWND_BROADCAST, msgSkypeControlAPIDiscover, (WPARAM)mainWindow, 0);
//    printf("after sending Skype discover message\n"); fflush(stdout);
    MSG message;
    int retval;
    while((retval = GetMessage(&message, mainWindow, 0, 0)) != 0) {
        if (retval == -1) {
            GetLastError();
            return 1;  // note that thread will be terminated automatically
        } else if (message.message != WM_QUIT) {
            TranslateMessage(&message);
            DispatchMessage(&message);
        }
    }
//    printf("exiting message handler thread\n");
//    fflush(stdout);    
    return message.wParam;
    return 0;
}
    
void teardownMainWindow() {
    if (mainWindow != NULL) {
		SendMessage( mainWindow, WM_DESTROY, NULL, NULL),mainWindow= NULL;          
    }
}

void destroy(void) {
    if (!CloseHandle(msgThread)) {
//    	printf("last error for CloseHandle = %ld\n", GetLastError()); fflush(stdout);
    }
    teardownMainWindow();
    teardownMainClass();
}

int primaryConnect() {
	
	if (dllConnected) { return 1;}
    if (createMainWindowClass() ){
        msgThread = (HANDLE) _beginthreadex(NULL, 
            0, 
            &skypeInputMsgHandlerThread,  
            NULL, 
            0, 
            &msgThreadID);
        if (msgThread != NULL) {
            dllConnected = true;
            return 0;
        }
    }
    else {
        teardownMainClass();  
    }
    return -1;       
}

bool primaryStart() {
    if (skypeIsRunning()) {
        msgSkypeControlAPIAttach=RegisterWindowMessage("SkypeControlAPIAttach");
        msgSkypeControlAPIDiscover=RegisterWindowMessage("SkypeControlAPIDiscover");
        return true;
    }
    else {
        // could not make Skype run, so exit
        return false;
    }    
}


/******************************************************************************************************************
 * JNI-callable methods follow
 ******************************************************************************************************************/
 
/**
 * Java_org_eclipse_vtp_dialer_skype_SkDialer_destroyNative is called to teardown the operation of this DLL
 */
JNIEXPORT void JNICALL Java_org_eclipse_vtp_dialer_skype_SkDialer_destroyNative
  (JNIEnv * envptr, jobject obj ){
//  	printf("destroyNative called\n"); fflush(stdout);
    if (dllInitialized) {
        destroy();   
        dllInitialized = false;
        dllConnected = false;     
    }
    envptr->DeleteGlobalRef(clscache);
    envptr->DeleteGlobalRef(objCache);
    printf("exiting destroyNative\n"); fflush(stdout);
}

/**
 * Java_org_eclipse_vtp_dialer_skype_SkDialer_connectNative is called to connect to the Skype
 * client. It creates the window structures, creates the listening thread, discovers, and 
 * attaches to Skype. Returns: -1 if it fails to do its job, 0 if it creates the window and thread,
 * 1 if already connected.
 */
 
JNIEXPORT jint JNICALL Java_org_eclipse_vtp_dialer_skype_SkDialer_connectNative
  (JNIEnv *envptr, jobject obj){
//    printf("connectNative()\n");
//    printf("enter connectNative: initialized: %s\tconnected: %s\n", 
//    		dllInitialized?"true":"false", dllConnected?"true":"false"); fflush(stdout);
    if (dllInitialized) {
		jint foo = primaryConnect();
//		printf("primaryConnect() returned %d\n", foo); fflush(stdout);
        return foo;  // may be 1, 0, -1
    }
    else {
    	// dll has not been initialized, so exit
        return -2;
    }
}

/**
 * Java_org_eclipse_vtp_dialer_skype_SkDialer_initializeNative is called to initialize the DLL
 * it creates the necessary Windows structures and starts a thread to listen
 * for messages coming from Skype
 */
 
JNIEXPORT jboolean JNICALL Java_org_eclipse_vtp_dialer_skype_SkDialer_initializeNative
  (JNIEnv * envptr, jobject obj) {
  	printf("initializeNative: initialized = %s\n", dllInitialized?"true":"false"); fflush(stdout);
  	if (dllInitialized) {return true;}
    //cache the JNIEnv pointer
    envCache = envptr;
    // cache the calling object
    objCache = (jobject) (envptr)->NewGlobalRef(obj);
    // cache a JavaVM pointer
	envptr->GetJavaVM(&jvm);
    
    jclass cls1 = (envptr)->GetObjectClass(obj); 
	jclass cls = (jclass)envptr->NewGlobalRef(cls1);
	clscache = cls;

    // find and cache the two callback method IDs
    receiveMessageFromSkypeID = 
        (envptr)->GetMethodID(cls, "receiveMessageFromSkype", "(Ljava/lang/String;)V"); 
    receiveConnectMessageFromSkypeID = 
        (envptr)->GetMethodID(cls, "receiveConnectMessageFromSkype", "(Ljava/lang/String;)V");
    if (jvm == NULL || receiveMessageFromSkypeID == NULL || receiveConnectMessageFromSkypeID == NULL) {
    	envptr->DeleteGlobalRef(clscache);
    	envptr->DeleteGlobalRef(objCache);
        return false; /* method not found */ 
    } 
    dllInitialized = primaryStart();
//  	printf("returning from initializeNative: initialized = %s\n", dllInitialized?"true":"false"); fflush(stdout);
    return dllInitialized;
}

/**
 * Java_org_eclipse_vtp_dialer_skype_SkDialer_sendNative is called with a string argument that represents a 
 * message of the Skype API. No processing is done to the message, it is sent as
 * presented here.
 */
 
JNIEXPORT jboolean JNICALL Java_org_eclipse_vtp_dialer_skype_SkDialer_sendNative
  (JNIEnv *envptr, jobject obj, jstring message){
    if (dllInitialized) {
        // get the message content out of the Java VM
        const char *str = (envptr)->GetStringUTFChars( message, 0);
        
        // if skypeAPIWindowHandle is null, that means we have not connected yet
        if (skypeAPIWindowHandle == NULL) {
            return false;
        }
        else {
            COPYDATASTRUCT windowsMessage;
    
            windowsMessage.dwData = 0;
            windowsMessage.lpData = (void *)str;
            windowsMessage.cbData = strlen(str)+1;
            
            SendMessage( skypeAPIWindowHandle, WM_COPYDATA, (WPARAM)mainWindow, (LPARAM)&windowsMessage);          
        }
        (envptr)->ReleaseStringChars(message,(const unsigned short*)str);
        
        return true;
    }
    else {
        jstring skypemsg = envCache->NewStringUTF("DLL Message 3: SkDialer.dll has not been initialized");
        envCache->CallVoidMethod(objCache, receiveMessageFromSkypeID, skypemsg);
        return false;
    }

  }
