/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.dialer.skype;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.vtp.internal.dialer.IVoipCall;
import org.eclipse.vtp.internal.dialer.IVoipConnection;
import org.eclipse.vtp.internal.dialer.IVoipDialerListener;
import org.eclipse.vtp.internal.dialer.VoipDialerException;

/**
 * Skype-level implementation of the IVoipCall interface.
 * 
 * This class encapsulates a single call through the Skype VoIP service. It is
 * created from the {@link SkypeConnection} object that was itself created from
 * a {@link SkypeDialer} instance. This class handles the dialing of a given
 * number (which may be a SIP URL or Skype connection string) and waits for the
 * acknowledgement messages from Skype that signal a successfully-connected
 * call.
 * 
 * This object is also used to send DTMF data through the active call, and to
 * hang-up a call.
 * 
 * @author rjray
 */
public class SkypeCall implements IVoipCall {
	// This is the status value we use when waiting for Skype responses for
	// things like dialing, hanging up, etc. All the interfaces define their
	// error code as the last status value, so shifting upwards from it is safe
	final private static int AWAITING_RESPONSE = IVoipCall.CALL_ERROR << 1;

	// Default delay between DTMF characters, figured out largely by trial and
	// error
	final private static int DEFAULT_DTMF_DELAY = 200;

	// The listener the user specified at the Dialer level (if any)
	final private IVoipDialerListener userListener;

	// The internal dialer object, used to send messages to Skype
	final private SkDialer skDialer;

	// The connection object on which this call has been made
	final private IVoipConnection connection;

	// Call's current status
	private int callStatus;

	// Last message received from Skype, for this call
	private String callLastMessage;

	// Object to synchronize on for status change/reference
	final SkypeCall objRef = this;

	// Flag to indicate the status change from Skype has been noted
	private boolean statusChangeOccurred = false;

	// Skype ID for this call. We need this when sending DTMF and hanging up
	private int callId;

	private int failureReason = 0;

	private static final String[] failureStrings = { "No reason", "Miscellaneous Error",
			"Non-existent user or phone number", "User is offline",
			"No proxy found", "Session terminated", "No common codec found",
			"Sound I/O error", "Problem with remote sound device",
			"Call blocked by recipient", "Recipient not a friend",
			"Current user not authorized by recipient", "Sound recording error" };

	// Hash-map of possible Skype-messages indicating a failed dial attempt
	private static final Map failureMap = new HashMap();

	// Control for debugging printout
	private static final boolean DEBUGGING = SkypePlugin.getDefault()
			.isDebugging();

	static {
		failureMap.put("REFUSED", ""); //$NON-NLS-1$
		failureMap.put("MISSED", ""); //$NON-NLS-1$
		failureMap.put("CANCELLED", ""); //$NON-NLS-1$
		failureMap.put("BUSY", ""); //$NON-NLS-1$
		failureMap.put("UNPLACED", ""); //$NON-NLS-1$
		failureMap.put("FAILED", ""); //$NON-NLS-1$
	}

	/**
	 * Create a new SkypeCall object associated with the given dialer and
	 * user-supplied listener. The dialer is the low-level {@link SkDialer}
	 * class, not the user-level dialer class. The listener is what was
	 * originally passed to the creation of the {@link SkypeDialer} object by
	 * the user.
	 * 
	 * @param listener
	 *            An object implementing the IVoipDialerListener interface
	 * @param dialer
	 *            An object of the SkDialer class, used to communicate with
	 *            Skype
	 */
	SkypeCall(final IVoipDialerListener listener, final SkDialer dialer,
			final IVoipConnection myconnection) {
		super();
		if (DEBUGGING) {
			System.out.println(time.format(new Date()) + ": SkypeCall create"); //$NON-NLS-1$
			System.out.flush();
		}
		userListener = listener;
		skDialer = dialer;
		callStatus = AWAITING_RESPONSE;
		callLastMessage = "";
		statusChangeOccurred = false;
		connection = myconnection;
	}

	SimpleDateFormat time = new SimpleDateFormat("mm:ss.SSS", Locale.getDefault()); //$NON-NLS-1$

	/**
	 * @param number
	 * @throws VoipDialerException
	 */

	void dialNumber(final String number) throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeCall.dialNumber() called"); //$NON-NLS-1$
			System.out.flush();
		}
		final ISkDialerListener skListener = getListener();

		skDialer.setListener(skListener);
		try {
			skDialer.sendMessage("#tellme CALL " + number);
		} catch (PhoneException pe) {
			callStatus = IVoipCall.CALL_ERROR;
			skDialer.setListener(null);
			throw new VoipDialerException(this.userListener, "Unable to call "
					+ number, pe);
		}

		synchronized (objRef) {
			if (DEBUGGING) {
				System.out.println(time.format(new Date())
						+ ": SkypeCall.dialNumber inside 60-second wait."); //$NON-NLS-1$
				System.out.flush();
			}
			if (!statusChangeOccurred) {
				try {
					objRef.wait(60000); // wait for one minute or until status
					// has changed
				} catch (InterruptedException e) {
				}
			}
			if (!statusChangeOccurred) {
				throw new VoipDialerException(userListener,
						"Dialing did not connect after 60 seconds");
			}
			statusChangeOccurred = false;
		}
		if (callStatus == IVoipCall.CALL_NO_PHONE) {
			throw new VoipDialerException(userListener,
					"Skype client not running");
		}
		if (callStatus == IVoipCall.CALL_ERROR) {
			if ("FAILED".equals(callLastMessage)){
				throw new VoipDialerException(userListener, "Dialing failed\nReason: "+failReason());
			}
			throw new VoipDialerException(userListener, "Dialing failed");
		}

	}

	/**
	 * Create a listener instance that implements {@link ISkDialerListener}.
	 * This gets uses to set a listener on the low-level {@link SkDialer} object
	 * so that messages can be captured during the dialing process.
	 * 
	 * @return An object that implements ISkDialerListener
	 */
	private ISkDialerListener getListener() {
		final Pattern initial = Pattern
				.compile("^#tellme CALL (\\d+) STATUS ROUTING"); //$NON-NLS-1$
		final Pattern statusLine = Pattern.compile("CALL (\\d+) STATUS (\\w+)"); //$NON-NLS-1$
		final Pattern failreason = Pattern
				.compile("CALL (\\d+) FAILUREREASON (\\d+)"); //$NON-NLS-1$

		final ISkDialerListener skListener = new ISkDialerListener() {
			int callNum = 0;

			int msgNum = 0;

			public void receiveMessage(final String message) {
				if (DEBUGGING) {
					System.out.println(time.format(new Date()) + ": Listener: " //$NON-NLS-1$
							+ message);
				}
				if ("SKYPE DEAD".equals(message)) { //$NON-NLS-1$
					synchronized (objRef) {
						objRef.callStatus = IVoipCall.CALL_NO_PHONE;
						objRef.callLastMessage = message;
						skDialer.setListener(null);
						objRef.statusChangeOccurred = true;
						objRef.notifyAll();
					}

				}
				if (callNum == 0) {
					Matcher match = initial.matcher(message);
					if (match.matches()) {
						callNum = Integer.parseInt(match.group(1));
						synchronized (objRef) {
							objRef.callId = callNum;
							objRef.callLastMessage = "ROUTING"; //$NON-NLS-1$
						}
					}
				} else {
					Matcher match = statusLine.matcher(message);
					if (match.matches()) {
						msgNum = Integer.parseInt(match.group(1));
						if (msgNum == callNum) {
							String status = match.group(2);
							if (status.equals("INPROGRESS")) { //$NON-NLS-1$
								synchronized (objRef) {
									objRef.callStatus = IVoipCall.CALL_ACTIVE;
									objRef.callLastMessage = status;
									objRef.statusChangeOccurred = true;
									objRef.notifyAll();
								}
							} else if (status.equals("FINISHED")) { //$NON-NLS-1$
								synchronized (objRef) {
									objRef.callStatus = IVoipCall.CALL_COMPLETED;
									objRef.callLastMessage = status;
									skDialer.setListener(null);
									objRef.statusChangeOccurred = true;
									objRef.notifyAll();
								}
							} else if (failureMap.containsKey(status)) {
								synchronized (objRef) {
									objRef.callStatus = IVoipCall.CALL_ERROR;
									objRef.callLastMessage = status;
									skDialer.setListener(null);
									objRef.statusChangeOccurred = true;
									objRef.notifyAll();
								}
							}
						}
					} else {
						match = failreason.matcher(message);
						if (match.matches()) {
							msgNum = Integer.parseInt(match.group(1));
							if (msgNum == callNum) {
								failureReason = Integer.parseInt(match.group(2));
							}
						}
					}
				}

			}
		};

		return skListener;
	}

	/**
	 * Retrieve the last message-string sent by Skype regarding this call
	 * 
	 * @return Last message-string, anything after "CALL xxx STATUS"
	 */
	public String lastMessage() {
		return callLastMessage;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipCall#status()
	 */
	public int status() {
		return callStatus;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipCall#sendDTMF(java.lang.String)
	 */
	public void sendDTMF(final String dtmf) throws VoipDialerException {
		sendDTMF(dtmf, DEFAULT_DTMF_DELAY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipCall#sendDTMF(java.lang.String, int)
	 */
	public void sendDTMF(final String dtmf, final int pause)
			throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeCall.sendDTMF() called"); //$NON-NLS-1$
			System.out.flush();
		}
		// Can't really do much if the call isn't active
		if (callStatus != IVoipCall.CALL_ACTIVE) {
			throw new VoipDialerException(userListener,
					"Cannot send DTMF unless call is still connected");
		}

		// Check that the string contains only digits, # and *
		final Pattern validDTMF = Pattern.compile("^[0-9#\\*]*$"); //$NON-NLS-1$
		final Matcher dtmfMatcher = validDTMF.matcher(dtmf);
		if (!dtmfMatcher.matches()) {
			throw new VoipDialerException(userListener, "Invalid DTMF string ("
					+ dtmf + ")");
		}

		if (dtmf.length() > 0) {
			final char[] dtmfChars = dtmf.toCharArray();
			// Send the first one, then loop from #2 onward. The puts the timer
			// call at the top of the loop, and keeps us from calling the timer
			// after the last character.
			try {
				skDialer.sendMessage("SET CALL " + callId + " DTMF " //$NON-NLS-1$//$NON-NLS-2$
						+ dtmfChars[0]);
			} catch (PhoneException pe) {
				throw new VoipDialerException(userListener,
						"Error sending DTMF to Skype", pe);
			}
			for (int idx = 1; idx < dtmfChars.length; idx++) {
				try {
					Thread.sleep(pause);
					skDialer.sendMessage("SET CALL " + callId + " DTMF " //$NON-NLS-1$//$NON-NLS-2$
							+ dtmfChars[idx]);
				} catch (PhoneException pe) {
					throw new VoipDialerException(userListener,
							"Error sending DTMF to Skype", pe);
				} catch (InterruptedException ex) {
					throw new VoipDialerException(userListener, ex);
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipCall#hangUp()
	 */
	public void hangUp() throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeCall.hangUp() called"); //$NON-NLS-1$
			System.out.flush();
		}
		try {
			skDialer.sendMessage("SET CALL " + callId + " STATUS FINISHED"); //$NON-NLS-1$//$NON-NLS-2$
		} catch (PhoneException pe) {
			throw new VoipDialerException(userListener,
					"Error sending hangup message to Skype", pe);
		}

		// callStatus = IVoipCall.CALL_OK;
		callStatus = IVoipCall.CALL_COMPLETED;
	}

	// getConnection returns the IVoipConnection object that is our "parent"
	public IVoipConnection getConnection() {
		return connection;
	}

	// failReason returns the Skype failure reason reported if the final call
	// status is FAILED
	private String failReason() {
		return failureStrings[failureReason];
	}
}
