/*******************************************************************************
 * Copyright (c) 2005 Tellme Networks, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tellme Networks, Inc. - Initial implementation
 *******************************************************************************/

package org.eclipse.vtp.dialer.skype;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.vtp.internal.dialer.IVoipConnection;
import org.eclipse.vtp.internal.dialer.IVoipDialer;
import org.eclipse.vtp.internal.dialer.IVoipDialerListener;
import org.eclipse.vtp.internal.dialer.VoipDialerException;

/**
 * Skype-specific implementation of IVoipDialer.
 * 
 * This class connects the user to a VoIP dialer in the form of the Skype VoIP
 * client. Through this class, the user may then connect to Skype, and use the
 * resultant connection to dial out.
 * 
 * @author rjray
 */
public class SkypeDialer implements IVoipDialer {
	private IVoipDialerListener userListener;

	private int dialerStatus;

	private final SkDialer skDialer = SkDialer.getDialer();

	private final SimpleDateFormat time = new SimpleDateFormat("mm:ss.SSS",		//$NON-NLS-1$
			Locale.getDefault());

	// Control for debugging printout
	private static final boolean DEBUGGING = SkypePlugin.getDefault()
			.isDebugging();
	
	private static final SkypeDialer singleton;

	private static boolean initialized = false;

	static {
		singleton = new SkypeDialer();
	}

	/**
	 * Create a new SkypeDialer handle.
	 */
	private SkypeDialer() {
		super();
		if (DEBUGGING) {
			System.out
					.println(time.format(new Date()) + ": SkypeDialer create");		//$NON-NLS-1$
			System.out.flush();
		}
		dialerStatus = IVoipDialer.DIALER_NO_INIT;
	}

	/**
	 * Return the singleton instance of this class
	 * 
	 * @return SkypeDialer singleton
	 */
	public static SkypeDialer instance() {
		return singleton;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipDialer#initialize()
	 */
	public void initialize() throws VoipDialerException {

			initialize(new IVoipDialerListener() {
				public void dialerMessage(final String msg) {
				}
			});

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipDialer#initialize(org.eclipse.vtp.dialer.IVoipDialerListener)
	 */
	public void initialize(final IVoipDialerListener listener)
			throws VoipDialerException {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeDialer.initialize() called");		//$NON-NLS-1$
			System.out.flush();
		}
		userListener = listener;

		// Init the SkDialer, set up the listener to null for now
		try {
			skDialer.initializeSkDialer(null);
		} catch (PhoneException pe) {
			// Pass it up the chain with a local annotation, using the
			// user-listener
			throw new VoipDialerException(listener, pe.getMessage(),
					pe);
		}

		dialerStatus = IVoipDialer.DIALER_OK;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipDialer#connect()
	 */
	public IVoipConnection connect() throws VoipDialerException {
		SkypeConnection newConnection;
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeDialer.connect() called");		//$NON-NLS-1$
			System.out.flush();
		}

		dialerStatus = IVoipDialer.DIALER_CONNECTING;
		newConnection = new SkypeConnection(userListener, skDialer);
		newConnection.connect(); // This blocks until connection, or throws
		dialerStatus = IVoipDialer.DIALER_OK;

		return newConnection;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.vtp.dialer.IVoipDialer#status()
	 */
	public int status() {
		return dialerStatus;
	}

	protected void finalize() throws VoipDialerException, Throwable {
		if (DEBUGGING) {
			System.out.println(time.format(new Date())
					+ ": SkypeDialer.finalize() called");		//$NON-NLS-1$
			(new Exception("where from?")).printStackTrace(System.out);		//$NON-NLS-1$
			System.out.flush();
		}
		try {
			skDialer.destroySkDialer();
		} catch (PhoneException pe) {
			throw new VoipDialerException(userListener,
					"Unable to destroy SkDialer", pe);
		}

		super.finalize();
	}
}
